# Virtual Media Management
use strict;
use warnings;
our (%gui, %hostspec);

sub show_vmm_dialog() {
    &log_msg("Retrieving media information...");
    &busy_window($gui{windowMain}, 0, 'watch');
    &fill_list_vmmhd();
    &fill_list_vmmdvd();
    &fill_list_vmmfloppy();
    &log_msg("Retrieved media information.");
    &busy_window($gui{windowMain}, 0);
    $gui{dialogVMM}->run;
    $gui{dialogVMM}->hide;
}

sub show_createhd_dialog() {
    $gui{entryCreateHDName}->set_text('NewHardDisk' . int(rand(9999)));
    $gui{spinbuttonCreateHDSize}->set_range(8.00, $hostspec{maxhdsize});
    $gui{spinbuttonCreateHDSize}->set_value(2048.00);
    my $response = $gui{dialogCreateHD}->run;
    $gui{dialogCreateHD}->hide;
    my $diskname = $gui{entryCreateHDName}->get_text();
    $diskname =~ s/\.vdi$//;
    my $disksize = $gui{spinbuttonCreateHDSize}->get_value_as_int();

    if ($diskname and ($response eq 'ok')) {
        if ($gui{checkbuttonCreateHDDynamic}->get_active()) { &create_new_hd($diskname, 'VDI', 'Standard', $disksize, $gui{dialogVMM}); }
        else { &create_new_hd($diskname, 'VDI', 'Fixed', $disksize, $gui{dialogVMM}); }
        &fill_list_vmmhd();
    }
}

sub create_new_hd() {
    my ($diskname, $format, $type, $size, $transwin) = @_;
    my $IMedium = IVirtualBox_createHardDisk($gui{websn}, $format, "$hostspec{hdfolder}/$diskname.vdi");
    my $IProgress = IMedium_createBaseStorage($IMedium, $size, $type);
    &show_progress_window($IProgress, "Creating Hard Disk...", '', $transwin);
    &log_msg("Created new hard disk: $hostspec{hdfolder}/$diskname.vdi");
    return $IMedium;
}

# SATA/SAS Controllers need a port created before attachments can be made
sub create_sas_sata_port() {
    my ($IStorCtr) = @_;
    my $bus = IStorageController_getBus($IStorCtr);
    if ($bus eq 'SATA' or $bus eq 'SAS') {
        my ($pc, $mpc) = (IStorageController_getPortCount($IStorCtr), IStorageController_getMaxPortCount($IStorCtr));
        IStorageController_setPortCount($IStorCtr, $pc + 1) if ($pc < $mpc);
    }
}

sub recurse_media_snapshot() {
    my ($treestore, $IMedium, $iter, $type) = @_;
    my $citer = $treestore->append($iter);
    my $gnames;
    my $name = IMedium_getName($IMedium);
    my @guuids = IMedium_getMachineIds($IMedium);
    IMedium_refreshState($IMedium); # Needed otherwise sizes may be 0

    foreach my $id (@guuids) {
        my $snames;
        my $IMachine = IVirtualBox_getMachine($gui{websn}, $id);
        my $mname = IMachine_getName($IMachine);
        my @suuids = IMedium_getSnapshotIds($IMedium, $id);

        foreach my $snapid (@suuids) {
            next if ($snapid eq $id);
            my $ISnapshot = IMachine_getSnapshot($IMachine, $snapid);
            my $snapname = ISnapshot_getName($ISnapshot);
            $snames .= "$snapname, ";
        }

        if ($snames) {
            $snames =~ s/, $//;
            $gnames .= "$mname ($snames), ";}
        else { $gnames .= "$mname, "; }
    }

    $gnames = '<None>' if (!$gnames);
    $gnames =~ s/, $//; # Remove any trailing,

    if ($type eq 'HardDisk') {
        my $asize = int(IMedium_getSize($IMedium) / 1048576);
        my $vsize = IMedium_getLogicalSize($IMedium);
        $treestore->set($citer, 0, $name, 1, $IMedium, 2, $gnames, 3, "$asize (MiB)", 4, "$vsize (MiB)");
    }
    elsif ($type eq 'DVD') {
        my $size = int(IMedium_getSize($IMedium) / 1048576);
        $treestore->set($citer, 0, $name, 1, $IMedium, 2, $gnames, 3, "$size (MiB)");
    }
    else {
        my $size = IMedium_getSize($IMedium);
        $treestore->set($citer, 0, $name, 1, $IMedium, 2, $gnames, 3, "$size (Bytes)");
    }
    &recurse_media_snapshot($treestore, $_, $citer, $type) foreach (IMedium_getChildren($IMedium));
}

sub vmm_tabchanged() {
    my ($widget, $focus, $page) = @_;
    $gui{entryVMMAttachedTo}->set_text('');
    &sensitive_off($gui{toolbuttonVMMRemove}, $gui{toolbuttonVMMRelease});

    #We need to re-evaluate selection on tab change
    if ($page == 0) {
        $gui{buttonVMMAdd}->set_label('_Add Hard Disk');
        $gui{labelVMMLocation}->set_text_with_mnemonic('HD _Location');
        &onsel_list_vmmhd() if ($gui{treeviewVMMHD}->get_selection->get_selected());
    }
    elsif ($page == 1) {
        $gui{buttonVMMAdd}->set_label('_Add CD/DVD');
        $gui{labelVMMLocation}->set_text_with_mnemonic('CD/DVD _Location');
        &onsel_list_vmmdvd() if ($gui{treeviewVMMDVD}->get_selection->get_selected());
    }
    else {
        $gui{buttonVMMAdd}->set_label('_Add Floppy');
        $gui{labelVMMLocation}->set_text_with_mnemonic('Floppy _Location');
        &onsel_list_vmmfloppy() if ($gui{treeviewVMMFloppy}->get_selection->get_selected());
    }
}

sub vmm_release() {
    my %medium;
    my $page = $gui{notebookVMM}->get_current_page();
    my $warn = 0;

    if ($page == 0) { %medium = &getsel_list_vmmhd(); }
    elsif ($page == 1) { %medium = &getsel_list_vmmdvd(); }
    else { %medium = &getsel_list_vmmfloppy(); }

    my @guuids = IMedium_getMachineIds($medium{IMedium});

    foreach my $id (@guuids) {
        my $IMachine = IVirtualBox_getMachine($gui{websn}, $id);
        my @IMediumAttachment = IMachine_getMediumAttachments($IMachine);

        foreach my $attach (@IMediumAttachment) {
            if ($$attach{medium} eq $medium{IMedium}) {
                my ($ISession, $IMachineM) = &get_mutable_session($IMachine);

                if ($ISession) {
                    IMachine_detachDevice($IMachineM, $$attach{controller}, $$attach{port}, $$attach{device});
                    IMachine_saveSettings($IMachineM);
                    ISession_close($ISession);
                }
                else { $warn = 1; }
            }
        }
    }

    if ($warn == 1) { &show_err_msg('mediuminuse', $gui{messagedialogWarning}, " ($medium{Name})"); }
    else { &log_msg("Medium $medium{Name} released."); }

    if ($page == 0) { &fill_list_vmmhd(); }
    elsif ($page == 1) { &fill_list_vmmdvd(); }
    else { &fill_list_vmmfloppy(); }
}

sub vmm_rem() {
    my $page = $gui{notebookVMM}->get_current_page();

    if ($page == 0) {
        my %hd = &getsel_list_vmmhd();
        my $response = $gui{dialogRemoveDelete}->run;
        $gui{dialogRemoveDelete}->hide;

        if ($response eq '1') { # Deletes Disk
            IMedium_refreshState($hd{IMedium}); # Ensure we have the latest disk info
            if (IMedium_getState($hd{IMedium}) eq 'Created') {
                my $IProgress = IMedium_deleteStorage($hd{IMedium});
                &show_progress_window($IProgress, "Deleting hard disk $hd{Name}...", 'cancel', $gui{dialogVMM});
                &log_msg("Deleted hard disk: $hd{Name}.");
                &fill_list_vmmhd();
            }
            else { &show_err_msg('deletemedium', $gui{messagedialogWarning}, " ( $hd{Name})"); }
        }
        elsif ($response eq '2') { # Removes Disk
            IMedium_close($hd{IMedium});
            &log_msg("Storage $hd{Name} removed.");
            &fill_list_vmmhd();
        }
    }
    elsif ($page == 1) {
        my %dvd = &getsel_list_vmmdvd();
        IMedium_close($dvd{IMedium});
        &log_msg("Storage $dvd{Name} removed.");
        &fill_list_vmmdvd();
    }
    else {
        my %floppy = &getsel_list_vmmfloppy();
        IMedium_close($floppy{IMedium});
        &log_msg("Storage $floppy{Name} removed.");
        &fill_list_vmmfloppy();
    }
}

sub vmm_add() {
    my ($widget, $pathentry) = @_;
    my $location = $pathentry->get_text();
    $pathentry->set_text('');
    my $page = $gui{notebookVMM}->get_current_page();

    if ($location) {
        if ($page == 0) {
            IVirtualBox_openHardDisk($gui{websn}, $location, 'ReadWrite', 0, '', 0, '');
            &fill_list_vmmhd();
        }
        elsif ($page == 1) {
            IVirtualBox_openDVDImage($gui{websn}, $location, undef);
            &fill_list_vmmdvd();
        }
        else {
            IVirtualBox_openFloppyImage($gui{websn}, $location, undef);
            &fill_list_vmmfloppy();
        }
    }
}

1;