# List Handling
use strict;
use warnings;
our (%gui, %osversion, %hostspec, %bool, %chooser, %vmc, %signal);

# Populates the list of available guests
sub fill_list_guest() {
    &addrow_log("Retrieving the list of guests from $endpoint...");
    &busy_pointer($gui{windowMain}, 1);
    &clr_list_guest();
    my %guestlist;
    my @IMachine = IVirtualBox_getMachines($gui{websn});
    $guestlist{$_} = IMachine_getName($_) foreach (@IMachine);

    foreach my $machine (sort { lc($guestlist{$a}) cmp lc($guestlist{$b}) } (keys %guestlist)) {
        my $prettygname = $guestlist{$machine};
        my $status = IMachine_getAccessible($machine);

        my $ISnapshot = IMachine_getCurrentSnapshot($machine);
        my $osid = IMachine_getOSTypeId($machine);
        my $uuid = IMachine_getId($machine);
        $prettygname .=  ' (' . ISnapshot_getName($ISnapshot) . ')' if ($ISnapshot);
        $status = ($status eq 'true') ? IMachine_getState($machine) : IVirtualBoxErrorInfo_getText(IMachine_getAccessError($machine));
        $prettygname .= "\n   $status";

        $gui{liststoreGuest}->set($gui{liststoreGuest}->append,
                                  0, $guestlist{$machine},
                                  1, $osversion{$osid}{description},
                                  2, $machine,
                                  3, $status,
                                  4, $osid,
                                  5, $uuid,
                                  6, (-e "$gui{THUMBDIR}/$uuid.png") ? Gtk2::Gdk::Pixbuf->new_from_file("$gui{THUMBDIR}/$uuid.png") : $osversion{$osid}{icon},
                                  7, $prettygname);
    }

    &busy_pointer($gui{windowMain}, 0);
    &addrow_log("List of guests from $endpoint retrieved.");

}

sub fill_list_vminfo() {
    my %weight = (bold => 800,
                  norm => 400);
    my $str;
    $gui{liststoreInfo}->clear();
    my $gref = &getsel_list_guest();
    &addrow_log("Retrieving more details of $$gref{Name}...");
    my $IVRDEServer = IMachine_getVRDEServer($$gref{IMachine});
    my @IStorageController = IMachine_getStorageControllers($$gref{IMachine});
    my $IAudioAdapter = IMachine_getAudioAdapter($$gref{IMachine});
    my $IUSBController = IMachine_getUSBController($$gref{IMachine});
    &addrow_info(0, 'GENERAL', 2, $weight{bold});
    &addrow_info(0, 'Name:', 1, $$gref{Name}, 2, $weight{norm});
    &addrow_info(0, 'OS Type:', 1, $$gref{Os}, 2, $weight{norm});
    &addrow_info(0, 'SYSTEM', 2, $weight{bold});
    &addrow_info(0, 'Firmware:', 1, IMachine_getFirmwareType($$gref{IMachine}), 2, $weight{norm});
    &addrow_info(0, 'Base Memory:', 1, IMachine_getMemorySize($$gref{IMachine}) . ' MB', 2, $weight{norm});
    &addrow_info(0, 'Processor(s):', 1, IMachine_getCPUCount($$gref{IMachine}), 2, $weight{norm});
    $str = '';

    foreach (1..4) {
        my $bdev = IMachine_getBootOrder($$gref{IMachine}, $_);
        $str .= "$bdev, " if ($bdev ne 'Null');
    }

    $str =~ s/, $//;
    &addrow_info(0, 'Boot Order:', 1, $str, 2, $weight{norm}) if ($str);
    $str = '';
    $str .= 'VT-x/AMD-V, ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'Enabled') eq 'true');
    $str .= 'VPID, ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'VPID') eq 'true');
    $str .= 'PAE/NX, ' if (IMachine_getCPUProperty($$gref{IMachine}, 'PAE') eq 'true');
    $str .= 'Nested Paging' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'NestedPaging') eq 'true');
    $str =~ s/, $//;
    &addrow_info(0, 'Acceleration:', 1, $str, 2, $weight{norm}) if ($str);
    &addrow_info(0, 'DISPLAY', 2, $weight{bold});
    &addrow_info(0, 'Video Memory:', 1, IMachine_getVRAMSize($$gref{IMachine}) . ' MB', 2, $weight{norm});
    &addrow_info(0, 'Screens: ', 1, IMachine_getMonitorCount($$gref{IMachine}), 2, $weight{norm});
    $str = '';
    $str .= '2D Video, ' if (IMachine_getAccelerate2DVideoEnabled($$gref{IMachine}) eq 'true');
    $str .= '3D' if (IMachine_getAccelerate3DEnabled($$gref{IMachine}) eq 'true');
    &addrow_info(0, 'Display Acceleration:', 1, $str, 2, $weight{norm}) if ($str);
    &addrow_info(0, 'Remote Display Port(s):', 1, IVRDEServer_getVRDEProperty($IVRDEServer, 'TCP/Ports'), 2, $weight{norm}) if (IVRDEServer_getEnabled($IVRDEServer) eq 'true');
    &addrow_info(0, 'STORAGE', 2, $weight{bold});

    foreach my $controller (@IStorageController) {
        my $controllername = IStorageController_getName($controller);
        &addrow_info(0, 'Controller:', 1, $controllername, 2, $weight{norm});
        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($$gref{IMachine}, $controllername);
        foreach my $attachment (@IMediumAttachment) {
            if ($$attachment{medium}) {
                IMedium_refreshState($$attachment{medium}); # Needed to bring in current sizes
                # Use the base medium for information purposes
                my $size = &bytesToX(IMedium_getLogicalSize($$attachment{medium}));
                &addrow_info(0, "  Port $$attachment{port}:",
                               1, IMedium_getName(IMedium_getBase($$attachment{medium})) . " ($$attachment{type}, $size)",
                               2, $weight{norm});
            }
        }
    }

    &addrow_info(0, 'AUDIO', 2, $weight{bold});

    if (IAudioAdapter_getEnabled($IAudioAdapter) eq 'true') {
        &addrow_info(0, 'Host Driver:', 1, IAudioAdapter_getAudioDriver($IAudioAdapter), 2, $weight{norm});
        &addrow_info(0, 'Controller:', 1, IAudioAdapter_getAudioController($IAudioAdapter), 2, $weight{norm});
    }
    else { &addrow_info(0, 'Audio Disabled', 2, $weight{norm}); }

    &addrow_info(0, 'NETWORK', 2, $weight{bold});

    foreach (0..($hostspec{maxnet}-1)) {
        my $INetworkAdapter = IMachine_getNetworkAdapter($$gref{IMachine}, $_);

        if (INetworkAdapter_getEnabled($INetworkAdapter) eq 'true') {
            my $attachtype = INetworkAdapter_getAttachmentType($INetworkAdapter);

            $str = INetworkAdapter_getAdapterType($INetworkAdapter) . ' (' . $attachtype;

            if ($attachtype eq 'Bridged') { $str .= ', ' . INetworkAdapter_getBridgedInterface($INetworkAdapter); }
            elsif ($attachtype eq 'HostOnly') { $str .= ', ' . INetworkAdapter_getHostOnlyInterface($INetworkAdapter); }
            elsif ($attachtype eq 'Internal') { $str .= ', ' . INetworkAdapter_getInternalNetwork($INetworkAdapter); }

            $str .= ')';

            &addrow_info(0, "Adapter $_:", 1, $str, 2, $weight{norm});
        }
    }

    &addrow_info(0, 'SERIAL PORTS', 2, $weight{bold});

    foreach (0..($hostspec{maxser}-1)) {
        my $ISerialPort = IMachine_getSerialPort($$gref{IMachine}, $_);
        if (ISerialPort_getEnabled($ISerialPort) eq 'true') {
            &addrow_info(0, "Port #$_:",
                           1, 'Enabled, ' . ISerialPort_getHostMode($ISerialPort) . ', ' . ISerialPort_getPath($ISerialPort),
                           2, $weight{norm});
        }
        else { &addrow_info(0, "Port #$_:", 1, 'Disabled', 2, $weight{norm}); }
    }

    &addrow_info(0, 'PARALLEL PORT', 2, $weight{bold});
    my $IParallelPort = IMachine_getParallelPort($$gref{IMachine}, 0);

    if (IParallelPort_getEnabled($IParallelPort) eq 'true') {
        &addrow_info(0, 'LPT:',
                       1, 'Enabled, ' . IParallelPort_getPath($IParallelPort),
                       2, $weight{norm});
    }
    else { &addrow_info(0, 'Disabled', 2, $weight{norm}); }

    &addrow_info(0, 'USB', 2, $weight{bold});

    if (IUSBController_getEnabled($IUSBController) eq 'true' ) {
        my @IUSBDeviceFilter = IUSBController_getDeviceFilters($IUSBController);
        my $active = 0;
        foreach (@IUSBDeviceFilter) {
            $active++ if (IUSBDeviceFilter_getActive($_) eq 'true');
        }
        &addrow_info(0, 'Device Filters:', 1,  scalar(@IUSBDeviceFilter) . " ($active active)", 2, $weight{norm});
    }
    else { &addrow_info(0, 'Disabled', 2, $weight{norm}); }

    my @sf = IMachine_getSharedFolders($$gref{IMachine});
    &addrow_info(0, 'SHARED FOLDERS', 2, $weight{bold});
    &addrow_info(0, 'Shared Folders:', 1, scalar(@sf), 2, $weight{norm});

    my $ISession = &get_existing_session($$gref{IMachine});

    if ($ISession) {
        &addrow_info(0, 'RUNTIME INFORMATION', 2, $weight{bold});
        my $IConsole = ISession_getConsole($ISession);
        my $IGuest = IConsole_getGuest($IConsole);
        &addrow_info(0, 'Operating System:', 1, IGuest_getOSTypeId($IGuest), 2, $weight{norm});
        my $additionsversion = IGuest_getAdditionsVersion($IGuest);
        if ($additionsversion) { &addrow_info(0, 'Guest Additions:', 1, $additionsversion, 2, $weight{norm}); }
        else { &addrow_info(0, 'Guest Additions:', 1, 'Not Installed (or have not loaded yet)', 2, $weight{norm}); }
        ISession_unlockMachine($ISession);
    }

    &addrow_log("More Details of $$gref{Name} retrieved.");
}

sub fill_list_snapshots() {
    &busy_pointer($gui{windowMain}, 1);
    &clr_list_snapshots();
    my $gref = &getsel_list_guest();
    my $ISnapshot_current = IMachine_getCurrentSnapshot($$gref{IMachine});
    &addrow_log("Retrieving snapshot information for $$gref{Name}...");

    if (IMachine_getSnapshotCount($$gref{IMachine}) > 0) {
        my $ISnapshot = IMachine_findSnapshot($$gref{IMachine}, undef); # get first snapshot
        &recurse_snapshot($ISnapshot, undef, $ISnapshot_current);
        $gui{treeviewSnapshots}->expand_all();
    }

    &addrow_log("Snapshot information for $$gref{Name} retrieved.");
    &busy_pointer($gui{windowMain}, 0);
}

# Fills the remote file chooser with a list of files. Involves a lot of splicing because
# of the crazy way VirtualBox returns a file list
sub fill_list_remotefiles() {
    my ($location) = @_;
    $location = File::Spec->canonpath($location) unless ($hostspec{os} =~ m/^WINDOWS/);
    $gui{liststoreRemoteFileChooser}->clear();
    IVFSExplorer_cd($gui{IVFSExplorer}, $location);
    IVFSExplorer_update($gui{IVFSExplorer});
    my @entries = IVFSExplorer_entryList($gui{IVFSExplorer});
    my $chop = (@entries / 4);
    my @filenames = splice @entries, 0, $chop;
    my @types = splice @entries, 0, $chop;
    my @sizes = splice @entries, 0, $chop;
    my @modes = splice @entries, 0, $chop;
    my %files;

    foreach my $ent (0..$#filenames) {
        $files{$filenames[$ent]}{type} = $types[$ent];
        $files{$filenames[$ent]}{size} = $sizes[$ent];
        $files{$filenames[$ent]}{mode} = $modes[$ent];
    }

    my $iter = $gui{liststoreRemoteFileChooser}->append();
    $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Parent)', 1, '..', 2, '', 3, '');

    foreach my $fname (sort { lc($a) cmp lc($b) } (keys %files)) {

        if ($files{$fname}{type} == 4) { # Always add in directories
            my $iter = $gui{liststoreRemoteFileChooser}->append();
            $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Dir)', 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
        }
        elsif ($fname =~ m/$chooser{filter}/i) { # Only add in if it matches the filter
            my $iter = $gui{liststoreRemoteFileChooser}->append();
            $fname =~ m/^.*\.(.*)$/;
            my $ext = $1 ? lc(".$1") : ' ';
            $gui{liststoreRemoteFileChooser}->set($iter, 0, $ext, 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
        }
    }

    $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
}

# Fills a list as returned from reading the remote log file
sub fill_list_log() {
    my ($IMachine, $lognum) = @_;
    $gui{liststoreInfo}->clear();
    my $log;
    my $offset = 0;

    # Reading logs is limited to a maximum chunk size - normally 32K. The chunks are base64 encoded so we
    # need to read a chunk, decode, calculate next offset. Limit loop to 40 runs (max 1MB retrieval)
    for (1..40) {
        my $rawlog = IMachine_readLog($IMachine, $lognum, $offset, 32768); # Request 32K max. Limit is usually 32K anyway
        last if (!$rawlog); # Terminate loop if we've reached the end or log is empty
        $log .= decode_base64($rawlog); # Rawlog is base64 encoded. Append to log
        $offset = length($log); # Set next offset into log to get the next chunk
    }

    if ($log) {
        my @logarr = split "\n", $log; # Do we need to include windows/mac EOL here?
        &addrow_info(0, "$_: ", 1, $logarr[$_]) foreach (0..$#logarr);
    }
    else { &addrow_info(0, ':', 1, '<Log Empty>'); }
}

# Fills a list of basic information about the remote server
sub fill_list_serverinfo() {
    $gui{liststoreInfo}->clear();
    &addrow_info(0, 'URL:', 1, $endpoint);
    &addrow_info(0, 'VirtualBox Version:', 1, $hostspec{vbver});
    &addrow_info(0, 'Build Revision:', 1, $hostspec{buildrev});
    &addrow_info(0, 'Package Type:', 1, $hostspec{pkgtype});
    &addrow_info(0, 'Global Settings File:', 1, $hostspec{settingsfile});
    &addrow_info(0, 'Machine Folder:', 1, $hostspec{machinedir});
    &addrow_info(0, 'Server Logical CPUs:', 1, $hostspec{maxhostcpuon});
    &addrow_info(0, 'Server CPU Type:', 1, $hostspec{cpudesc});
    &addrow_info(0, 'Server CPU Speed:', 1, "$hostspec{cpuspeed} Mhz (approx)");
    &addrow_info(0, 'Server Memory Size:', 1, "$hostspec{memsize} MB");
    &addrow_info(0, 'Server OS:', 1, $hostspec{os});
    &addrow_info(0, 'Server OS Version:', 1, $hostspec{osver});
    &addrow_info(0, 'Min Guest RAM:', 1, "$hostspec{minguestram} MB");
    &addrow_info(0, 'Max Guest RAM:', 1, &bytesToX($hostspec{maxguestram} * 1048576));
    &addrow_info(0, 'Min Guest Video RAM:', 1, "$hostspec{minguestvram} MB");
    &addrow_info(0, 'Max Guest Video RAM:', 1, "$hostspec{maxguestvram} MB");
    &addrow_info(0, 'Max Guest CPUs:', 1, $hostspec{maxguestcpu});
    &addrow_info(0, 'Max Guest Monitors:', 1, $hostspec{maxmonitors});
    &addrow_info(0, 'Max HD Image Size:', 1, &bytesToX($hostspec{maxhdsize}));
}

# Fill the hard disk media list in the VMM
sub fill_list_vmmhd() {
    &busy_pointer($gui{dialogVMM}, 1);
    $gui{labelVMMTypeField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMHD}->clear();
    my $IMediumRef = &get_all_media('HardDisk');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMHD}, $$IMediumRef{$_}, undef, 'HardDisk');
    }

    &busy_pointer($gui{dialogVMM}, 0);
    $gui{toolbuttonVMMRemove}->set_sensitive(0);
    $gui{toolbuttonVMMCopy}->set_sensitive(0);
    $gui{toolbuttonVMMModify}->set_sensitive(0);
    $gui{toolbuttonVMMRelease}->set_sensitive(0);
}

# Fill the DVD media list in the VMM
sub fill_list_vmmdvd() {
    &busy_pointer($gui{dialogVMM}, 1);
    $gui{labelVMMTypeField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMDVD}->clear();
    my $IMediumRef = &get_all_media('DVD');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMDVD}, $$IMediumRef{$_}, undef, 'DVD');
    }

    &busy_pointer($gui{dialogVMM}, 0);
    $gui{toolbuttonVMMRemove}->set_sensitive(0);
    $gui{toolbuttonVMMCopy}->set_sensitive(0);
    $gui{toolbuttonVMMModify}->set_sensitive(0);
    $gui{toolbuttonVMMRelease}->set_sensitive(0);
}

# Fill the floppy media list in the VMM
sub fill_list_vmmfloppy() {
    &busy_pointer($gui{dialogVMM}, 1);
    $gui{labelVMMTypeField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMFloppy}->clear();
    my $IMediumRef = &get_all_media('Floppy');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMFloppy}, $$IMediumRef{$_}, undef, 'Floppy');
    }

    &busy_pointer($gui{dialogVMM}, 0);
    $gui{toolbuttonVMMRemove}->set_sensitive(0);
    $gui{toolbuttonVMMCopy}->set_sensitive(0);
    $gui{toolbuttonVMMModify}->set_sensitive(0);
    $gui{toolbuttonVMMRelease}->set_sensitive(0);
}

# Populate the shared folder list for editing guest settings
sub fill_list_editshared() {
    my ($IMachine) = @_;
    $gui{buttonEditSharedRemove}->set_sensitive(0);
    $gui{buttonEditSharedEdit}->set_sensitive(0);
    $gui{liststoreEditShared}->clear();
    my @ISharedFolder = IMachine_getSharedFolders($IMachine);

    foreach my $share (@ISharedFolder) {
        my $access = 'Full';
        my $automount = 'No';
        my $tooltip = "$$share{name}: $$share{hostPath}";
        $access = 'Read-Only' if ($$share{writable} eq 'false');
        $automount = 'Yes' if ($$share{autoMount} eq 'true');
        $tooltip = $$share{lastAccessError} if ($$share{lastAccessError});
        $tooltip = 'Share is not accessible' if ($$share{accessible} eq 'false');
        my $iter = $gui{liststoreEditShared}->append;
        if ($$share{accessible} eq 'false') { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 4, $gui{pixbufstatuserror16}, 5, $tooltip); }
        else { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 5, $tooltip); }
    }
}

sub fill_list_editstorage() {
    my ($IMachine) = @_;
    &busy_pointer($gui{dialogEdit}, 1);
    &sens_editstor_nosel();
    $gui{treestoreEditStor}->clear();
    my @IStorageController = IMachine_getStorageControllers($IMachine);

    foreach my $controller (@IStorageController) {
        my %ctr_attr = (name  => 1,
                        bus   => 1);
        &get_icontroller_attrs(\%ctr_attr, $controller); # Fill hash with attributes
        my $iter = $gui{treestoreEditStor}->append(undef);

        $gui{treestoreEditStor}->set($iter, 0, $ctr_attr{name},                 # Display Name
                                            1, $ctr_attr{bus} . ' Controller',  # Display Type
                                            2, $ctr_attr{bus} . ' Controller',  # Tooltip
                                            3, 1,                               # Is it a controller
                                            4, $ctr_attr{bus},                  # Controller BUS
                                            5, $ctr_attr{name},                 # Controller's Name
                                            7, $controller);                    # IStorageController object

        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($IMachine, $ctr_attr{name});

        foreach my $attach (@IMediumAttachment) {
            my $citer = $gui{treestoreEditStor}->append($iter);
            my %medium_attr = (refresh  => 1,
                               size     => 1,
                               logsize  => 1,
                               location => 1);
            &get_imedium_attrs(\%medium_attr, $$attach{medium});

            if ($$attach{medium}) { # Is it a medium or empty drive
                my $baseIMedium = IMedium_getBase($$attach{medium});
                my $mediumname = ($$attach{medium} eq $baseIMedium) ? IMedium_getName($baseIMedium) : "(*) " . IMedium_getName($baseIMedium); #Tests for snapshots
                $gui{treestoreEditStor}->set($citer, 0,  $mediumname,                            # Display Name
                                                     1,  $$attach{type},                         # Display Type
                                                     2,  "$medium_attr{location}\nPhysical Size: " .
                                                         &bytesToX($medium_attr{size}) . "\nLogical Size: " .
                                                         &bytesToX($medium_attr{logsize}),       # ToolTip
                                                     3,  0,                                      # Is it a controller
                                                     4,  $ctr_attr{bus},                         # The bus the medium is on
                                                     5,  $ctr_attr{name},                        # The name of the controller it is on
                                                     6,  $$attach{medium},                       # IMedium Object
                                                     7,  $controller,                            # IStorageController it is on
                                                     8,  $$attach{type},                         # Medium Type
                                                     9,  $$attach{device},                       # Device number
                                                     10, $$attach{port},                         # Port Number
                                                     11, $medium_attr{location}); # Location

            }
            else {
                $gui{treestoreEditStor}->set($citer, 0,  '<empty>',                     # Display Name
                                                     1,  $$attach{type},                # Display Typee
                                                     2,  "Empty $$attach{type} Drive",  # Tooltip
                                                     3,  0,                             # Is it a controller
                                                     4,  $ctr_attr{bus},                # The bus the medium is on
                                                     5,  $ctr_attr{name},               # The name of the controller it is on
                                                     7,  $controller,                   # IStorageController it is on
                                                     8,  $$attach{type},                # Medium Type
                                                     9,  $$attach{device},              # Device number
                                                     10, $$attach{port});               # Port Number
            }
        }
    }

    $gui{treeviewEditStor}->expand_all();
    &busy_pointer($gui{dialogEdit}, 0);
}

sub fill_list_usbfilters() {
    my ($IUSBController) = @_;
    &busy_pointer($gui{dialogEdit}, 1);
    $gui{buttonEditUSBEdit}->set_sensitive(0);
    $gui{buttonEditUSBRemove}->set_sensitive(0);
    $gui{buttonEditUSBUp}->set_sensitive(0);
    $gui{buttonEditUSBDown}->set_sensitive(0);
    $gui{liststoreEditUSBFilter}->clear();
    my @IUSBDeviceFilters = IUSBController_getDeviceFilters($IUSBController);
    my $pos = 0;

    foreach my $filter (@IUSBDeviceFilters) {
        my $iter = $gui{liststoreEditUSBFilter}->append();
        $gui{liststoreEditUSBFilter}->set($iter,
                                      0, $bool{IUSBDeviceFilter_getActive($filter)},
                                      1, $filter,
                                      2, IUSBDeviceFilter_getName($filter),
                                      3, $pos);
        $pos++;
    }
    &busy_pointer($gui{dialogEdit}, 0);
}

sub fill_list_vbprefsnet() {
    &busy_pointer($gui{dialogVBPrefs}, 1);
    $gui{buttonVBPrefsDelNet}->set_sensitive(0);
    $gui{buttonVBPrefsEditNet}->set_sensitive(0);
    $gui{liststoreVBPrefsNet}->clear();
    my $IHost = IVirtualBox_getHost($gui{websn});
    my @IHostNetworkInterface = IHost_findHostNetworkInterfacesOfType($IHost, 'HostOnly');

    foreach my $if (@IHostNetworkInterface) {
        my $iter = $gui{liststoreVBPrefsNet}->append();
        $gui{liststoreVBPrefsNet}->set($iter,
                                       0, IHostNetworkInterface_getName($if),
                                       1, $if,
                                       2, IHostNetworkInterface_getId($if));
    }
    &busy_pointer($gui{dialogVBPrefs}, 0);
}

sub clr_list_guest() {
    &sens_unselected();
    $gui{liststoreGuest}->clear();
    &clr_summarydetails();
    &clr_list_snapshots();
}

# Clear snapshot list and set sensitivity
sub clr_list_snapshots() {
    $gui{buttonRestoreSnapshot}->set_sensitive(0);
    $gui{buttonDeleteSnapshot}->set_sensitive(0);
    $gui{buttonDetailsSnapshot}->set_sensitive(0);
    $gui{treestoreSnapshots}->clear();
}

# Adds a message to message log and scrolls to bottom
sub addrow_log() {
    my ($msg) = @_;
    my $iter = $gui{liststoreLog}->append;
    $msg = localtime() . ": $msg";
    $gui{liststoreLog}->set($iter, 0, $msg);
    $gui{treeviewLog}->scroll_to_cell($gui{treeviewLog}->get_model->get_path($iter));
}

sub addrow_info() {
    my $iter = $gui{liststoreInfo}->append;
    $gui{liststoreInfo}->set($iter, @_);
    return $iter;
}

sub getsel_combo() {
    my ($widget, $col) = @_;
    my $model = $widget->get_model();
    my $iter = $widget->get_active_iter();

    if (defined($col)) { return $model->get($iter, $col); }
    else { return $model->get($iter); }
}

sub getsel_list_guest() {
    my $model = $gui{treeviewGuest}->get_model();
    my ($path) = $gui{treeviewGuest}->get_cursor();
    my $iter = $gui{treeviewGuest}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Os', 'IMachine', 'Status', 'Osid', 'Uuid', 'Icon', 'Prettyname');
    return \%hash;
}

# Handles single and multiple selections
sub getsel_list_remotefiles() {
    my @filearray;
    my $model = $gui{treeviewRemoteFileChooser}->get_model();
    my @selections = $gui{treeviewRemoteFileChooser}->get_selection->get_selected_rows();

    foreach my $select (@selections) {
        my $iter = $model->get_iter($select);
        next if (!$iter);
        my @row = $model->get($iter);

        push @filearray, {Type     => $row[0],
                          FileName => $row[1],
                          Size     => $row[2],
                          Mode     => $row[3]};
    }

    return \@filearray;
}

sub getsel_list_snapshots() {
    my $model = $gui{treeviewSnapshots}->get_model();
    my ($path) = $gui{treeviewSnapshots}->get_cursor();
    my $iter = $gui{treeviewSnapshots}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Date', 'ISnapshot');
    return \%hash;
}

sub getsel_list_vmmhd() {
    my $model = $gui{treeviewVMMHD}->get_model();
    my ($path) = $gui{treeviewVMMHD}->get_cursor();
    my $iter = $gui{treeviewVMMHD}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Asize', 'Vsize', 'Accessible', 'Tooltip', 'Location', 'Type');
    return \%hash;
}

sub getsel_list_vmmdvd() {
    my $model = $gui{treeviewVMMDVD}->get_model();
    my ($path) = $gui{treeviewVMMDVD}->get_cursor();
    my $iter = $gui{treeviewVMMDVD}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Size', 'Accessible', 'Tooltip', 'Location', 'Type');
    return \%hash;
}


# Return the selected entry in the VMM floppy disk list
sub getsel_list_vmmfloppy() {
    my $model = $gui{treeviewVMMFloppy}->get_model();
    my ($path) = $gui{treeviewVMMFloppy}->get_cursor();
    my $iter = $gui{treeviewVMMFloppy}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Size', 'Accessible', 'Tooltip', 'Location', 'Type');
    return \%hash;
}

sub getsel_list_editshared() {
    my $model = $gui{treeviewEditShared}->get_model();
    my ($path) = $gui{treeviewEditShared}->get_cursor();
    my $iter = $gui{treeviewEditShared}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Folder', 'Access', 'Mount', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_editstorage() {
    my $model = $gui{treeviewEditStor}->get_model();
    my ($path) = $gui{treeviewEditStor}->get_cursor();
    my $iter = $gui{treeviewEditStor}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('DisplayName', 'DisplayType', 'Tooltip', 'IsController', 'Bus', 'ControllerName', 'IMedium', 'IStorageController', 'MediumType', 'Device', 'Port', 'Location');
    return \%hash;
}

sub getsel_list_usbfilters() {
    my $model = $gui{treeviewEditUSBFilters}->get_model();
    my ($path) = $gui{treeviewEditUSBFilters}->get_cursor();
    my $iter = $gui{treeviewEditUSBFilters}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Enabled', 'IUSBDeviceFilter', 'Name', 'Position');
    return \%hash;
}

sub getsel_list_vbprefsnet() {
    my $model = $gui{treeviewVBPrefsNet}->get_model();
    my ($path) = $gui{treeviewVBPrefsNet}->get_cursor();
    my $iter = $gui{treeviewVBPrefsNet}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IHostNetworkInterface', 'Uuid');
    return \%hash;
}

sub onsel_list_guest() {
    my $gref = &getsel_list_guest();
    &fill_summarydetails();
    &sens_unselected();
    my $status = IMachine_getState($$gref{IMachine});
    &fill_list_snapshots();

    if ($status eq 'Running' | $status eq 'Starting') {
        my @IMediumAttachment = IMachine_getMediumAttachments($$gref{IMachine});
        my $IUSBController = IMachine_getUSBController($$gref{IMachine});
        $gui{menuitemAction}->set_sensitive(1);
        $gui{menuitemStop}->set_sensitive(1);
        $gui{menuitemPause}->set_sensitive(1);
        $gui{menuitemReset}->set_sensitive(1);
        $gui{menuitemKeyboard}->set_sensitive(1);
        $gui{menuitemDisplay}->set_sensitive(1);
        $gui{menuitemLogs}->set_sensitive(1);
        $gui{toolbuttonStop}->set_sensitive(1);
        $gui{toolbuttonReset}->set_sensitive(1);
        $gui{toolbuttonRemoteDisplay}->set_sensitive(1);
        $gui{buttonShowDetails}->set_sensitive(1);
        $gui{buttonRefreshSnapshots}->set_sensitive(1);
        $gui{buttonTakeSnapshot}->set_sensitive(1);
        $gui{menuitemScreenshot}->set_sensitive(1);
        $gui{menuitemUSB}->set_sensitive(1) if $bool{IUSBController_getEnabled($IUSBController)};

        foreach my $attach (@IMediumAttachment) {
            $gui{menuitemDVD}->set_sensitive(1) if ($$attach{type} eq 'DVD');
            $gui{menuitemFloppy}->set_sensitive(1) if ($$attach{type} eq 'Floppy');
        }

    }
    elsif ($status eq 'Saved') {
        $gui{menuitemAction}->set_sensitive(1);
        $gui{menuitemStart}->set_sensitive(1);
        $gui{menuitemClone}->set_sensitive(1);
        $gui{menuitemLogs}->set_sensitive(1);
        $gui{toolbuttonStart}->set_sensitive(1);
        $gui{buttonShowDetails}->set_sensitive(1);
        $gui{buttonRefreshSnapshots}->set_sensitive(1);
        $gui{buttonTakeSnapshot}->set_sensitive(1);
    }
    elsif ($status eq 'Paused') {
        $gui{menuitemAction}->set_sensitive(1);
        $gui{menuitemResume}->set_sensitive(1);
        $gui{menuitemRemoteDisplay}->set_sensitive(1);
        $gui{menuitemLogs}->set_sensitive(1);
        $gui{toolbuttonRemoteDisplay}->set_sensitive(1);
        $gui{buttonShowDetails}->set_sensitive(1);
    }
    elsif ($status eq 'PoweredOff' | $status eq 'Aborted') {
        $gui{menuitemAction}->set_sensitive(1);
        $gui{menuitemStart}->set_sensitive(1);
        $gui{menuitemSettings}->set_sensitive(1);
        $gui{menuitemClone}->set_sensitive(1);
        $gui{menuitemRemove}->set_sensitive(1);
        $gui{menuitemLogs}->set_sensitive(1);
        $gui{toolbuttonStart}->set_sensitive(1);
        $gui{toolbuttonSettings}->set_sensitive(1);
        $gui{buttonShowDetails}->set_sensitive(1);
        $gui{buttonRefreshSnapshots}->set_sensitive(1);
        $gui{buttonTakeSnapshot}->set_sensitive(1);
    }
    else { &sens_unselected(); }
}

sub onsel_list_remotefiles() {
    my $filearrayref = &getsel_list_remotefiles();

    # We only care about the first file selected, and only if it's a directory
    my $fileref = ${$filearrayref}[0];

    if ($$fileref{FileName} eq '..') { &cdup_remotefilechooser(); }
    elsif ($$fileref{Type} eq '(Dir)') {
        my $path = IVFSExplorer_getPath($gui{IVFSExplorer});
        IVFSExplorer_cd($gui{IVFSExplorer}, "$path/$$fileref{FileName}");
        &fill_list_remotefiles(IVFSExplorer_getPath($gui{IVFSExplorer}));
    }
}

sub onsel_list_vmmhd() {
    my $hdref = &getsel_list_vmmhd();
    $gui{toolbuttonVMMCopy}->set_sensitive(1);
    $gui{toolbuttonVMMModify}->set_sensitive(1);

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$hdref{IMedium})) {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    elsif ($$hdref{Attached} eq '<None>') {
        $gui{toolbuttonVMMRemove}->set_sensitive(1);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    else {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(1);
    }

    $gui{labelVMMTypeField}->set_text($$hdref{Type});
    $gui{labelVMMAttachedToField}->set_text($$hdref{Attached});
    $gui{labelVMMLocationField}->set_text($$hdref{Location});
}

sub onsel_list_vmmdvd() {
    my $dref = &getsel_list_vmmdvd();
    $gui{toolbuttonVMMCopy}->set_sensitive(0);
    $gui{toolbuttonVMMModify}->set_sensitive(0);

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$dref{IMedium})) {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    elsif ($$dref{Attached} eq '<None>') {
        $gui{toolbuttonVMMRemove}->set_sensitive(1);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    else {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(1);
    }

    $gui{labelVMMTypeField}->set_text($$dref{Type});
    $gui{labelVMMAttachedToField}->set_text($$dref{Attached});
    $gui{labelVMMLocationField}->set_text($$dref{Location});
}

sub onsel_list_vmmfloppy() {
    my $fref = &getsel_list_vmmfloppy();
    $gui{toolbuttonVMMCopy}->set_sensitive(0);
    $gui{toolbuttonVMMModify}->set_sensitive(0);

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$fref{IMedium})) {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    elsif ($$fref{Attached} eq '<None>') {
        $gui{toolbuttonVMMRemove}->set_sensitive(1);
        $gui{toolbuttonVMMRelease}->set_sensitive(0);
    }
    else {
        $gui{toolbuttonVMMRemove}->set_sensitive(0);
        $gui{toolbuttonVMMRelease}->set_sensitive(1);
    }

    $gui{labelVMMTypeField}->set_text($$fref{Type});
    $gui{labelVMMAttachedToField}->set_text($$fref{Attached});
    $gui{labelVMMLocationField}->set_text($$fref{Location});
}

# Activates when selecting an item in the edit storage list, could be reduce a little
# as a lot of the options are the same for each controller but this gives flexibility
# to expand
sub onsel_list_editstorage() {
    my $storref = &getsel_list_editstorage();

    if ($$storref{IsController}) {
        &sens_editstor_allctr();
        $gui{comboboxEditStorCtrType}->signal_handler_block($signal{stortype});
        $gui{entryEditStorCtrName}->set_text($$storref{ControllerName});
        $gui{checkbuttonEditStorCache}->set_active($bool{IStorageController_getUseHostIOCache($$storref{IStorageController})});
        my $variant = IStorageController_getControllerType($$storref{IStorageController});

        if ($$storref{Bus} eq 'IDE') {
            &sens_editstor_ide_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorIDECtrType});
        }
        elsif ($$storref{Bus} eq 'SATA') {
            &sens_editstor_sata_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSATACtrType});
            $gui{adjustmentEditStorPortCount}->set_value(IStorageController_getPortCount($$storref{IStorageController}));

        }
        elsif ($$storref{Bus} eq 'Floppy') {
            &sens_editstor_floppy_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorFloppyCtrType});
        }
        elsif ($$storref{Bus} eq 'SAS') {
            &sens_editstor_sas_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSASCtrType});
        }
        else { # Defaults to SCSI
            &sens_editstor_scsi_ctr();
            $gui{comboboxEditStorCtrType}->set_model($gui{liststoreEditStorSCSICtrType});
        }

        &combobox_set_active_text($gui{comboboxEditStorCtrType}, $variant);
        $gui{comboboxEditStorCtrType}->signal_handler_unblock($signal{stortype});
    }
    else {
        &sens_editstor_allmedia();

        if ($$storref{MediumType} eq 'DVD') {
            my $attach = IMachine_getMediumAttachment($vmc{IMachine}, $$storref{ControllerName}, $$storref{Port}, $$storref{Device});
            $gui{checkbuttonEditStorLive}->set_active($bool{$$attach{temporaryEject}});
            &sens_editstor_dvd();
        }
        elsif ($$storref{MediumType} eq 'Floppy') { &sens_editstor_floppy(); }
        else { # Default to HD
            my $attach = IMachine_getMediumAttachment($vmc{IMachine}, $$storref{ControllerName}, $$storref{Port}, $$storref{Device});
            $gui{checkbuttonEditStorSSD}->set_active($bool{$$attach{nonRotational}});
            &sens_editstor_hd();
        }

        # We also need to setup the port comboboxEditStorDevPort
        if ($$storref{Bus} eq 'SATA') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSATA});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'IDE') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortIDE});
            if ($$storref{Device} == 0 and $$storref{Port} == 0) {
                $gui{comboboxEditStorDevPort}->set_active(0);
            }
            elsif ($$storref{Device} == 1 and $$storref{Port} == 0) {
                $gui{comboboxEditStorDevPort}->set_active(1);
            }
            elsif ($$storref{Device} == 0 and $$storref{Port} == 1) {
                $gui{comboboxEditStorDevPort}->set_active(2);
            }
            elsif ($$storref{Device} == 1 and $$storref{Port} == 1) {
                $gui{comboboxEditStorDevPort}->set_active(3);
            }
        }
        elsif ($$storref{Bus} eq 'SAS') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSAS});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'SCSI') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortSCSI});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Port});
        }
        elsif ($$storref{Bus} eq 'Floppy') {
            $gui{comboboxEditStorDevPort}->set_model($gui{liststoreEditStorDevPortFloppy});
            $gui{comboboxEditStorDevPort}->set_active($$storref{Device});
        }
    }
}

1;

# ***
# Some versions of perl-Gtk2 (for example 1.223) appear to emit the "cursor-changed"
# signal for treeviews, when in fact the cursor hasn't changed. This leads to failure
# when trying to get the selected item because there isn't one! So I default to returning
# the first entry for "phantom" cursor changes. No doubt there's a better way of doing this
