/** @file
 * CPUM - CPU Monitor(/ Manager), Context Structures.
 */

/*
 * Copyright (C) 2006-2023 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * The contents of this file may alternatively be used under the terms
 * of the Common Development and Distribution License Version 1.0
 * (CDDL), a copy of it is provided in the "COPYING.CDDL" file included
 * in the VirtualBox distribution, in which case the provisions of the
 * CDDL are applicable instead of those of the GPL.
 *
 * You may elect to license modified versions of this file under the
 * terms and conditions of either the GPL or the CDDL or both.
 *
 * SPDX-License-Identifier: GPL-3.0-only OR CDDL-1.0
 */



#pragma D  depends_on library x86.d




/** @defgroup grp_cpum_ctx  The CPUM Context Structures
 * @ingroup grp_cpum
 * @{
 */

/**
 * Selector hidden registers.
 */
typedef struct CPUMSELREG
{
    /** The selector register. */
    RTSEL       Sel;
    /** Padding, don't use. */
    RTSEL       PaddingSel;
    /** The selector which info resides in u64Base, u32Limit and Attr, provided
     * that CPUMSELREG_FLAGS_VALID is set. */
    RTSEL       ValidSel;
    /** Flags, see CPUMSELREG_FLAGS_XXX. */
    uint16_t    fFlags;

    /** Base register.
     *
     * Long mode remarks:
     *  - Unused in long mode for CS, DS, ES, SS
     *  - 32 bits for FS & GS; FS(GS)_BASE msr used for the base address
     *  - 64 bits for TR & LDTR
     */
    uint64_t    u64Base;
    /** Limit (expanded). */
    uint32_t    u32Limit;
    /** Flags.
     * This is the high 32-bit word of the descriptor entry.
     * Only the flags, dpl and type are used. */
    X86DESCATTR Attr;
} CPUMSELREG;

/** @name CPUMSELREG_FLAGS_XXX - CPUMSELREG::fFlags values.
 * @{ */
inline uint16_t CPUMSELREG_FLAGS_VALID = 0x0001;

inline uint16_t CPUMSELREG_FLAGS_STALE = 0x0002;

inline uint16_t CPUMSELREG_FLAGS_VALID_MASK = 0x0003;

/** @} */

/** Checks if the hidden parts of the selector register are valid. */


/** Old type used for the hidden register part.
 * @deprecated  */
typedef CPUMSELREG CPUMSELREGHID;

/**
 * The sysenter register set.
 */
typedef struct CPUMSYSENTER
{
    /** Ring 0 cs.
     * This value +  8 is the Ring 0 ss.
     * This value + 16 is the Ring 3 cs.
     * This value + 24 is the Ring 3 ss.
     */
    uint64_t    cs;
    /** Ring 0 eip. */
    uint64_t    eip;
    /** Ring 0 esp. */
    uint64_t    esp;
} CPUMSYSENTER;

/** @def CPUM_UNION_NM
 * For compilers (like DTrace) that does not grok nameless unions, we have a
 * little hack to make them palatable.
 */
/** @def CPUM_STRUCT_NM
 * For compilers (like DTrace) that does not grok nameless structs (it is
 * non-standard C++), we have a little hack to make them palatable.
 */


/** @def CPUM_UNION_STRUCT_NM
 * Combines CPUM_UNION_NM and CPUM_STRUCT_NM to avoid hitting the right side of
 * the screen in the compile time assertions.
 */


/** A general register (union). */
typedef union CPUMCTXGREG
{
    /** Natural unsigned integer view. */
    uint64_t            u;
    /** 64-bit view. */
    uint64_t            u64;
    /** 32-bit view. */
    uint32_t            u32;
    /** 16-bit view. */
    uint16_t            u16;
    /** 8-bit view. */
    uint8_t             u8;
    /** 8-bit low/high view.    */
     struct
    {
        /** Low byte (al, cl, dl, bl, ++). */
        uint8_t         bLo;
        /** High byte in the first word - ah, ch, dh, bh. */
        uint8_t         bHi;
    } s  ;
} CPUMCTXGREG;



/**
 * SVM Host-state area (Nested Hw.virt - VirtualBox's layout).
 *
 * @warning Exercise caution while modifying the layout of this struct. It's
 *          part of VM saved states.
 */
typedef struct SVMHOSTSTATE
{
    uint64_t    uEferMsr;
    uint64_t    uCr0;
    uint64_t    uCr4;
    uint64_t    uCr3;
    uint64_t    uRip;
    uint64_t    uRsp;
    uint64_t    uRax;
    X86RFLAGS   rflags;
    CPUMSELREG  es;
    CPUMSELREG  cs;
    CPUMSELREG  ss;
    CPUMSELREG  ds;
    VBOXGDTR    gdtr;
    VBOXIDTR    idtr;
    uint8_t     abPadding[4];
} SVMHOSTSTATE;
/** Pointer to the SVMHOSTSTATE structure. */
typedef SVMHOSTSTATE *PSVMHOSTSTATE;
/** Pointer to a const SVMHOSTSTATE structure. */
typedef const SVMHOSTSTATE *PCSVMHOSTSTATE;


/**
 * CPU hardware virtualization types.
 */
typedef enum
{
    CPUMHWVIRT_NONE = 0,
    CPUMHWVIRT_VMX,
    CPUMHWVIRT_SVM,
    CPUMHWVIRT_32BIT_HACK = 0x7fffffff
} CPUMHWVIRT;

/** Number of EFLAGS bits we put aside for the hardware EFLAGS, with the bits
 * above this we use for storing internal state not visible to the guest.
 *
 * Using a value less than 32 here means some code bloat when loading and
 * fetching the hardware EFLAGS value.  Comparing VMMR0.r0 text size when
 * compiling release build using gcc 11.3.1 on linux:
 *      - 32 bits: 2475709 bytes
 *      - 24 bits: 2482069 bytes; +6360 bytes.
 *      - 22 bits: 2482261 bytes; +6552 bytes.
 * Same for windows (virtual size of .text):
 *      - 32 bits: 1498502 bytes
 *      - 24 bits: 1502278 bytes; +3776 bytes.
 *      - 22 bits: 1502198 bytes; +3696 bytes.
 *
 * In addition we pass pointer the 32-bit EFLAGS to a number of IEM assembly
 * functions, so it would be safer to not store anything in the lower 32 bits.
 * OTOH, we'd sooner discover buggy assembly code by doing so, as we've had one
 * example of accidental EFLAGS trashing by these functions already.
 *
 * It would be more efficient for IEM to store the interrupt shadow bit (and
 * anything else that needs to be cleared at the same time) in the 30:22 bit
 * range, because that would allow using a simple AND imm32 instruction on x86
 * and a MOVN imm16,16 instruction to load the constant on ARM64 (assuming the
 * other flag needing clearing is RF (bit 16)).  Putting it in the 63:32 range
 * means we that on x86 we'll either use a memory variant of AND or require a
 * separate load instruction for the immediate, whereas on ARM we'll need more
 * instructions to construct the immediate value.
 *
 * Comparing the instruction exit thruput via the bs2-test-1 testcase, there
 * seems to be little difference between 32 and 24 here (best results out of 9
 * runs on Linux/VT-x).  So, unless the results are really wrong and there is
 * clear drop in thruput, it would on the whole make the most sense to use 24
 * here.
 *
 * Update: We need more than 8 bits because of DBGF, so using 22 now.
 */
inline uint8_t CPUMX86EFLAGS_HW_BITS = 22;

/** Mask for the hardware EFLAGS bits, 64-bit version. */

/** Mask for the hardware EFLAGS bits, 32-bit version. */


/** Mask of internal flags kept with EFLAGS, 64-bit version.
 * Bits 22-24 are taken by CPUMCTX_INHIBIT_SHADOW_SS, CPUMCTX_INHIBIT_SHADOW_STI
 * and CPUMCTX_INHIBIT_NMI, bits 25-28 are for CPUMCTX_DBG_HIT_DRX_MASK, and
 * bits 29-30 are for DBGF events and breakpoints.
 *
 * @todo The two DBGF bits could be merged.  The NMI inhibiting could move to
 *       bit 32 or higher as it isn't automatically cleared on instruction
 *       completion (except for iret).
 */
inline uint64_t CPUMX86EFLAGS_INT_MASK_64 = 0x00000000ffc00000;

/** Mask of internal flags kept with EFLAGS, 32-bit version. */
inline uint32_t CPUMX86EFLAGS_INT_MASK_32 = 0xffc00000;



/**
 * CPUM EFLAGS.
 *
 * This differs from X86EFLAGS in that we could use bits 31:22 for internal
 * purposes, see CPUMX86EFLAGS_HW_BITS.
 */
typedef union CPUMX86EFLAGS
{
    /** The full unsigned view, both hardware and VBox bits. */
    uint32_t        uBoth;
    /** The plain unsigned view of the hardware bits. */
    uint32_t        u : 22 ;
} CPUMX86EFLAGS;
/** Pointer to CPUM EFLAGS. */
typedef CPUMX86EFLAGS *PCPUMX86EFLAGS;
/** Pointer to const CPUM EFLAGS. */
typedef const CPUMX86EFLAGS *PCCPUMX86EFLAGS;

/**
 * CPUM RFLAGS.
 *
 * This differs from X86EFLAGS in that we use could be using bits 63:22 for
 * internal purposes, see CPUMX86EFLAGS_HW_BITS.
 */
typedef union CPUMX86RFLAGS
{
    /** The full unsigned view, both hardware and VBox bits. */
    uint64_t        uBoth;
    /** The plain unsigned view of the hardware bits. */
    uint32_t        u : 22 ;
} CPUMX86RFLAGS;
/** Pointer to CPUM RFLAGS. */
typedef CPUMX86RFLAGS *PCPUMX86RFLAGS;
/** Pointer to const CPUM RFLAGS. */
typedef const CPUMX86RFLAGS *PCCPUMX86RFLAGS;


/**
 * CPU context.
 */
typedef struct CPUMCTX
{
    /** General purpose registers. */
    union /* no tag! */
    {
        /** The general purpose register array view, indexed by X86_GREG_XXX. */
        CPUMCTXGREG     aGRegs[16];

        /** 64-bit general purpose register view. */
         struct /* no tag! */
        {
            uint64_t    rax, rcx, rdx, rbx, rsp, rbp, rsi, rdi, r8, r9, r10, r11, r12, r13, r14, r15;
        } qw  ;
        /** 64-bit general purpose register view. */
         struct /* no tag! */
        {
            uint64_t    r0, r1, r2, r3, r4, r5, r6, r7;
        } qw2  ;
        /** 32-bit general purpose register view. */
         struct /* no tag! */
        {
            uint32_t     eax, u32Pad00,      ecx, u32Pad01,      edx, u32Pad02,      ebx, u32Pad03,
                         esp, u32Pad04,      ebp, u32Pad05,      esi, u32Pad06,      edi, u32Pad07,
                         r8d, u32Pad08,      r9d, u32Pad09,     r10d, u32Pad10,     r11d, u32Pad11,
                        r12d, u32Pad12,     r13d, u32Pad13,     r14d, u32Pad14,     r15d, u32Pad15;
        } dw  ;
        /** 16-bit general purpose register view. */
         struct /* no tag! */
        {
            uint16_t      ax, au16Pad00[3],   cx, au16Pad01[3],   dx, au16Pad02[3],   bx, au16Pad03[3],
                          sp, au16Pad04[3],   bp, au16Pad05[3],   si, au16Pad06[3],   di, au16Pad07[3],
                         r8w, au16Pad08[3],  r9w, au16Pad09[3], r10w, au16Pad10[3], r11w, au16Pad11[3],
                        r12w, au16Pad12[3], r13w, au16Pad13[3], r14w, au16Pad14[3], r15w, au16Pad15[3];
        } w  ;
         struct /* no tag! */
        {
            uint8_t   al, ah, abPad00[6], cl, ch, abPad01[6], dl, dh, abPad02[6], bl, bh, abPad03[6],
                         spl, abPad04[7],    bpl, abPad05[7],    sil, abPad06[7],    dil, abPad07[7],
                         r8l, abPad08[7],    r9l, abPad09[7],   r10l, abPad10[7],   r11l, abPad11[7],
                        r12l, abPad12[7],   r13l, abPad13[7],   r14l, abPad14[7],   r15l, abPad15[7];
        } b  ;
    } g  ;

    /** Segment registers. */
    union /* no tag! */
    {
        /** The segment register array view, indexed by X86_SREG_XXX. */
        CPUMSELREG      aSRegs[6];
        /** The named segment register view. */
         struct /* no tag! */
        {
            CPUMSELREG  es, cs, ss, ds, fs, gs;
        } n  ;
    } s  ;

    /** The task register.
     * Only the guest context uses all the members. */
    CPUMSELREG          ldtr;
    /** The task register.
     * Only the guest context uses all the members. */
    CPUMSELREG          tr;

    /** The program counter. */
    union
    {
        uint16_t        ip;
        uint32_t        eip;
        uint64_t        rip;
    } rip  ;

    /** The flags register. */
    union
    {
        CPUMX86EFLAGS   eflags;
        CPUMX86RFLAGS   rflags;
    } rflags  ;

    /** 0x150 - Externalized state tracker, CPUMCTX_EXTRN_XXX. */
    uint64_t            fExtrn;

    /** The RIP value an interrupt shadow is/was valid for. */
    uint64_t            uRipInhibitInt;

    /** @name Control registers.
     * @{ */
    uint64_t            cr0;
    uint64_t            cr2;
    uint64_t            cr3;
    uint64_t            cr4;
    /** @} */

    /** Debug registers.
     * @remarks DR4 and DR5 should not be used since they are aliases for
     *          DR6 and DR7 respectively on both AMD and Intel CPUs.
     * @remarks DR8-15 are currently not supported by AMD or Intel, so
     *          neither do we.
     */
    uint64_t            dr[8];

    /** Padding before the structure so the 64-bit member is correctly aligned.
     * @todo fix this structure!  */
    uint16_t            gdtrPadding[3];
    /** Global Descriptor Table register. */
    VBOXGDTR            gdtr;

    /** Padding before the structure so the 64-bit member is correctly aligned.
     * @todo fix this structure!  */
    uint16_t            idtrPadding[3];
    /** Interrupt Descriptor Table register. */
    VBOXIDTR            idtr;

    /** The sysenter msr registers.
     * This member is not used by the hypervisor context. */
    CPUMSYSENTER        SysEnter;

    /** @name System MSRs.
     * @{ */
    uint64_t            msrEFER; /**< @todo move EFER up to the crX registers for better cacheline mojo */
    uint64_t            msrSTAR;            /**< Legacy syscall eip, cs & ss. */
    uint64_t            msrPAT;             /**< Page attribute table. */
    uint64_t            msrLSTAR;           /**< 64 bits mode syscall rip. */
    uint64_t            msrCSTAR;           /**< Compatibility mode syscall rip. */
    uint64_t            msrSFMASK;          /**< syscall flag mask. */
    uint64_t            msrKERNELGSBASE;    /**< swapgs exchange value. */
    /** @} */

    uint64_t            au64Unused[2];

    /** 0x240 - PAE PDPTEs. */
    X86PDPE             aPaePdpes[4];

    /** 0x260 - The XCR0..XCR1 registers. */
    uint64_t            aXcr[2];
    /** 0x270 - The mask to pass to XSAVE/XRSTOR in EDX:EAX.  If zero we use
     *  FXSAVE/FXRSTOR (since bit 0 will always be set, we only need to test it). */
    uint64_t            fXStateMask;
    /** 0x278 - Mirror of CPUMCPU::fUseFlags[CPUM_USED_FPU_GUEST]. */
    uint8_t                fUsedFpuGuest;
    uint8_t             afUnused[7];

    /* ---- Start of members not zeroed at reset. ---- */

    /** 0x280 - State component offsets into pXState, UINT16_MAX if not present.
     * @note Everything before this member will be memset to zero during reset. */
    uint16_t            aoffXState[64];
    /** 0x300 - The extended state (FPU/SSE/AVX/AVX-2/XXXX).
     * Aligned on 256 byte boundrary (min req is currently 64 bytes). */
    union /* no tag */
    {
        X86XSAVEAREA    XState;
        /** Byte view for simple indexing and space allocation. */
        uint8_t         abXState[0x4000 - 0x300];
    } u  ;

    /** 0x4000 - Hardware virtualization state.
     * @note This is page aligned, so an full page member comes first in the
     *       substructures. */
    struct
    {
        union   /* no tag! */
        {
            struct
            {
                /** 0x4000 - Cache of the nested-guest VMCB. */
                SVMVMCB                 Vmcb;
                /** 0x5000 - The MSRPM (MSR Permission bitmap).
                 *
                 * This need not be physically contiguous pages because we use the one from
                 * HMPHYSCPU while executing the nested-guest using hardware-assisted SVM.
                 * This one is just used for caching the bitmap from guest physical memory.
                 *
                 * @todo r=bird: This is not used directly by AMD-V hardware, so it doesn't
                 *       really need to even be page aligned.
                 *
                 *       Also, couldn't we just access the guest page directly when we need to,
                 *       or do we have to use a cached copy of it? */
                uint8_t                 abMsrBitmap[SVM_MSRPM_PAGES * X86_PAGE_SIZE];
                /** 0x7000 - The IOPM (IO Permission bitmap).
                 *
                 * This need not be physically contiguous pages because we re-use the ring-0
                 * allocated IOPM while executing the nested-guest using hardware-assisted SVM
                 * because it's identical (we trap all IO accesses).
                 *
                 * This one is just used for caching the IOPM from guest physical memory in
                 * case the guest hypervisor allows direct access to some IO ports.
                 *
                 * @todo r=bird: This is not used directly by AMD-V hardware, so it doesn't
                 *       really need to even be page aligned.
                 *
                 *       Also, couldn't we just access the guest page directly when we need to,
                 *       or do we have to use a cached copy of it? */
                uint8_t                 abIoBitmap[SVM_IOPM_PAGES * X86_PAGE_SIZE];

                /** 0xa000 - MSR holding physical address of the Guest's Host-state. */
                uint64_t                uMsrHSavePa;
                /** 0xa008 - Guest physical address of the nested-guest VMCB. */
                RTGCPHYS                GCPhysVmcb;
                /** 0xa010 - Guest's host-state save area. */
                SVMHOSTSTATE            HostState;
                /** 0xa0c8 - Guest TSC time-stamp of when the previous PAUSE instr. was
                 *  executed. */
                uint64_t                uPrevPauseTick;
                /** 0xa0d0 - Pause filter count. */
                uint16_t                cPauseFilter;
                /** 0xa0d2 - Pause filter threshold. */
                uint16_t                cPauseFilterThreshold;
                /** 0xa0d4 - Whether the injected event is subject to event intercepts. */
                uint8_t                    fInterceptEvents;
                /** 0xa0d5 - Padding. */
                uint8_t                    afPadding[3];
            } svm;

            struct
            {
                /** 0x4000 - The current VMCS. */
                VMXVVMCS                Vmcs;
                /** 0X5000 - The shadow VMCS. */
                VMXVVMCS                ShadowVmcs;
                /** 0x6000 - The VMREAD bitmap.
                 * @todo r=bird: Do we really need to keep copies for these?  Couldn't we just
                 *       access the guest memory directly as needed?   */
                uint8_t                 abVmreadBitmap[VMX_V_VMREAD_VMWRITE_BITMAP_SIZE];
                /** 0x7000 - The VMWRITE bitmap.
                 * @todo r=bird: Do we really need to keep copies for these?  Couldn't we just
                 *       access the guest memory directly as needed?  */
                uint8_t                 abVmwriteBitmap[VMX_V_VMREAD_VMWRITE_BITMAP_SIZE];
                /** 0x8000 - The VM-entry MSR-load area. */
                VMXAUTOMSR              aEntryMsrLoadArea[VMX_V_AUTOMSR_AREA_SIZE / sizeof(VMXAUTOMSR)];
                /** 0xa000 - The VM-exit MSR-store area. */
                VMXAUTOMSR              aExitMsrStoreArea[VMX_V_AUTOMSR_AREA_SIZE / sizeof(VMXAUTOMSR)];
                /** 0xc000 - The VM-exit MSR-load area. */
                VMXAUTOMSR              aExitMsrLoadArea[VMX_V_AUTOMSR_AREA_SIZE / sizeof(VMXAUTOMSR)];
                /** 0xe000 - The MSR permission bitmap.
                 * @todo r=bird: Do we really need to keep copies for these?  Couldn't we just
                 *       access the guest memory directly as needed?  */
                uint8_t                 abMsrBitmap[VMX_V_MSR_BITMAP_SIZE];
                /** 0xf000 - The I/O permission bitmap.
                 * @todo r=bird: Do we really need to keep copies for these?  Couldn't we just
                 *       access the guest memory directly as needed? */
                uint8_t                 abIoBitmap[VMX_V_IO_BITMAP_A_SIZE + VMX_V_IO_BITMAP_B_SIZE];

                /** 0x11000 - Guest physical address of the VMXON region. */
                RTGCPHYS                GCPhysVmxon;
                /** 0x11008 - Guest physical address of the current VMCS pointer. */
                RTGCPHYS                GCPhysVmcs;
                /** 0x11010 - Guest physical address of the shadow VMCS pointer. */
                RTGCPHYS                GCPhysShadowVmcs;
                /** 0x11018 - Last emulated VMX instruction/VM-exit diagnostic. */
                VMXVDIAG                enmDiag;
                /** 0x1101c - VMX abort reason. */
                VMXABORT                enmAbort;
                /** 0x11020 - Last emulated VMX instruction/VM-exit diagnostic auxiliary info.
                 *  (mainly used for info. that's not part of the VMCS). */
                uint64_t                uDiagAux;
                /** 0x11028 - VMX abort auxiliary info. */
                uint32_t                uAbortAux;
                /** 0x1102c - Whether the guest is in VMX root mode. */
                uint8_t                    fInVmxRootMode;
                /** 0x1102d - Whether the guest is in VMX non-root mode. */
                uint8_t                    fInVmxNonRootMode;
                /** 0x1102e - Whether the injected events are subjected to event intercepts.  */
                uint8_t                    fInterceptEvents;
                /** 0x1102f - Whether blocking of NMI (or virtual-NMIs) was in effect in VMX
                 *  non-root mode before execution of IRET. */
                uint8_t                    fNmiUnblockingIret;
                /** 0x11030 - Guest TSC timestamp of the first PAUSE instruction that is
                 *  considered to be the first in a loop. */
                uint64_t                uFirstPauseLoopTick;
                /** 0x11038 - Guest TSC timestamp of the previous PAUSE instruction. */
                uint64_t                uPrevPauseTick;
                /** 0x11040 - Guest TSC timestamp of VM-entry (used for VMX-preemption
                 *  timer). */
                uint64_t                uEntryTick;
                /** 0x11048 - Virtual-APIC write offset (until trap-like VM-exit). */
                uint16_t                offVirtApicWrite;
                /** 0x1104a - Whether virtual-NMI blocking is in effect. */
                uint8_t                    fVirtNmiBlocking;
                /** 0x1104b - Padding. */
                uint8_t                 abPadding0[5];
                /** 0x11050 - Guest VMX MSRs. */
                VMXMSRS                 Msrs;
            } vmx;
        } s  ;

        /** 0x11130 - Hardware virtualization type currently in use. */
        CPUMHWVIRT              enmHwvirt;
        /** 0x11134 - Global interrupt flag - AMD only (always true on Intel). */
        uint8_t                    fGif;
        /** 0x11135 - Padding. */
        uint8_t                    afPadding0[3];
        /** 0x11138 - A subset of guest inhibit flags (CPUMCTX_INHIBIT_XXX) that are
         *  saved while running the nested-guest. */
        uint32_t                fSavedInhibit;
        /** 0x1113c - Pad to 64 byte boundary. */
        uint8_t                 abPadding1[4];
    } hwvirt;
} CPUMCTX;



/** @name CPUMCTX_EXTRN_XXX
 * Used for parts of the CPUM state that is externalized and needs fetching
 * before use.
 *
 * @{ */
/** External state keeper: Invalid.  */
inline uint64_t CPUMCTX_EXTRN_KEEPER_INVALID = 0x0000000000000000;

/** External state keeper: HM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_HM = 0x0000000000000001;

/** External state keeper: NEM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_NEM = 0x0000000000000002;

/** External state keeper: REM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_REM = 0x0000000000000003;

/** External state keeper mask. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_MASK = 0x0000000000000003;


/** The RIP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RIP = 0x0000000000000004;

/** The RFLAGS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_RFLAGS = 0x0000000000000008;


/** The RAX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RAX = 0x0000000000000010;

/** The RCX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RCX = 0x0000000000000020;

/** The RDX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RDX = 0x0000000000000040;

/** The RBX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RBX = 0x0000000000000080;

/** The RSP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RSP = 0x0000000000000100;

/** The RBP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RBP = 0x0000000000000200;

/** The RSI register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RSI = 0x0000000000000400;

/** The RDI register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RDI = 0x0000000000000800;

/** The R8 thru R15 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_R8_R15 = 0x0000000000001000;

/** General purpose registers mask. */
inline uint64_t CPUMCTX_EXTRN_GPRS_MASK = 0x0000000000001ff0;


/** The ES register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_ES = 0x0000000000002000;

/** The CS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_CS = 0x0000000000004000;

/** The SS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SS = 0x0000000000008000;

/** The DS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_DS = 0x0000000000010000;

/** The FS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_FS = 0x0000000000020000;

/** The GS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_GS = 0x0000000000040000;

/** Segment registers (includes CS). */
inline uint64_t CPUMCTX_EXTRN_SREG_MASK = 0x000000000007e000;

/** Converts a X86_XREG_XXX index to a CPUMCTX_EXTRN_xS mask. */


/** The GDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_GDTR = 0x0000000000080000;

/** The IDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_IDTR = 0x0000000000100000;

/** The LDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_LDTR = 0x0000000000200000;

/** The TR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_TR = 0x0000000000400000;

/** Table register mask. */
inline uint64_t CPUMCTX_EXTRN_TABLE_MASK = 0x0000000000780000;


/** The CR0 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR0 = 0x0000000000800000;

/** The CR2 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR2 = 0x0000000001000000;

/** The CR3 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR3 = 0x0000000002000000;

/** The CR4 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR4 = 0x0000000004000000;

/** Control register mask. */
inline uint64_t CPUMCTX_EXTRN_CR_MASK = 0x0000000007800000;

/** The TPR/CR8 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_APIC_TPR = 0x0000000008000000;

/** The EFER register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_EFER = 0x0000000010000000;


/** The DR0, DR1, DR2 and DR3 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR0_DR3 = 0x0000000020000000;

/** The DR6 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR6 = 0x0000000040000000;

/** The DR7 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR7 = 0x0000000080000000;

/** Debug register mask. */
inline uint64_t CPUMCTX_EXTRN_DR_MASK = 0x00000000e0000000;


/** The XSAVE_C_X87 state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_X87 = 0x0000000100000000;

/** The XSAVE_C_SSE, XSAVE_C_YMM, XSAVE_C_ZMM_HI256, XSAVE_C_ZMM_16HI and
 * XSAVE_C_OPMASK state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_SSE_AVX = 0x0000000200000000;

/** The state of XSAVE components not covered by CPUMCTX_EXTRN_X87 and
 * CPUMCTX_EXTRN_SEE_AVX is kept externally. */
inline uint64_t CPUMCTX_EXTRN_OTHER_XSAVE = 0x0000000400000000;

/** The state of XCR0 and XCR1 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_XCRx = 0x0000000800000000;



/** The KERNEL GS BASE MSR value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_KERNEL_GS_BASE = 0x0000001000000000;

/** The STAR, LSTAR, CSTAR and SFMASK MSR values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SYSCALL_MSRS = 0x0000002000000000;

/** The SYSENTER_CS, SYSENTER_EIP and SYSENTER_ESP MSR values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SYSENTER_MSRS = 0x0000004000000000;

/** The TSC_AUX MSR is kept externally. */
inline uint64_t CPUMCTX_EXTRN_TSC_AUX = 0x0000008000000000;

/** All other stateful MSRs not covered by CPUMCTX_EXTRN_EFER,
 * CPUMCTX_EXTRN_KERNEL_GS_BASE, CPUMCTX_EXTRN_SYSCALL_MSRS,
 * CPUMCTX_EXTRN_SYSENTER_MSRS, and CPUMCTX_EXTRN_TSC_AUX.  */
inline uint64_t CPUMCTX_EXTRN_OTHER_MSRS = 0x0000010000000000;


/** Mask of all the MSRs. */


/** Hardware-virtualization (SVM or VMX) state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_HWVIRT = 0x0000020000000000;


/** Inhibit maskable interrupts (VMCPU_FF_INHIBIT_INTERRUPTS) */
inline uint64_t CPUMCTX_EXTRN_INHIBIT_INT = 0x0000040000000000;

/** Inhibit non-maskable interrupts (VMCPU_FF_BLOCK_NMIS). */
inline uint64_t CPUMCTX_EXTRN_INHIBIT_NMI = 0x0000080000000000;


/** Mask of bits the keepers can use for state tracking. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_STATE_MASK = 0xffff000000000000;


/** NEM/Win: Event injection (known was interruption) pending state. */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_EVENT_INJECT = 0x0001000000000000;

/** NEM/Win: Mask. */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_MASK = 0x0001000000000000;


/** HM/SVM: Nested-guest interrupt pending (VMCPU_FF_INTERRUPT_NESTED_GUEST). */
inline uint64_t CPUMCTX_EXTRN_HM_SVM_HWVIRT_VIRQ = 0x0001000000000000;

/** HM/SVM: Mask. */
inline uint64_t CPUMCTX_EXTRN_HM_SVM_MASK = 0x0001000000000000;


/** All CPUM state bits, not including keeper specific ones. */
inline uint64_t CPUMCTX_EXTRN_ALL = 0x00000ffffffffffc;

/** All CPUM state bits, including keeper specific ones. */
inline uint64_t CPUMCTX_EXTRN_ABSOLUTELY_ALL = 0xfffffffffffffffc;

/** @} */


/** @name CPUMCTX_INHIBIT_XXX - Interrupt inhibiting flags.
 * @{ */
/** Interrupt shadow following MOV SS or POP SS.
 *
 * When this in effect, both maskable and non-maskable interrupts are blocked
 * from delivery for one instruction.  Same for certain debug exceptions too,
 * unlike the STI variant.
 *
 * It is implementation specific whether a sequence of two or more of these
 * instructions will have any effect on the instruction following the last one
 * of them. */
inline uint32_t CPUMCTX_INHIBIT_SHADOW_SS = 1U << 0;

/** Interrupt shadow following STI.
 * Same as CPUMCTX_INHIBIT_SHADOW_SS but without blocking any debug exceptions. */
inline uint32_t CPUMCTX_INHIBIT_SHADOW_STI = 1U << 1;

/** Mask combining STI and SS shadowing. */


/** Interrupts blocked by NMI delivery.  This condition is cleared by IRET.
 *
 * Section "6.7 NONMASKABLE INTERRUPT (NMI)" in Intel SDM Vol 3A states that
 * "The processor also invokes certain hardware conditions to ensure that no
 * other interrupts, including NMI interrupts, are received until the NMI
 * handler has completed executing."  This flag indicates that these
 * conditions are currently active.
 *
 * @todo this does not really need to be in the lower 32-bits of EFLAGS.
 */
inline uint32_t CPUMCTX_INHIBIT_NMI = 1U << 2;


/** Mask containing all the interrupt inhibit bits. */

 ;
/** @} */

/** @name CPUMCTX_DBG_XXX - Pending debug events.
 * @{ */
/** Hit guest DR0 breakpoint. */


/** Hit guest DR1 breakpoint. */


/** Hit guest DR2 breakpoint. */


/** Hit guest DR3 breakpoint. */


/** Shift for the CPUMCTX_DBG_HIT_DRx bits. */

/** Mask of all guest pending DR0-DR3 breakpoint indicators. */

/** DBGF event/breakpoint pending. */


/** DBGF event/breakpoint pending. */


/** Mask of all DBGF indicators. */

 ;
/** @}  */



/**
 * Additional guest MSRs (i.e. not part of the CPU context structure).
 *
 * @remarks Never change the order here because of the saved stated!  The size
 *          can in theory be changed, but keep older VBox versions in mind.
 */
typedef union CPUMCTXMSRS
{
    struct
    {
        uint64_t    TscAux;             /**< MSR_K8_TSC_AUX */
        uint64_t    MiscEnable;         /**< MSR_IA32_MISC_ENABLE */
        uint64_t    MtrrDefType;        /**< IA32_MTRR_DEF_TYPE */
        uint64_t    MtrrFix64K_00000;   /**< IA32_MTRR_FIX16K_80000 */
        uint64_t    MtrrFix16K_80000;   /**< IA32_MTRR_FIX16K_80000 */
        uint64_t    MtrrFix16K_A0000;   /**< IA32_MTRR_FIX16K_A0000 */
        uint64_t    MtrrFix4K_C0000;    /**< IA32_MTRR_FIX4K_C0000 */
        uint64_t    MtrrFix4K_C8000;    /**< IA32_MTRR_FIX4K_C8000 */
        uint64_t    MtrrFix4K_D0000;    /**< IA32_MTRR_FIX4K_D0000 */
        uint64_t    MtrrFix4K_D8000;    /**< IA32_MTRR_FIX4K_D8000 */
        uint64_t    MtrrFix4K_E0000;    /**< IA32_MTRR_FIX4K_E0000 */
        uint64_t    MtrrFix4K_E8000;    /**< IA32_MTRR_FIX4K_E8000 */
        uint64_t    MtrrFix4K_F0000;    /**< IA32_MTRR_FIX4K_F0000 */
        uint64_t    MtrrFix4K_F8000;    /**< IA32_MTRR_FIX4K_F8000 */
        uint64_t    PkgCStateCfgCtrl;   /**< MSR_PKG_CST_CONFIG_CONTROL */
        uint64_t    SpecCtrl;           /**< IA32_SPEC_CTRL */
        uint64_t    ArchCaps;           /**< IA32_ARCH_CAPABILITIES */
    } msr;
    uint64_t    au64[64];
} CPUMCTXMSRS;
/** Pointer to the guest MSR state. */
typedef CPUMCTXMSRS *PCPUMCTXMSRS;
/** Pointer to the const guest MSR state. */
typedef const CPUMCTXMSRS *PCCPUMCTXMSRS;

/** @}  */




