# Error Handling
use strict;
use warnings;
our (%gui);

my %err = (auth          => ['Failed to Authenticate',
                             'Login credentials are incorrect or the web service is mis-configured.'],
           connect       => ['Failed to Connect',
                             'Credentials are incorrect or the VirtualBox Web Service is not running at the specified URL.'],
           blankguest    => ['Blank Guest Name',
                             'You cannot create a new guest with a blank name.'],
           createguest   => ['Failed to Create Guest',
                             'This is usually due to a guest already existing with the same name.'],
           copydisk      => ['Failed to Copy Hard Disk',
                             'You do not have permission to write to the chosen location or you\'ve specified an invalid filename'],
           nocpuhot      => ['CPU Hot Plug Disabled',
                             'This guest does not have hot CPU hot plugging enabled. You must change this by editing the guest settings'],
           ctrallocated  => ['Controller Already Allocated',
                             'The guest already has a storage controller of this type assigned to it.'],
           ctrinuse      => ['Controller in Use',
                             'The controller still has media attached. Remove media before removing controller.'],
           ctrfull       => ['Controller Full',
                             'The selected controller is full. No more media can be attached.'],
           deletemedium  => ['Unable to Delete Medium',
                             'The selected medium is in a state which does not permit deletion. It' .
                             ' may be locked by another process.'],
           emptynamepath => ['Invalid Name or Path',
                             'You cannot specify an empty name or path.'],
           existattach   => ['Port and Device In Use',
                             'The chosen port already has a medium attached to it'],
           guiexception  => ['A GUI Exception Occurred',
                             ''],
           mediuminuse   => ['Medium In Use',
                             'The selected medium could not be released from at least one guest. This is because the guest is' .
                             ' running or is being edited by another user.'],
           noextensions  => ['Oracle VM VirtualBox Extension Pack Required',
                             'The server does not have the Oracle Extension Pack installed or the pack is outdated. ' .
                             'Please download the extension pack from the URL below and and install it on the ' .
                             "server. RemoteBox will continue but you may experience failures or loss of functionality.\n\n" .
                             'http://www.virtualbox.org/wiki/Downloads'],
           noscreenshot  => ['Screenshot Failed',
                             'Either the guest is not in a state suitable for taking a screenshot or your VirtualBox ' .
                             'installation does not support PNG images'],
           remotedisplay => ['Remote Display Not Enabled',
                             'The remote display server for this guest has not been enabled or is not running. If you are ' .
                             'certain it has been enabled, ensure its RDP port does not conflict with another guest'],
           restorefail   => ['Restore Failed',
                             'Cannot restore a snapshot while the guest is in use or powered on.'],
           savedesc      => ['Unable to Save Description',
                             'The description of this guest cannot be changed until its lock is released by the process which' .
                             ' set it. The guest may be running or being edited by another user.'],
           sessionopen   => ['Session In Use',
                             'This guest already has an open session associated with it, either from RemoteBox or another program.' .
                             ' Please close that session and try again.'],
           settings      => ['Unable to Edit Settings',
                             'The settings of this guest cannot be changed until its lock is released by the process which set it.' .
                             ' The guest may be running or being edited by another user.'],
           snapshotfail  => ['Snapshot Failed',
                             'A session to the guest could not be obtained.'],
           snapdelete    => ['Deleting Snapshot Failed',
                             'The snapshot could not be deleted because a session to the guest could not be obtained.'],
           snapdelchild  => ['Deleting Snapshot Failed',
                             'The snapshot could not be deleted because it has 2 or more children. You must delete the child snapshots first.'],
           startguest    => ['Failed to Start Guest',
                             ''],
           transport     => ['Server Closed Connection',
                             'The server has closed or invalidated the current connection. RemoteBox will try to recover but ' .
                             "you will need to reconnect.\n\n"],
           vboxver       => ['Unsupported VirtualBox Version',
                             'This version of RemoteBox is not intended for use with the version of VirtualBox running on the' .
                             " server.\nVisit http://remotebox.knobgoblin.org.uk and download an appropriate version of RemoteBox. " .
                             "RemoteBox will continue, but you may experience failures and loss of functionality.\n" .
                             'Supported VirtualBox Version: 4.2.x'],
           webservice    => ['VirtualBox Returned an Error',
                             '']);

sub show_err_msg() {
    my ($key, $dialog, $append) = @_;
    my ($title, $body) = @{ $err{$key} };
    $body .= $append if ($append);
    &addrow_log("Error Dialog:    $title:    $body");
    $dialog->set_markup("<big><b>$title</b></big>");
    # Filter out <> to avoid them being treated as markup
    $body =~ s/\</&lt;/g;
    $body =~ s/\>/&gt;/g;
    $dialog->format_secondary_markup($body);
    $dialog->run;
    $dialog->hide;
}

sub show_invalid_object_msg() {
    my ($append) = @_;
    my $body = 'VirtualBox has returned an invalid object reference error. If you continue, ' .
               'the connection may be in an unknown state. Disconnecting will try to recover ' .
               "but will require you to reconnect.\n\nAdditional:\n";
    $body .= $append if ($append);
    &addrow_log("Error Dialog:    Invalid Object Reference:    $body");
    $gui{messagedialogInvalidObject}->format_secondary_markup($body);

    my $response = $gui{messagedialogInvalidObject}->run;
    $gui{messagedialogInvalidObject}->hide;
    return $response;
}

# Callback which is triggered on a SOAP fault
sub vboxerror() {
    my ($soap, $res) = @_;

    if(ref($res)) {
        if ($res->faultstring =~ m/Invalid managed object reference/) {
            my $response = &show_invalid_object_msg($res->faultstring);
            if ($response eq 'cancel') {
                $gui{websn} = undef; # Invalidate current session to prevent deep recursion
                &virtualbox_logoff(1); # Cleanup - but logoff will not happen because sessions is invalidated
            }
            # OK means continue (ie do nothing) - Cancel means disconnect
        }
        else { &show_err_msg('webservice', $gui{messagedialogError}, $res->faultstring); }
    }
    else {
        $gui{websn} = undef; # Invalidate current session to prevent deep recursion
        &virtualbox_logoff(1); # Cleanup - but logoff will not happen because sessions is invalidated
        &show_err_msg('transport', $gui{messagedialogError}, $soap->transport->status);
    }
}

1;