/* $Id: whirlpool.c 227 2010-06-16 17:28:38Z tp $ */
/*
 * WHIRLPOOL implementation.
 *
 * Internally, we use little-endian convention, on the assumption that
 * architectures which favour big-endian encoding are:
 *   1. rarer
 *   2. in decreasing numbers
 *   3. able to decode little-endian data efficiently anyway
 *
 * The most common big-endian architecture is Sparc, and Ultrasparc CPU
 * include special opcodes to perform little-endian accesses, which we use
 * (see sph_types.h). Most modern CPU designs can work with both endianness
 * and architecture designer now favour little-endian (basically, x86 has
 * won the endianness war).
 *
 * TODO: implement a 32-bit version. Not only such a version would be handy
 * for non-64-bit-able architectures, but it may also use smaller tables,
 * at the expense of more lookups and XORs.
 *
 * ==========================(LICENSE BEGIN)============================
 *
 * Copyright (c) 2007-2010  Projet RNRT SAPHIR
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ===========================(LICENSE END)=============================
 *
 * @author   Thomas Pornin <thomas.pornin@cryptolog.com>
 */

#include <stddef.h>
#include <string.h>

#include "sph_whirlpool.h"
#include "memdbg.h"

#if SPH_64

#if SPH_SMALL_FOOTPRINT && !defined SPH_SMALL_FOOTPRINT_WHIRLPOOL
#define SPH_SMALL_FOOTPRINT_WHIRLPOOL   1
#endif

/* ====================================================================== */
/*
 * Constants for plain WHIRLPOOL (current version).
 */

static const sph_u64 plain_T0[256] = {
	SPH_C64(0xD83078C018601818), SPH_C64(0x2646AF05238C2323),
	SPH_C64(0xB891F97EC63FC6C6), SPH_C64(0xFBCD6F13E887E8E8),
	SPH_C64(0xCB13A14C87268787), SPH_C64(0x116D62A9B8DAB8B8),
	SPH_C64(0x0902050801040101), SPH_C64(0x0D9E6E424F214F4F),
	SPH_C64(0x9B6CEEAD36D83636), SPH_C64(0xFF510459A6A2A6A6),
	SPH_C64(0x0CB9BDDED26FD2D2), SPH_C64(0x0EF706FBF5F3F5F5),
	SPH_C64(0x96F280EF79F97979), SPH_C64(0x30DECE5F6FA16F6F),
	SPH_C64(0x6D3FEFFC917E9191), SPH_C64(0xF8A407AA52555252),
	SPH_C64(0x47C0FD27609D6060), SPH_C64(0x35657689BCCABCBC),
	SPH_C64(0x372BCDAC9B569B9B), SPH_C64(0x8A018C048E028E8E),
	SPH_C64(0xD25B1571A3B6A3A3), SPH_C64(0x6C183C600C300C0C),
	SPH_C64(0x84F68AFF7BF17B7B), SPH_C64(0x806AE1B535D43535),
	SPH_C64(0xF53A69E81D741D1D), SPH_C64(0xB3DD4753E0A7E0E0),
	SPH_C64(0x21B3ACF6D77BD7D7), SPH_C64(0x9C99ED5EC22FC2C2),
	SPH_C64(0x435C966D2EB82E2E), SPH_C64(0x29967A624B314B4B),
	SPH_C64(0x5DE121A3FEDFFEFE), SPH_C64(0xD5AE168257415757),
	SPH_C64(0xBD2A41A815541515), SPH_C64(0xE8EEB69F77C17777),
	SPH_C64(0x926EEBA537DC3737), SPH_C64(0x9ED7567BE5B3E5E5),
	SPH_C64(0x1323D98C9F469F9F), SPH_C64(0x23FD17D3F0E7F0F0),
	SPH_C64(0x20947F6A4A354A4A), SPH_C64(0x44A9959EDA4FDADA),
	SPH_C64(0xA2B025FA587D5858), SPH_C64(0xCF8FCA06C903C9C9),
	SPH_C64(0x7C528D5529A42929), SPH_C64(0x5A1422500A280A0A),
	SPH_C64(0x507F4FE1B1FEB1B1), SPH_C64(0xC95D1A69A0BAA0A0),
	SPH_C64(0x14D6DA7F6BB16B6B), SPH_C64(0xD917AB5C852E8585),
	SPH_C64(0x3C677381BDCEBDBD), SPH_C64(0x8FBA34D25D695D5D),
	SPH_C64(0x9020508010401010), SPH_C64(0x07F503F3F4F7F4F4),
	SPH_C64(0xDD8BC016CB0BCBCB), SPH_C64(0xD37CC6ED3EF83E3E),
	SPH_C64(0x2D0A112805140505), SPH_C64(0x78CEE61F67816767),
	SPH_C64(0x97D55373E4B7E4E4), SPH_C64(0x024EBB25279C2727),
	SPH_C64(0x7382583241194141), SPH_C64(0xA70B9D2C8B168B8B),
	SPH_C64(0xF6530151A7A6A7A7), SPH_C64(0xB2FA94CF7DE97D7D),
	SPH_C64(0x4937FBDC956E9595), SPH_C64(0x56AD9F8ED847D8D8),
	SPH_C64(0x70EB308BFBCBFBFB), SPH_C64(0xCDC17123EE9FEEEE),
	SPH_C64(0xBBF891C77CED7C7C), SPH_C64(0x71CCE31766856666),
	SPH_C64(0x7BA78EA6DD53DDDD), SPH_C64(0xAF2E4BB8175C1717),
	SPH_C64(0x458E460247014747), SPH_C64(0x1A21DC849E429E9E),
	SPH_C64(0xD489C51ECA0FCACA), SPH_C64(0x585A99752DB42D2D),
	SPH_C64(0x2E637991BFC6BFBF), SPH_C64(0x3F0E1B38071C0707),
	SPH_C64(0xAC472301AD8EADAD), SPH_C64(0xB0B42FEA5A755A5A),
	SPH_C64(0xEF1BB56C83368383), SPH_C64(0xB666FF8533CC3333),
	SPH_C64(0x5CC6F23F63916363), SPH_C64(0x12040A1002080202),
	SPH_C64(0x93493839AA92AAAA), SPH_C64(0xDEE2A8AF71D97171),
	SPH_C64(0xC68DCF0EC807C8C8), SPH_C64(0xD1327DC819641919),
	SPH_C64(0x3B92707249394949), SPH_C64(0x5FAF9A86D943D9D9),
	SPH_C64(0x31F91DC3F2EFF2F2), SPH_C64(0xA8DB484BE3ABE3E3),
	SPH_C64(0xB9B62AE25B715B5B), SPH_C64(0xBC0D9234881A8888),
	SPH_C64(0x3E29C8A49A529A9A), SPH_C64(0x0B4CBE2D26982626),
	SPH_C64(0xBF64FA8D32C83232), SPH_C64(0x597D4AE9B0FAB0B0),
	SPH_C64(0xF2CF6A1BE983E9E9), SPH_C64(0x771E33780F3C0F0F),
	SPH_C64(0x33B7A6E6D573D5D5), SPH_C64(0xF41DBA74803A8080),
	SPH_C64(0x27617C99BEC2BEBE), SPH_C64(0xEB87DE26CD13CDCD),
	SPH_C64(0x8968E4BD34D03434), SPH_C64(0x3290757A483D4848),
	SPH_C64(0x54E324ABFFDBFFFF), SPH_C64(0x8DF48FF77AF57A7A),
	SPH_C64(0x643DEAF4907A9090), SPH_C64(0x9DBE3EC25F615F5F),
	SPH_C64(0x3D40A01D20802020), SPH_C64(0x0FD0D56768BD6868),
	SPH_C64(0xCA3472D01A681A1A), SPH_C64(0xB7412C19AE82AEAE),
	SPH_C64(0x7D755EC9B4EAB4B4), SPH_C64(0xCEA8199A544D5454),
	SPH_C64(0x7F3BE5EC93769393), SPH_C64(0x2F44AA0D22882222),
	SPH_C64(0x63C8E907648D6464), SPH_C64(0x2AFF12DBF1E3F1F1),
	SPH_C64(0xCCE6A2BF73D17373), SPH_C64(0x82245A9012481212),
	SPH_C64(0x7A805D3A401D4040), SPH_C64(0x4810284008200808),
	SPH_C64(0x959BE856C32BC3C3), SPH_C64(0xDFC57B33EC97ECEC),
	SPH_C64(0x4DAB9096DB4BDBDB), SPH_C64(0xC05F1F61A1BEA1A1),
	SPH_C64(0x9107831C8D0E8D8D), SPH_C64(0xC87AC9F53DF43D3D),
	SPH_C64(0x5B33F1CC97669797), SPH_C64(0x0000000000000000),
	SPH_C64(0xF983D436CF1BCFCF), SPH_C64(0x6E5687452BAC2B2B),
	SPH_C64(0xE1ECB39776C57676), SPH_C64(0xE619B06482328282),
	SPH_C64(0x28B1A9FED67FD6D6), SPH_C64(0xC33677D81B6C1B1B),
	SPH_C64(0x74775BC1B5EEB5B5), SPH_C64(0xBE432911AF86AFAF),
	SPH_C64(0x1DD4DF776AB56A6A), SPH_C64(0xEAA00DBA505D5050),
	SPH_C64(0x578A4C1245094545), SPH_C64(0x38FB18CBF3EBF3F3),
	SPH_C64(0xAD60F09D30C03030), SPH_C64(0xC4C3742BEF9BEFEF),
	SPH_C64(0xDA7EC3E53FFC3F3F), SPH_C64(0xC7AA1C9255495555),
	SPH_C64(0xDB591079A2B2A2A2), SPH_C64(0xE9C96503EA8FEAEA),
	SPH_C64(0x6ACAEC0F65896565), SPH_C64(0x036968B9BAD2BABA),
	SPH_C64(0x4A5E93652FBC2F2F), SPH_C64(0x8E9DE74EC027C0C0),
	SPH_C64(0x60A181BEDE5FDEDE), SPH_C64(0xFC386CE01C701C1C),
	SPH_C64(0x46E72EBBFDD3FDFD), SPH_C64(0x1F9A64524D294D4D),
	SPH_C64(0x7639E0E492729292), SPH_C64(0xFAEABC8F75C97575),
	SPH_C64(0x360C1E3006180606), SPH_C64(0xAE0998248A128A8A),
	SPH_C64(0x4B7940F9B2F2B2B2), SPH_C64(0x85D15963E6BFE6E6),
	SPH_C64(0x7E1C36700E380E0E), SPH_C64(0xE73E63F81F7C1F1F),
	SPH_C64(0x55C4F73762956262), SPH_C64(0x3AB5A3EED477D4D4),
	SPH_C64(0x814D3229A89AA8A8), SPH_C64(0x5231F4C496629696),
	SPH_C64(0x62EF3A9BF9C3F9F9), SPH_C64(0xA397F666C533C5C5),
	SPH_C64(0x104AB13525942525), SPH_C64(0xABB220F259795959),
	SPH_C64(0xD015AE54842A8484), SPH_C64(0xC5E4A7B772D57272),
	SPH_C64(0xEC72DDD539E43939), SPH_C64(0x1698615A4C2D4C4C),
	SPH_C64(0x94BC3BCA5E655E5E), SPH_C64(0x9FF085E778FD7878),
	SPH_C64(0xE570D8DD38E03838), SPH_C64(0x980586148C0A8C8C),
	SPH_C64(0x17BFB2C6D163D1D1), SPH_C64(0xE4570B41A5AEA5A5),
	SPH_C64(0xA1D94D43E2AFE2E2), SPH_C64(0x4EC2F82F61996161),
	SPH_C64(0x427B45F1B3F6B3B3), SPH_C64(0x3442A51521842121),
	SPH_C64(0x0825D6949C4A9C9C), SPH_C64(0xEE3C66F01E781E1E),
	SPH_C64(0x6186522243114343), SPH_C64(0xB193FC76C73BC7C7),
	SPH_C64(0x4FE52BB3FCD7FCFC), SPH_C64(0x2408142004100404),
	SPH_C64(0xE3A208B251595151), SPH_C64(0x252FC7BC995E9999),
	SPH_C64(0x22DAC44F6DA96D6D), SPH_C64(0x651A39680D340D0D),
	SPH_C64(0x79E93583FACFFAFA), SPH_C64(0x69A384B6DF5BDFDF),
	SPH_C64(0xA9FC9BD77EE57E7E), SPH_C64(0x1948B43D24902424),
	SPH_C64(0xFE76D7C53BEC3B3B), SPH_C64(0x9A4B3D31AB96ABAB),
	SPH_C64(0xF081D13ECE1FCECE), SPH_C64(0x9922558811441111),
	SPH_C64(0x8303890C8F068F8F), SPH_C64(0x049C6B4A4E254E4E),
	SPH_C64(0x667351D1B7E6B7B7), SPH_C64(0xE0CB600BEB8BEBEB),
	SPH_C64(0xC178CCFD3CF03C3C), SPH_C64(0xFD1FBF7C813E8181),
	SPH_C64(0x4035FED4946A9494), SPH_C64(0x1CF30CEBF7FBF7F7),
	SPH_C64(0x186F67A1B9DEB9B9), SPH_C64(0x8B265F98134C1313),
	SPH_C64(0x51589C7D2CB02C2C), SPH_C64(0x05BBB8D6D36BD3D3),
	SPH_C64(0x8CD35C6BE7BBE7E7), SPH_C64(0x39DCCB576EA56E6E),
	SPH_C64(0xAA95F36EC437C4C4), SPH_C64(0x1B060F18030C0303),
	SPH_C64(0xDCAC138A56455656), SPH_C64(0x5E88491A440D4444),
	SPH_C64(0xA0FE9EDF7FE17F7F), SPH_C64(0x884F3721A99EA9A9),
	SPH_C64(0x6754824D2AA82A2A), SPH_C64(0x0A6B6DB1BBD6BBBB),
	SPH_C64(0x879FE246C123C1C1), SPH_C64(0xF1A602A253515353),
	SPH_C64(0x72A58BAEDC57DCDC), SPH_C64(0x531627580B2C0B0B),
	SPH_C64(0x0127D39C9D4E9D9D), SPH_C64(0x2BD8C1476CAD6C6C),
	SPH_C64(0xA462F59531C43131), SPH_C64(0xF3E8B98774CD7474),
	SPH_C64(0x15F109E3F6FFF6F6), SPH_C64(0x4C8C430A46054646),
	SPH_C64(0xA5452609AC8AACAC), SPH_C64(0xB50F973C891E8989),
	SPH_C64(0xB42844A014501414), SPH_C64(0xBADF425BE1A3E1E1),
	SPH_C64(0xA62C4EB016581616), SPH_C64(0xF774D2CD3AE83A3A),
	SPH_C64(0x06D2D06F69B96969), SPH_C64(0x41122D4809240909),
	SPH_C64(0xD7E0ADA770DD7070), SPH_C64(0x6F7154D9B6E2B6B6),
	SPH_C64(0x1EBDB7CED067D0D0), SPH_C64(0xD6C77E3BED93EDED),
	SPH_C64(0xE285DB2ECC17CCCC), SPH_C64(0x6884572A42154242),
	SPH_C64(0x2C2DC2B4985A9898), SPH_C64(0xED550E49A4AAA4A4),
	SPH_C64(0x7550885D28A02828), SPH_C64(0x86B831DA5C6D5C5C),
	SPH_C64(0x6BED3F93F8C7F8F8), SPH_C64(0xC211A44486228686)
};

#if !SPH_SMALL_FOOTPRINT_WHIRLPOOL

static const sph_u64 plain_T1[256] = {
	SPH_C64(0x3078C018601818D8), SPH_C64(0x46AF05238C232326),
	SPH_C64(0x91F97EC63FC6C6B8), SPH_C64(0xCD6F13E887E8E8FB),
	SPH_C64(0x13A14C87268787CB), SPH_C64(0x6D62A9B8DAB8B811),
	SPH_C64(0x0205080104010109), SPH_C64(0x9E6E424F214F4F0D),
	SPH_C64(0x6CEEAD36D836369B), SPH_C64(0x510459A6A2A6A6FF),
	SPH_C64(0xB9BDDED26FD2D20C), SPH_C64(0xF706FBF5F3F5F50E),
	SPH_C64(0xF280EF79F9797996), SPH_C64(0xDECE5F6FA16F6F30),
	SPH_C64(0x3FEFFC917E91916D), SPH_C64(0xA407AA52555252F8),
	SPH_C64(0xC0FD27609D606047), SPH_C64(0x657689BCCABCBC35),
	SPH_C64(0x2BCDAC9B569B9B37), SPH_C64(0x018C048E028E8E8A),
	SPH_C64(0x5B1571A3B6A3A3D2), SPH_C64(0x183C600C300C0C6C),
	SPH_C64(0xF68AFF7BF17B7B84), SPH_C64(0x6AE1B535D4353580),
	SPH_C64(0x3A69E81D741D1DF5), SPH_C64(0xDD4753E0A7E0E0B3),
	SPH_C64(0xB3ACF6D77BD7D721), SPH_C64(0x99ED5EC22FC2C29C),
	SPH_C64(0x5C966D2EB82E2E43), SPH_C64(0x967A624B314B4B29),
	SPH_C64(0xE121A3FEDFFEFE5D), SPH_C64(0xAE168257415757D5),
	SPH_C64(0x2A41A815541515BD), SPH_C64(0xEEB69F77C17777E8),
	SPH_C64(0x6EEBA537DC373792), SPH_C64(0xD7567BE5B3E5E59E),
	SPH_C64(0x23D98C9F469F9F13), SPH_C64(0xFD17D3F0E7F0F023),
	SPH_C64(0x947F6A4A354A4A20), SPH_C64(0xA9959EDA4FDADA44),
	SPH_C64(0xB025FA587D5858A2), SPH_C64(0x8FCA06C903C9C9CF),
	SPH_C64(0x528D5529A429297C), SPH_C64(0x1422500A280A0A5A),
	SPH_C64(0x7F4FE1B1FEB1B150), SPH_C64(0x5D1A69A0BAA0A0C9),
	SPH_C64(0xD6DA7F6BB16B6B14), SPH_C64(0x17AB5C852E8585D9),
	SPH_C64(0x677381BDCEBDBD3C), SPH_C64(0xBA34D25D695D5D8F),
	SPH_C64(0x2050801040101090), SPH_C64(0xF503F3F4F7F4F407),
	SPH_C64(0x8BC016CB0BCBCBDD), SPH_C64(0x7CC6ED3EF83E3ED3),
	SPH_C64(0x0A1128051405052D), SPH_C64(0xCEE61F6781676778),
	SPH_C64(0xD55373E4B7E4E497), SPH_C64(0x4EBB25279C272702),
	SPH_C64(0x8258324119414173), SPH_C64(0x0B9D2C8B168B8BA7),
	SPH_C64(0x530151A7A6A7A7F6), SPH_C64(0xFA94CF7DE97D7DB2),
	SPH_C64(0x37FBDC956E959549), SPH_C64(0xAD9F8ED847D8D856),
	SPH_C64(0xEB308BFBCBFBFB70), SPH_C64(0xC17123EE9FEEEECD),
	SPH_C64(0xF891C77CED7C7CBB), SPH_C64(0xCCE3176685666671),
	SPH_C64(0xA78EA6DD53DDDD7B), SPH_C64(0x2E4BB8175C1717AF),
	SPH_C64(0x8E46024701474745), SPH_C64(0x21DC849E429E9E1A),
	SPH_C64(0x89C51ECA0FCACAD4), SPH_C64(0x5A99752DB42D2D58),
	SPH_C64(0x637991BFC6BFBF2E), SPH_C64(0x0E1B38071C07073F),
	SPH_C64(0x472301AD8EADADAC), SPH_C64(0xB42FEA5A755A5AB0),
	SPH_C64(0x1BB56C83368383EF), SPH_C64(0x66FF8533CC3333B6),
	SPH_C64(0xC6F23F639163635C), SPH_C64(0x040A100208020212),
	SPH_C64(0x493839AA92AAAA93), SPH_C64(0xE2A8AF71D97171DE),
	SPH_C64(0x8DCF0EC807C8C8C6), SPH_C64(0x327DC819641919D1),
	SPH_C64(0x927072493949493B), SPH_C64(0xAF9A86D943D9D95F),
	SPH_C64(0xF91DC3F2EFF2F231), SPH_C64(0xDB484BE3ABE3E3A8),
	SPH_C64(0xB62AE25B715B5BB9), SPH_C64(0x0D9234881A8888BC),
	SPH_C64(0x29C8A49A529A9A3E), SPH_C64(0x4CBE2D269826260B),
	SPH_C64(0x64FA8D32C83232BF), SPH_C64(0x7D4AE9B0FAB0B059),
	SPH_C64(0xCF6A1BE983E9E9F2), SPH_C64(0x1E33780F3C0F0F77),
	SPH_C64(0xB7A6E6D573D5D533), SPH_C64(0x1DBA74803A8080F4),
	SPH_C64(0x617C99BEC2BEBE27), SPH_C64(0x87DE26CD13CDCDEB),
	SPH_C64(0x68E4BD34D0343489), SPH_C64(0x90757A483D484832),
	SPH_C64(0xE324ABFFDBFFFF54), SPH_C64(0xF48FF77AF57A7A8D),
	SPH_C64(0x3DEAF4907A909064), SPH_C64(0xBE3EC25F615F5F9D),
	SPH_C64(0x40A01D208020203D), SPH_C64(0xD0D56768BD68680F),
	SPH_C64(0x3472D01A681A1ACA), SPH_C64(0x412C19AE82AEAEB7),
	SPH_C64(0x755EC9B4EAB4B47D), SPH_C64(0xA8199A544D5454CE),
	SPH_C64(0x3BE5EC937693937F), SPH_C64(0x44AA0D228822222F),
	SPH_C64(0xC8E907648D646463), SPH_C64(0xFF12DBF1E3F1F12A),
	SPH_C64(0xE6A2BF73D17373CC), SPH_C64(0x245A901248121282),
	SPH_C64(0x805D3A401D40407A), SPH_C64(0x1028400820080848),
	SPH_C64(0x9BE856C32BC3C395), SPH_C64(0xC57B33EC97ECECDF),
	SPH_C64(0xAB9096DB4BDBDB4D), SPH_C64(0x5F1F61A1BEA1A1C0),
	SPH_C64(0x07831C8D0E8D8D91), SPH_C64(0x7AC9F53DF43D3DC8),
	SPH_C64(0x33F1CC976697975B), SPH_C64(0x0000000000000000),
	SPH_C64(0x83D436CF1BCFCFF9), SPH_C64(0x5687452BAC2B2B6E),
	SPH_C64(0xECB39776C57676E1), SPH_C64(0x19B06482328282E6),
	SPH_C64(0xB1A9FED67FD6D628), SPH_C64(0x3677D81B6C1B1BC3),
	SPH_C64(0x775BC1B5EEB5B574), SPH_C64(0x432911AF86AFAFBE),
	SPH_C64(0xD4DF776AB56A6A1D), SPH_C64(0xA00DBA505D5050EA),
	SPH_C64(0x8A4C124509454557), SPH_C64(0xFB18CBF3EBF3F338),
	SPH_C64(0x60F09D30C03030AD), SPH_C64(0xC3742BEF9BEFEFC4),
	SPH_C64(0x7EC3E53FFC3F3FDA), SPH_C64(0xAA1C9255495555C7),
	SPH_C64(0x591079A2B2A2A2DB), SPH_C64(0xC96503EA8FEAEAE9),
	SPH_C64(0xCAEC0F658965656A), SPH_C64(0x6968B9BAD2BABA03),
	SPH_C64(0x5E93652FBC2F2F4A), SPH_C64(0x9DE74EC027C0C08E),
	SPH_C64(0xA181BEDE5FDEDE60), SPH_C64(0x386CE01C701C1CFC),
	SPH_C64(0xE72EBBFDD3FDFD46), SPH_C64(0x9A64524D294D4D1F),
	SPH_C64(0x39E0E49272929276), SPH_C64(0xEABC8F75C97575FA),
	SPH_C64(0x0C1E300618060636), SPH_C64(0x0998248A128A8AAE),
	SPH_C64(0x7940F9B2F2B2B24B), SPH_C64(0xD15963E6BFE6E685),
	SPH_C64(0x1C36700E380E0E7E), SPH_C64(0x3E63F81F7C1F1FE7),
	SPH_C64(0xC4F7376295626255), SPH_C64(0xB5A3EED477D4D43A),
	SPH_C64(0x4D3229A89AA8A881), SPH_C64(0x31F4C49662969652),
	SPH_C64(0xEF3A9BF9C3F9F962), SPH_C64(0x97F666C533C5C5A3),
	SPH_C64(0x4AB1352594252510), SPH_C64(0xB220F259795959AB),
	SPH_C64(0x15AE54842A8484D0), SPH_C64(0xE4A7B772D57272C5),
	SPH_C64(0x72DDD539E43939EC), SPH_C64(0x98615A4C2D4C4C16),
	SPH_C64(0xBC3BCA5E655E5E94), SPH_C64(0xF085E778FD78789F),
	SPH_C64(0x70D8DD38E03838E5), SPH_C64(0x0586148C0A8C8C98),
	SPH_C64(0xBFB2C6D163D1D117), SPH_C64(0x570B41A5AEA5A5E4),
	SPH_C64(0xD94D43E2AFE2E2A1), SPH_C64(0xC2F82F619961614E),
	SPH_C64(0x7B45F1B3F6B3B342), SPH_C64(0x42A5152184212134),
	SPH_C64(0x25D6949C4A9C9C08), SPH_C64(0x3C66F01E781E1EEE),
	SPH_C64(0x8652224311434361), SPH_C64(0x93FC76C73BC7C7B1),
	SPH_C64(0xE52BB3FCD7FCFC4F), SPH_C64(0x0814200410040424),
	SPH_C64(0xA208B251595151E3), SPH_C64(0x2FC7BC995E999925),
	SPH_C64(0xDAC44F6DA96D6D22), SPH_C64(0x1A39680D340D0D65),
	SPH_C64(0xE93583FACFFAFA79), SPH_C64(0xA384B6DF5BDFDF69),
	SPH_C64(0xFC9BD77EE57E7EA9), SPH_C64(0x48B43D2490242419),
	SPH_C64(0x76D7C53BEC3B3BFE), SPH_C64(0x4B3D31AB96ABAB9A),
	SPH_C64(0x81D13ECE1FCECEF0), SPH_C64(0x2255881144111199),
	SPH_C64(0x03890C8F068F8F83), SPH_C64(0x9C6B4A4E254E4E04),
	SPH_C64(0x7351D1B7E6B7B766), SPH_C64(0xCB600BEB8BEBEBE0),
	SPH_C64(0x78CCFD3CF03C3CC1), SPH_C64(0x1FBF7C813E8181FD),
	SPH_C64(0x35FED4946A949440), SPH_C64(0xF30CEBF7FBF7F71C),
	SPH_C64(0x6F67A1B9DEB9B918), SPH_C64(0x265F98134C13138B),
	SPH_C64(0x589C7D2CB02C2C51), SPH_C64(0xBBB8D6D36BD3D305),
	SPH_C64(0xD35C6BE7BBE7E78C), SPH_C64(0xDCCB576EA56E6E39),
	SPH_C64(0x95F36EC437C4C4AA), SPH_C64(0x060F18030C03031B),
	SPH_C64(0xAC138A56455656DC), SPH_C64(0x88491A440D44445E),
	SPH_C64(0xFE9EDF7FE17F7FA0), SPH_C64(0x4F3721A99EA9A988),
	SPH_C64(0x54824D2AA82A2A67), SPH_C64(0x6B6DB1BBD6BBBB0A),
	SPH_C64(0x9FE246C123C1C187), SPH_C64(0xA602A253515353F1),
	SPH_C64(0xA58BAEDC57DCDC72), SPH_C64(0x1627580B2C0B0B53),
	SPH_C64(0x27D39C9D4E9D9D01), SPH_C64(0xD8C1476CAD6C6C2B),
	SPH_C64(0x62F59531C43131A4), SPH_C64(0xE8B98774CD7474F3),
	SPH_C64(0xF109E3F6FFF6F615), SPH_C64(0x8C430A460546464C),
	SPH_C64(0x452609AC8AACACA5), SPH_C64(0x0F973C891E8989B5),
	SPH_C64(0x2844A014501414B4), SPH_C64(0xDF425BE1A3E1E1BA),
	SPH_C64(0x2C4EB016581616A6), SPH_C64(0x74D2CD3AE83A3AF7),
	SPH_C64(0xD2D06F69B9696906), SPH_C64(0x122D480924090941),
	SPH_C64(0xE0ADA770DD7070D7), SPH_C64(0x7154D9B6E2B6B66F),
	SPH_C64(0xBDB7CED067D0D01E), SPH_C64(0xC77E3BED93EDEDD6),
	SPH_C64(0x85DB2ECC17CCCCE2), SPH_C64(0x84572A4215424268),
	SPH_C64(0x2DC2B4985A98982C), SPH_C64(0x550E49A4AAA4A4ED),
	SPH_C64(0x50885D28A0282875), SPH_C64(0xB831DA5C6D5C5C86),
	SPH_C64(0xED3F93F8C7F8F86B), SPH_C64(0x11A44486228686C2)
};

static const sph_u64 plain_T2[256] = {
	SPH_C64(0x78C018601818D830), SPH_C64(0xAF05238C23232646),
	SPH_C64(0xF97EC63FC6C6B891), SPH_C64(0x6F13E887E8E8FBCD),
	SPH_C64(0xA14C87268787CB13), SPH_C64(0x62A9B8DAB8B8116D),
	SPH_C64(0x0508010401010902), SPH_C64(0x6E424F214F4F0D9E),
	SPH_C64(0xEEAD36D836369B6C), SPH_C64(0x0459A6A2A6A6FF51),
	SPH_C64(0xBDDED26FD2D20CB9), SPH_C64(0x06FBF5F3F5F50EF7),
	SPH_C64(0x80EF79F9797996F2), SPH_C64(0xCE5F6FA16F6F30DE),
	SPH_C64(0xEFFC917E91916D3F), SPH_C64(0x07AA52555252F8A4),
	SPH_C64(0xFD27609D606047C0), SPH_C64(0x7689BCCABCBC3565),
	SPH_C64(0xCDAC9B569B9B372B), SPH_C64(0x8C048E028E8E8A01),
	SPH_C64(0x1571A3B6A3A3D25B), SPH_C64(0x3C600C300C0C6C18),
	SPH_C64(0x8AFF7BF17B7B84F6), SPH_C64(0xE1B535D43535806A),
	SPH_C64(0x69E81D741D1DF53A), SPH_C64(0x4753E0A7E0E0B3DD),
	SPH_C64(0xACF6D77BD7D721B3), SPH_C64(0xED5EC22FC2C29C99),
	SPH_C64(0x966D2EB82E2E435C), SPH_C64(0x7A624B314B4B2996),
	SPH_C64(0x21A3FEDFFEFE5DE1), SPH_C64(0x168257415757D5AE),
	SPH_C64(0x41A815541515BD2A), SPH_C64(0xB69F77C17777E8EE),
	SPH_C64(0xEBA537DC3737926E), SPH_C64(0x567BE5B3E5E59ED7),
	SPH_C64(0xD98C9F469F9F1323), SPH_C64(0x17D3F0E7F0F023FD),
	SPH_C64(0x7F6A4A354A4A2094), SPH_C64(0x959EDA4FDADA44A9),
	SPH_C64(0x25FA587D5858A2B0), SPH_C64(0xCA06C903C9C9CF8F),
	SPH_C64(0x8D5529A429297C52), SPH_C64(0x22500A280A0A5A14),
	SPH_C64(0x4FE1B1FEB1B1507F), SPH_C64(0x1A69A0BAA0A0C95D),
	SPH_C64(0xDA7F6BB16B6B14D6), SPH_C64(0xAB5C852E8585D917),
	SPH_C64(0x7381BDCEBDBD3C67), SPH_C64(0x34D25D695D5D8FBA),
	SPH_C64(0x5080104010109020), SPH_C64(0x03F3F4F7F4F407F5),
	SPH_C64(0xC016CB0BCBCBDD8B), SPH_C64(0xC6ED3EF83E3ED37C),
	SPH_C64(0x1128051405052D0A), SPH_C64(0xE61F6781676778CE),
	SPH_C64(0x5373E4B7E4E497D5), SPH_C64(0xBB25279C2727024E),
	SPH_C64(0x5832411941417382), SPH_C64(0x9D2C8B168B8BA70B),
	SPH_C64(0x0151A7A6A7A7F653), SPH_C64(0x94CF7DE97D7DB2FA),
	SPH_C64(0xFBDC956E95954937), SPH_C64(0x9F8ED847D8D856AD),
	SPH_C64(0x308BFBCBFBFB70EB), SPH_C64(0x7123EE9FEEEECDC1),
	SPH_C64(0x91C77CED7C7CBBF8), SPH_C64(0xE3176685666671CC),
	SPH_C64(0x8EA6DD53DDDD7BA7), SPH_C64(0x4BB8175C1717AF2E),
	SPH_C64(0x460247014747458E), SPH_C64(0xDC849E429E9E1A21),
	SPH_C64(0xC51ECA0FCACAD489), SPH_C64(0x99752DB42D2D585A),
	SPH_C64(0x7991BFC6BFBF2E63), SPH_C64(0x1B38071C07073F0E),
	SPH_C64(0x2301AD8EADADAC47), SPH_C64(0x2FEA5A755A5AB0B4),
	SPH_C64(0xB56C83368383EF1B), SPH_C64(0xFF8533CC3333B666),
	SPH_C64(0xF23F639163635CC6), SPH_C64(0x0A10020802021204),
	SPH_C64(0x3839AA92AAAA9349), SPH_C64(0xA8AF71D97171DEE2),
	SPH_C64(0xCF0EC807C8C8C68D), SPH_C64(0x7DC819641919D132),
	SPH_C64(0x7072493949493B92), SPH_C64(0x9A86D943D9D95FAF),
	SPH_C64(0x1DC3F2EFF2F231F9), SPH_C64(0x484BE3ABE3E3A8DB),
	SPH_C64(0x2AE25B715B5BB9B6), SPH_C64(0x9234881A8888BC0D),
	SPH_C64(0xC8A49A529A9A3E29), SPH_C64(0xBE2D269826260B4C),
	SPH_C64(0xFA8D32C83232BF64), SPH_C64(0x4AE9B0FAB0B0597D),
	SPH_C64(0x6A1BE983E9E9F2CF), SPH_C64(0x33780F3C0F0F771E),
	SPH_C64(0xA6E6D573D5D533B7), SPH_C64(0xBA74803A8080F41D),
	SPH_C64(0x7C99BEC2BEBE2761), SPH_C64(0xDE26CD13CDCDEB87),
	SPH_C64(0xE4BD34D034348968), SPH_C64(0x757A483D48483290),
	SPH_C64(0x24ABFFDBFFFF54E3), SPH_C64(0x8FF77AF57A7A8DF4),
	SPH_C64(0xEAF4907A9090643D), SPH_C64(0x3EC25F615F5F9DBE),
	SPH_C64(0xA01D208020203D40), SPH_C64(0xD56768BD68680FD0),
	SPH_C64(0x72D01A681A1ACA34), SPH_C64(0x2C19AE82AEAEB741),
	SPH_C64(0x5EC9B4EAB4B47D75), SPH_C64(0x199A544D5454CEA8),
	SPH_C64(0xE5EC937693937F3B), SPH_C64(0xAA0D228822222F44),
	SPH_C64(0xE907648D646463C8), SPH_C64(0x12DBF1E3F1F12AFF),
	SPH_C64(0xA2BF73D17373CCE6), SPH_C64(0x5A90124812128224),
	SPH_C64(0x5D3A401D40407A80), SPH_C64(0x2840082008084810),
	SPH_C64(0xE856C32BC3C3959B), SPH_C64(0x7B33EC97ECECDFC5),
	SPH_C64(0x9096DB4BDBDB4DAB), SPH_C64(0x1F61A1BEA1A1C05F),
	SPH_C64(0x831C8D0E8D8D9107), SPH_C64(0xC9F53DF43D3DC87A),
	SPH_C64(0xF1CC976697975B33), SPH_C64(0x0000000000000000),
	SPH_C64(0xD436CF1BCFCFF983), SPH_C64(0x87452BAC2B2B6E56),
	SPH_C64(0xB39776C57676E1EC), SPH_C64(0xB06482328282E619),
	SPH_C64(0xA9FED67FD6D628B1), SPH_C64(0x77D81B6C1B1BC336),
	SPH_C64(0x5BC1B5EEB5B57477), SPH_C64(0x2911AF86AFAFBE43),
	SPH_C64(0xDF776AB56A6A1DD4), SPH_C64(0x0DBA505D5050EAA0),
	SPH_C64(0x4C1245094545578A), SPH_C64(0x18CBF3EBF3F338FB),
	SPH_C64(0xF09D30C03030AD60), SPH_C64(0x742BEF9BEFEFC4C3),
	SPH_C64(0xC3E53FFC3F3FDA7E), SPH_C64(0x1C9255495555C7AA),
	SPH_C64(0x1079A2B2A2A2DB59), SPH_C64(0x6503EA8FEAEAE9C9),
	SPH_C64(0xEC0F658965656ACA), SPH_C64(0x68B9BAD2BABA0369),
	SPH_C64(0x93652FBC2F2F4A5E), SPH_C64(0xE74EC027C0C08E9D),
	SPH_C64(0x81BEDE5FDEDE60A1), SPH_C64(0x6CE01C701C1CFC38),
	SPH_C64(0x2EBBFDD3FDFD46E7), SPH_C64(0x64524D294D4D1F9A),
	SPH_C64(0xE0E4927292927639), SPH_C64(0xBC8F75C97575FAEA),
	SPH_C64(0x1E3006180606360C), SPH_C64(0x98248A128A8AAE09),
	SPH_C64(0x40F9B2F2B2B24B79), SPH_C64(0x5963E6BFE6E685D1),
	SPH_C64(0x36700E380E0E7E1C), SPH_C64(0x63F81F7C1F1FE73E),
	SPH_C64(0xF7376295626255C4), SPH_C64(0xA3EED477D4D43AB5),
	SPH_C64(0x3229A89AA8A8814D), SPH_C64(0xF4C4966296965231),
	SPH_C64(0x3A9BF9C3F9F962EF), SPH_C64(0xF666C533C5C5A397),
	SPH_C64(0xB13525942525104A), SPH_C64(0x20F259795959ABB2),
	SPH_C64(0xAE54842A8484D015), SPH_C64(0xA7B772D57272C5E4),
	SPH_C64(0xDDD539E43939EC72), SPH_C64(0x615A4C2D4C4C1698),
	SPH_C64(0x3BCA5E655E5E94BC), SPH_C64(0x85E778FD78789FF0),
	SPH_C64(0xD8DD38E03838E570), SPH_C64(0x86148C0A8C8C9805),
	SPH_C64(0xB2C6D163D1D117BF), SPH_C64(0x0B41A5AEA5A5E457),
	SPH_C64(0x4D43E2AFE2E2A1D9), SPH_C64(0xF82F619961614EC2),
	SPH_C64(0x45F1B3F6B3B3427B), SPH_C64(0xA515218421213442),
	SPH_C64(0xD6949C4A9C9C0825), SPH_C64(0x66F01E781E1EEE3C),
	SPH_C64(0x5222431143436186), SPH_C64(0xFC76C73BC7C7B193),
	SPH_C64(0x2BB3FCD7FCFC4FE5), SPH_C64(0x1420041004042408),
	SPH_C64(0x08B251595151E3A2), SPH_C64(0xC7BC995E9999252F),
	SPH_C64(0xC44F6DA96D6D22DA), SPH_C64(0x39680D340D0D651A),
	SPH_C64(0x3583FACFFAFA79E9), SPH_C64(0x84B6DF5BDFDF69A3),
	SPH_C64(0x9BD77EE57E7EA9FC), SPH_C64(0xB43D249024241948),
	SPH_C64(0xD7C53BEC3B3BFE76), SPH_C64(0x3D31AB96ABAB9A4B),
	SPH_C64(0xD13ECE1FCECEF081), SPH_C64(0x5588114411119922),
	SPH_C64(0x890C8F068F8F8303), SPH_C64(0x6B4A4E254E4E049C),
	SPH_C64(0x51D1B7E6B7B76673), SPH_C64(0x600BEB8BEBEBE0CB),
	SPH_C64(0xCCFD3CF03C3CC178), SPH_C64(0xBF7C813E8181FD1F),
	SPH_C64(0xFED4946A94944035), SPH_C64(0x0CEBF7FBF7F71CF3),
	SPH_C64(0x67A1B9DEB9B9186F), SPH_C64(0x5F98134C13138B26),
	SPH_C64(0x9C7D2CB02C2C5158), SPH_C64(0xB8D6D36BD3D305BB),
	SPH_C64(0x5C6BE7BBE7E78CD3), SPH_C64(0xCB576EA56E6E39DC),
	SPH_C64(0xF36EC437C4C4AA95), SPH_C64(0x0F18030C03031B06),
	SPH_C64(0x138A56455656DCAC), SPH_C64(0x491A440D44445E88),
	SPH_C64(0x9EDF7FE17F7FA0FE), SPH_C64(0x3721A99EA9A9884F),
	SPH_C64(0x824D2AA82A2A6754), SPH_C64(0x6DB1BBD6BBBB0A6B),
	SPH_C64(0xE246C123C1C1879F), SPH_C64(0x02A253515353F1A6),
	SPH_C64(0x8BAEDC57DCDC72A5), SPH_C64(0x27580B2C0B0B5316),
	SPH_C64(0xD39C9D4E9D9D0127), SPH_C64(0xC1476CAD6C6C2BD8),
	SPH_C64(0xF59531C43131A462), SPH_C64(0xB98774CD7474F3E8),
	SPH_C64(0x09E3F6FFF6F615F1), SPH_C64(0x430A460546464C8C),
	SPH_C64(0x2609AC8AACACA545), SPH_C64(0x973C891E8989B50F),
	SPH_C64(0x44A014501414B428), SPH_C64(0x425BE1A3E1E1BADF),
	SPH_C64(0x4EB016581616A62C), SPH_C64(0xD2CD3AE83A3AF774),
	SPH_C64(0xD06F69B9696906D2), SPH_C64(0x2D48092409094112),
	SPH_C64(0xADA770DD7070D7E0), SPH_C64(0x54D9B6E2B6B66F71),
	SPH_C64(0xB7CED067D0D01EBD), SPH_C64(0x7E3BED93EDEDD6C7),
	SPH_C64(0xDB2ECC17CCCCE285), SPH_C64(0x572A421542426884),
	SPH_C64(0xC2B4985A98982C2D), SPH_C64(0x0E49A4AAA4A4ED55),
	SPH_C64(0x885D28A028287550), SPH_C64(0x31DA5C6D5C5C86B8),
	SPH_C64(0x3F93F8C7F8F86BED), SPH_C64(0xA44486228686C211)
};

static const sph_u64 plain_T3[256] = {
	SPH_C64(0xC018601818D83078), SPH_C64(0x05238C23232646AF),
	SPH_C64(0x7EC63FC6C6B891F9), SPH_C64(0x13E887E8E8FBCD6F),
	SPH_C64(0x4C87268787CB13A1), SPH_C64(0xA9B8DAB8B8116D62),
	SPH_C64(0x0801040101090205), SPH_C64(0x424F214F4F0D9E6E),
	SPH_C64(0xAD36D836369B6CEE), SPH_C64(0x59A6A2A6A6FF5104),
	SPH_C64(0xDED26FD2D20CB9BD), SPH_C64(0xFBF5F3F5F50EF706),
	SPH_C64(0xEF79F9797996F280), SPH_C64(0x5F6FA16F6F30DECE),
	SPH_C64(0xFC917E91916D3FEF), SPH_C64(0xAA52555252F8A407),
	SPH_C64(0x27609D606047C0FD), SPH_C64(0x89BCCABCBC356576),
	SPH_C64(0xAC9B569B9B372BCD), SPH_C64(0x048E028E8E8A018C),
	SPH_C64(0x71A3B6A3A3D25B15), SPH_C64(0x600C300C0C6C183C),
	SPH_C64(0xFF7BF17B7B84F68A), SPH_C64(0xB535D43535806AE1),
	SPH_C64(0xE81D741D1DF53A69), SPH_C64(0x53E0A7E0E0B3DD47),
	SPH_C64(0xF6D77BD7D721B3AC), SPH_C64(0x5EC22FC2C29C99ED),
	SPH_C64(0x6D2EB82E2E435C96), SPH_C64(0x624B314B4B29967A),
	SPH_C64(0xA3FEDFFEFE5DE121), SPH_C64(0x8257415757D5AE16),
	SPH_C64(0xA815541515BD2A41), SPH_C64(0x9F77C17777E8EEB6),
	SPH_C64(0xA537DC3737926EEB), SPH_C64(0x7BE5B3E5E59ED756),
	SPH_C64(0x8C9F469F9F1323D9), SPH_C64(0xD3F0E7F0F023FD17),
	SPH_C64(0x6A4A354A4A20947F), SPH_C64(0x9EDA4FDADA44A995),
	SPH_C64(0xFA587D5858A2B025), SPH_C64(0x06C903C9C9CF8FCA),
	SPH_C64(0x5529A429297C528D), SPH_C64(0x500A280A0A5A1422),
	SPH_C64(0xE1B1FEB1B1507F4F), SPH_C64(0x69A0BAA0A0C95D1A),
	SPH_C64(0x7F6BB16B6B14D6DA), SPH_C64(0x5C852E8585D917AB),
	SPH_C64(0x81BDCEBDBD3C6773), SPH_C64(0xD25D695D5D8FBA34),
	SPH_C64(0x8010401010902050), SPH_C64(0xF3F4F7F4F407F503),
	SPH_C64(0x16CB0BCBCBDD8BC0), SPH_C64(0xED3EF83E3ED37CC6),
	SPH_C64(0x28051405052D0A11), SPH_C64(0x1F6781676778CEE6),
	SPH_C64(0x73E4B7E4E497D553), SPH_C64(0x25279C2727024EBB),
	SPH_C64(0x3241194141738258), SPH_C64(0x2C8B168B8BA70B9D),
	SPH_C64(0x51A7A6A7A7F65301), SPH_C64(0xCF7DE97D7DB2FA94),
	SPH_C64(0xDC956E95954937FB), SPH_C64(0x8ED847D8D856AD9F),
	SPH_C64(0x8BFBCBFBFB70EB30), SPH_C64(0x23EE9FEEEECDC171),
	SPH_C64(0xC77CED7C7CBBF891), SPH_C64(0x176685666671CCE3),
	SPH_C64(0xA6DD53DDDD7BA78E), SPH_C64(0xB8175C1717AF2E4B),
	SPH_C64(0x0247014747458E46), SPH_C64(0x849E429E9E1A21DC),
	SPH_C64(0x1ECA0FCACAD489C5), SPH_C64(0x752DB42D2D585A99),
	SPH_C64(0x91BFC6BFBF2E6379), SPH_C64(0x38071C07073F0E1B),
	SPH_C64(0x01AD8EADADAC4723), SPH_C64(0xEA5A755A5AB0B42F),
	SPH_C64(0x6C83368383EF1BB5), SPH_C64(0x8533CC3333B666FF),
	SPH_C64(0x3F639163635CC6F2), SPH_C64(0x100208020212040A),
	SPH_C64(0x39AA92AAAA934938), SPH_C64(0xAF71D97171DEE2A8),
	SPH_C64(0x0EC807C8C8C68DCF), SPH_C64(0xC819641919D1327D),
	SPH_C64(0x72493949493B9270), SPH_C64(0x86D943D9D95FAF9A),
	SPH_C64(0xC3F2EFF2F231F91D), SPH_C64(0x4BE3ABE3E3A8DB48),
	SPH_C64(0xE25B715B5BB9B62A), SPH_C64(0x34881A8888BC0D92),
	SPH_C64(0xA49A529A9A3E29C8), SPH_C64(0x2D269826260B4CBE),
	SPH_C64(0x8D32C83232BF64FA), SPH_C64(0xE9B0FAB0B0597D4A),
	SPH_C64(0x1BE983E9E9F2CF6A), SPH_C64(0x780F3C0F0F771E33),
	SPH_C64(0xE6D573D5D533B7A6), SPH_C64(0x74803A8080F41DBA),
	SPH_C64(0x99BEC2BEBE27617C), SPH_C64(0x26CD13CDCDEB87DE),
	SPH_C64(0xBD34D034348968E4), SPH_C64(0x7A483D4848329075),
	SPH_C64(0xABFFDBFFFF54E324), SPH_C64(0xF77AF57A7A8DF48F),
	SPH_C64(0xF4907A9090643DEA), SPH_C64(0xC25F615F5F9DBE3E),
	SPH_C64(0x1D208020203D40A0), SPH_C64(0x6768BD68680FD0D5),
	SPH_C64(0xD01A681A1ACA3472), SPH_C64(0x19AE82AEAEB7412C),
	SPH_C64(0xC9B4EAB4B47D755E), SPH_C64(0x9A544D5454CEA819),
	SPH_C64(0xEC937693937F3BE5), SPH_C64(0x0D228822222F44AA),
	SPH_C64(0x07648D646463C8E9), SPH_C64(0xDBF1E3F1F12AFF12),
	SPH_C64(0xBF73D17373CCE6A2), SPH_C64(0x901248121282245A),
	SPH_C64(0x3A401D40407A805D), SPH_C64(0x4008200808481028),
	SPH_C64(0x56C32BC3C3959BE8), SPH_C64(0x33EC97ECECDFC57B),
	SPH_C64(0x96DB4BDBDB4DAB90), SPH_C64(0x61A1BEA1A1C05F1F),
	SPH_C64(0x1C8D0E8D8D910783), SPH_C64(0xF53DF43D3DC87AC9),
	SPH_C64(0xCC976697975B33F1), SPH_C64(0x0000000000000000),
	SPH_C64(0x36CF1BCFCFF983D4), SPH_C64(0x452BAC2B2B6E5687),
	SPH_C64(0x9776C57676E1ECB3), SPH_C64(0x6482328282E619B0),
	SPH_C64(0xFED67FD6D628B1A9), SPH_C64(0xD81B6C1B1BC33677),
	SPH_C64(0xC1B5EEB5B574775B), SPH_C64(0x11AF86AFAFBE4329),
	SPH_C64(0x776AB56A6A1DD4DF), SPH_C64(0xBA505D5050EAA00D),
	SPH_C64(0x1245094545578A4C), SPH_C64(0xCBF3EBF3F338FB18),
	SPH_C64(0x9D30C03030AD60F0), SPH_C64(0x2BEF9BEFEFC4C374),
	SPH_C64(0xE53FFC3F3FDA7EC3), SPH_C64(0x9255495555C7AA1C),
	SPH_C64(0x79A2B2A2A2DB5910), SPH_C64(0x03EA8FEAEAE9C965),
	SPH_C64(0x0F658965656ACAEC), SPH_C64(0xB9BAD2BABA036968),
	SPH_C64(0x652FBC2F2F4A5E93), SPH_C64(0x4EC027C0C08E9DE7),
	SPH_C64(0xBEDE5FDEDE60A181), SPH_C64(0xE01C701C1CFC386C),
	SPH_C64(0xBBFDD3FDFD46E72E), SPH_C64(0x524D294D4D1F9A64),
	SPH_C64(0xE4927292927639E0), SPH_C64(0x8F75C97575FAEABC),
	SPH_C64(0x3006180606360C1E), SPH_C64(0x248A128A8AAE0998),
	SPH_C64(0xF9B2F2B2B24B7940), SPH_C64(0x63E6BFE6E685D159),
	SPH_C64(0x700E380E0E7E1C36), SPH_C64(0xF81F7C1F1FE73E63),
	SPH_C64(0x376295626255C4F7), SPH_C64(0xEED477D4D43AB5A3),
	SPH_C64(0x29A89AA8A8814D32), SPH_C64(0xC4966296965231F4),
	SPH_C64(0x9BF9C3F9F962EF3A), SPH_C64(0x66C533C5C5A397F6),
	SPH_C64(0x3525942525104AB1), SPH_C64(0xF259795959ABB220),
	SPH_C64(0x54842A8484D015AE), SPH_C64(0xB772D57272C5E4A7),
	SPH_C64(0xD539E43939EC72DD), SPH_C64(0x5A4C2D4C4C169861),
	SPH_C64(0xCA5E655E5E94BC3B), SPH_C64(0xE778FD78789FF085),
	SPH_C64(0xDD38E03838E570D8), SPH_C64(0x148C0A8C8C980586),
	SPH_C64(0xC6D163D1D117BFB2), SPH_C64(0x41A5AEA5A5E4570B),
	SPH_C64(0x43E2AFE2E2A1D94D), SPH_C64(0x2F619961614EC2F8),
	SPH_C64(0xF1B3F6B3B3427B45), SPH_C64(0x15218421213442A5),
	SPH_C64(0x949C4A9C9C0825D6), SPH_C64(0xF01E781E1EEE3C66),
	SPH_C64(0x2243114343618652), SPH_C64(0x76C73BC7C7B193FC),
	SPH_C64(0xB3FCD7FCFC4FE52B), SPH_C64(0x2004100404240814),
	SPH_C64(0xB251595151E3A208), SPH_C64(0xBC995E9999252FC7),
	SPH_C64(0x4F6DA96D6D22DAC4), SPH_C64(0x680D340D0D651A39),
	SPH_C64(0x83FACFFAFA79E935), SPH_C64(0xB6DF5BDFDF69A384),
	SPH_C64(0xD77EE57E7EA9FC9B), SPH_C64(0x3D249024241948B4),
	SPH_C64(0xC53BEC3B3BFE76D7), SPH_C64(0x31AB96ABAB9A4B3D),
	SPH_C64(0x3ECE1FCECEF081D1), SPH_C64(0x8811441111992255),
	SPH_C64(0x0C8F068F8F830389), SPH_C64(0x4A4E254E4E049C6B),
	SPH_C64(0xD1B7E6B7B7667351), SPH_C64(0x0BEB8BEBEBE0CB60),
	SPH_C64(0xFD3CF03C3CC178CC), SPH_C64(0x7C813E8181FD1FBF),
	SPH_C64(0xD4946A94944035FE), SPH_C64(0xEBF7FBF7F71CF30C),
	SPH_C64(0xA1B9DEB9B9186F67), SPH_C64(0x98134C13138B265F),
	SPH_C64(0x7D2CB02C2C51589C), SPH_C64(0xD6D36BD3D305BBB8),
	SPH_C64(0x6BE7BBE7E78CD35C), SPH_C64(0x576EA56E6E39DCCB),
	SPH_C64(0x6EC437C4C4AA95F3), SPH_C64(0x18030C03031B060F),
	SPH_C64(0x8A56455656DCAC13), SPH_C64(0x1A440D44445E8849),
	SPH_C64(0xDF7FE17F7FA0FE9E), SPH_C64(0x21A99EA9A9884F37),
	SPH_C64(0x4D2AA82A2A675482), SPH_C64(0xB1BBD6BBBB0A6B6D),
	SPH_C64(0x46C123C1C1879FE2), SPH_C64(0xA253515353F1A602),
	SPH_C64(0xAEDC57DCDC72A58B), SPH_C64(0x580B2C0B0B531627),
	SPH_C64(0x9C9D4E9D9D0127D3), SPH_C64(0x476CAD6C6C2BD8C1),
	SPH_C64(0x9531C43131A462F5), SPH_C64(0x8774CD7474F3E8B9),
	SPH_C64(0xE3F6FFF6F615F109), SPH_C64(0x0A460546464C8C43),
	SPH_C64(0x09AC8AACACA54526), SPH_C64(0x3C891E8989B50F97),
	SPH_C64(0xA014501414B42844), SPH_C64(0x5BE1A3E1E1BADF42),
	SPH_C64(0xB016581616A62C4E), SPH_C64(0xCD3AE83A3AF774D2),
	SPH_C64(0x6F69B9696906D2D0), SPH_C64(0x480924090941122D),
	SPH_C64(0xA770DD7070D7E0AD), SPH_C64(0xD9B6E2B6B66F7154),
	SPH_C64(0xCED067D0D01EBDB7), SPH_C64(0x3BED93EDEDD6C77E),
	SPH_C64(0x2ECC17CCCCE285DB), SPH_C64(0x2A42154242688457),
	SPH_C64(0xB4985A98982C2DC2), SPH_C64(0x49A4AAA4A4ED550E),
	SPH_C64(0x5D28A02828755088), SPH_C64(0xDA5C6D5C5C86B831),
	SPH_C64(0x93F8C7F8F86BED3F), SPH_C64(0x4486228686C211A4)
};

static const sph_u64 plain_T4[256] = {
	SPH_C64(0x18601818D83078C0), SPH_C64(0x238C23232646AF05),
	SPH_C64(0xC63FC6C6B891F97E), SPH_C64(0xE887E8E8FBCD6F13),
	SPH_C64(0x87268787CB13A14C), SPH_C64(0xB8DAB8B8116D62A9),
	SPH_C64(0x0104010109020508), SPH_C64(0x4F214F4F0D9E6E42),
	SPH_C64(0x36D836369B6CEEAD), SPH_C64(0xA6A2A6A6FF510459),
	SPH_C64(0xD26FD2D20CB9BDDE), SPH_C64(0xF5F3F5F50EF706FB),
	SPH_C64(0x79F9797996F280EF), SPH_C64(0x6FA16F6F30DECE5F),
	SPH_C64(0x917E91916D3FEFFC), SPH_C64(0x52555252F8A407AA),
	SPH_C64(0x609D606047C0FD27), SPH_C64(0xBCCABCBC35657689),
	SPH_C64(0x9B569B9B372BCDAC), SPH_C64(0x8E028E8E8A018C04),
	SPH_C64(0xA3B6A3A3D25B1571), SPH_C64(0x0C300C0C6C183C60),
	SPH_C64(0x7BF17B7B84F68AFF), SPH_C64(0x35D43535806AE1B5),
	SPH_C64(0x1D741D1DF53A69E8), SPH_C64(0xE0A7E0E0B3DD4753),
	SPH_C64(0xD77BD7D721B3ACF6), SPH_C64(0xC22FC2C29C99ED5E),
	SPH_C64(0x2EB82E2E435C966D), SPH_C64(0x4B314B4B29967A62),
	SPH_C64(0xFEDFFEFE5DE121A3), SPH_C64(0x57415757D5AE1682),
	SPH_C64(0x15541515BD2A41A8), SPH_C64(0x77C17777E8EEB69F),
	SPH_C64(0x37DC3737926EEBA5), SPH_C64(0xE5B3E5E59ED7567B),
	SPH_C64(0x9F469F9F1323D98C), SPH_C64(0xF0E7F0F023FD17D3),
	SPH_C64(0x4A354A4A20947F6A), SPH_C64(0xDA4FDADA44A9959E),
	SPH_C64(0x587D5858A2B025FA), SPH_C64(0xC903C9C9CF8FCA06),
	SPH_C64(0x29A429297C528D55), SPH_C64(0x0A280A0A5A142250),
	SPH_C64(0xB1FEB1B1507F4FE1), SPH_C64(0xA0BAA0A0C95D1A69),
	SPH_C64(0x6BB16B6B14D6DA7F), SPH_C64(0x852E8585D917AB5C),
	SPH_C64(0xBDCEBDBD3C677381), SPH_C64(0x5D695D5D8FBA34D2),
	SPH_C64(0x1040101090205080), SPH_C64(0xF4F7F4F407F503F3),
	SPH_C64(0xCB0BCBCBDD8BC016), SPH_C64(0x3EF83E3ED37CC6ED),
	SPH_C64(0x051405052D0A1128), SPH_C64(0x6781676778CEE61F),
	SPH_C64(0xE4B7E4E497D55373), SPH_C64(0x279C2727024EBB25),
	SPH_C64(0x4119414173825832), SPH_C64(0x8B168B8BA70B9D2C),
	SPH_C64(0xA7A6A7A7F6530151), SPH_C64(0x7DE97D7DB2FA94CF),
	SPH_C64(0x956E95954937FBDC), SPH_C64(0xD847D8D856AD9F8E),
	SPH_C64(0xFBCBFBFB70EB308B), SPH_C64(0xEE9FEEEECDC17123),
	SPH_C64(0x7CED7C7CBBF891C7), SPH_C64(0x6685666671CCE317),
	SPH_C64(0xDD53DDDD7BA78EA6), SPH_C64(0x175C1717AF2E4BB8),
	SPH_C64(0x47014747458E4602), SPH_C64(0x9E429E9E1A21DC84),
	SPH_C64(0xCA0FCACAD489C51E), SPH_C64(0x2DB42D2D585A9975),
	SPH_C64(0xBFC6BFBF2E637991), SPH_C64(0x071C07073F0E1B38),
	SPH_C64(0xAD8EADADAC472301), SPH_C64(0x5A755A5AB0B42FEA),
	SPH_C64(0x83368383EF1BB56C), SPH_C64(0x33CC3333B666FF85),
	SPH_C64(0x639163635CC6F23F), SPH_C64(0x0208020212040A10),
	SPH_C64(0xAA92AAAA93493839), SPH_C64(0x71D97171DEE2A8AF),
	SPH_C64(0xC807C8C8C68DCF0E), SPH_C64(0x19641919D1327DC8),
	SPH_C64(0x493949493B927072), SPH_C64(0xD943D9D95FAF9A86),
	SPH_C64(0xF2EFF2F231F91DC3), SPH_C64(0xE3ABE3E3A8DB484B),
	SPH_C64(0x5B715B5BB9B62AE2), SPH_C64(0x881A8888BC0D9234),
	SPH_C64(0x9A529A9A3E29C8A4), SPH_C64(0x269826260B4CBE2D),
	SPH_C64(0x32C83232BF64FA8D), SPH_C64(0xB0FAB0B0597D4AE9),
	SPH_C64(0xE983E9E9F2CF6A1B), SPH_C64(0x0F3C0F0F771E3378),
	SPH_C64(0xD573D5D533B7A6E6), SPH_C64(0x803A8080F41DBA74),
	SPH_C64(0xBEC2BEBE27617C99), SPH_C64(0xCD13CDCDEB87DE26),
	SPH_C64(0x34D034348968E4BD), SPH_C64(0x483D48483290757A),
	SPH_C64(0xFFDBFFFF54E324AB), SPH_C64(0x7AF57A7A8DF48FF7),
	SPH_C64(0x907A9090643DEAF4), SPH_C64(0x5F615F5F9DBE3EC2),
	SPH_C64(0x208020203D40A01D), SPH_C64(0x68BD68680FD0D567),
	SPH_C64(0x1A681A1ACA3472D0), SPH_C64(0xAE82AEAEB7412C19),
	SPH_C64(0xB4EAB4B47D755EC9), SPH_C64(0x544D5454CEA8199A),
	SPH_C64(0x937693937F3BE5EC), SPH_C64(0x228822222F44AA0D),
	SPH_C64(0x648D646463C8E907), SPH_C64(0xF1E3F1F12AFF12DB),
	SPH_C64(0x73D17373CCE6A2BF), SPH_C64(0x1248121282245A90),
	SPH_C64(0x401D40407A805D3A), SPH_C64(0x0820080848102840),
	SPH_C64(0xC32BC3C3959BE856), SPH_C64(0xEC97ECECDFC57B33),
	SPH_C64(0xDB4BDBDB4DAB9096), SPH_C64(0xA1BEA1A1C05F1F61),
	SPH_C64(0x8D0E8D8D9107831C), SPH_C64(0x3DF43D3DC87AC9F5),
	SPH_C64(0x976697975B33F1CC), SPH_C64(0x0000000000000000),
	SPH_C64(0xCF1BCFCFF983D436), SPH_C64(0x2BAC2B2B6E568745),
	SPH_C64(0x76C57676E1ECB397), SPH_C64(0x82328282E619B064),
	SPH_C64(0xD67FD6D628B1A9FE), SPH_C64(0x1B6C1B1BC33677D8),
	SPH_C64(0xB5EEB5B574775BC1), SPH_C64(0xAF86AFAFBE432911),
	SPH_C64(0x6AB56A6A1DD4DF77), SPH_C64(0x505D5050EAA00DBA),
	SPH_C64(0x45094545578A4C12), SPH_C64(0xF3EBF3F338FB18CB),
	SPH_C64(0x30C03030AD60F09D), SPH_C64(0xEF9BEFEFC4C3742B),
	SPH_C64(0x3FFC3F3FDA7EC3E5), SPH_C64(0x55495555C7AA1C92),
	SPH_C64(0xA2B2A2A2DB591079), SPH_C64(0xEA8FEAEAE9C96503),
	SPH_C64(0x658965656ACAEC0F), SPH_C64(0xBAD2BABA036968B9),
	SPH_C64(0x2FBC2F2F4A5E9365), SPH_C64(0xC027C0C08E9DE74E),
	SPH_C64(0xDE5FDEDE60A181BE), SPH_C64(0x1C701C1CFC386CE0),
	SPH_C64(0xFDD3FDFD46E72EBB), SPH_C64(0x4D294D4D1F9A6452),
	SPH_C64(0x927292927639E0E4), SPH_C64(0x75C97575FAEABC8F),
	SPH_C64(0x06180606360C1E30), SPH_C64(0x8A128A8AAE099824),
	SPH_C64(0xB2F2B2B24B7940F9), SPH_C64(0xE6BFE6E685D15963),
	SPH_C64(0x0E380E0E7E1C3670), SPH_C64(0x1F7C1F1FE73E63F8),
	SPH_C64(0x6295626255C4F737), SPH_C64(0xD477D4D43AB5A3EE),
	SPH_C64(0xA89AA8A8814D3229), SPH_C64(0x966296965231F4C4),
	SPH_C64(0xF9C3F9F962EF3A9B), SPH_C64(0xC533C5C5A397F666),
	SPH_C64(0x25942525104AB135), SPH_C64(0x59795959ABB220F2),
	SPH_C64(0x842A8484D015AE54), SPH_C64(0x72D57272C5E4A7B7),
	SPH_C64(0x39E43939EC72DDD5), SPH_C64(0x4C2D4C4C1698615A),
	SPH_C64(0x5E655E5E94BC3BCA), SPH_C64(0x78FD78789FF085E7),
	SPH_C64(0x38E03838E570D8DD), SPH_C64(0x8C0A8C8C98058614),
	SPH_C64(0xD163D1D117BFB2C6), SPH_C64(0xA5AEA5A5E4570B41),
	SPH_C64(0xE2AFE2E2A1D94D43), SPH_C64(0x619961614EC2F82F),
	SPH_C64(0xB3F6B3B3427B45F1), SPH_C64(0x218421213442A515),
	SPH_C64(0x9C4A9C9C0825D694), SPH_C64(0x1E781E1EEE3C66F0),
	SPH_C64(0x4311434361865222), SPH_C64(0xC73BC7C7B193FC76),
	SPH_C64(0xFCD7FCFC4FE52BB3), SPH_C64(0x0410040424081420),
	SPH_C64(0x51595151E3A208B2), SPH_C64(0x995E9999252FC7BC),
	SPH_C64(0x6DA96D6D22DAC44F), SPH_C64(0x0D340D0D651A3968),
	SPH_C64(0xFACFFAFA79E93583), SPH_C64(0xDF5BDFDF69A384B6),
	SPH_C64(0x7EE57E7EA9FC9BD7), SPH_C64(0x249024241948B43D),
	SPH_C64(0x3BEC3B3BFE76D7C5), SPH_C64(0xAB96ABAB9A4B3D31),
	SPH_C64(0xCE1FCECEF081D13E), SPH_C64(0x1144111199225588),
	SPH_C64(0x8F068F8F8303890C), SPH_C64(0x4E254E4E049C6B4A),
	SPH_C64(0xB7E6B7B7667351D1), SPH_C64(0xEB8BEBEBE0CB600B),
	SPH_C64(0x3CF03C3CC178CCFD), SPH_C64(0x813E8181FD1FBF7C),
	SPH_C64(0x946A94944035FED4), SPH_C64(0xF7FBF7F71CF30CEB),
	SPH_C64(0xB9DEB9B9186F67A1), SPH_C64(0x134C13138B265F98),
	SPH_C64(0x2CB02C2C51589C7D), SPH_C64(0xD36BD3D305BBB8D6),
	SPH_C64(0xE7BBE7E78CD35C6B), SPH_C64(0x6EA56E6E39DCCB57),
	SPH_C64(0xC437C4C4AA95F36E), SPH_C64(0x030C03031B060F18),
	SPH_C64(0x56455656DCAC138A), SPH_C64(0x440D44445E88491A),
	SPH_C64(0x7FE17F7FA0FE9EDF), SPH_C64(0xA99EA9A9884F3721),
	SPH_C64(0x2AA82A2A6754824D), SPH_C64(0xBBD6BBBB0A6B6DB1),
	SPH_C64(0xC123C1C1879FE246), SPH_C64(0x53515353F1A602A2),
	SPH_C64(0xDC57DCDC72A58BAE), SPH_C64(0x0B2C0B0B53162758),
	SPH_C64(0x9D4E9D9D0127D39C), SPH_C64(0x6CAD6C6C2BD8C147),
	SPH_C64(0x31C43131A462F595), SPH_C64(0x74CD7474F3E8B987),
	SPH_C64(0xF6FFF6F615F109E3), SPH_C64(0x460546464C8C430A),
	SPH_C64(0xAC8AACACA5452609), SPH_C64(0x891E8989B50F973C),
	SPH_C64(0x14501414B42844A0), SPH_C64(0xE1A3E1E1BADF425B),
	SPH_C64(0x16581616A62C4EB0), SPH_C64(0x3AE83A3AF774D2CD),
	SPH_C64(0x69B9696906D2D06F), SPH_C64(0x0924090941122D48),
	SPH_C64(0x70DD7070D7E0ADA7), SPH_C64(0xB6E2B6B66F7154D9),
	SPH_C64(0xD067D0D01EBDB7CE), SPH_C64(0xED93EDEDD6C77E3B),
	SPH_C64(0xCC17CCCCE285DB2E), SPH_C64(0x421542426884572A),
	SPH_C64(0x985A98982C2DC2B4), SPH_C64(0xA4AAA4A4ED550E49),
	SPH_C64(0x28A028287550885D), SPH_C64(0x5C6D5C5C86B831DA),
	SPH_C64(0xF8C7F8F86BED3F93), SPH_C64(0x86228686C211A444)
};

static const sph_u64 plain_T5[256] = {
	SPH_C64(0x601818D83078C018), SPH_C64(0x8C23232646AF0523),
	SPH_C64(0x3FC6C6B891F97EC6), SPH_C64(0x87E8E8FBCD6F13E8),
	SPH_C64(0x268787CB13A14C87), SPH_C64(0xDAB8B8116D62A9B8),
	SPH_C64(0x0401010902050801), SPH_C64(0x214F4F0D9E6E424F),
	SPH_C64(0xD836369B6CEEAD36), SPH_C64(0xA2A6A6FF510459A6),
	SPH_C64(0x6FD2D20CB9BDDED2), SPH_C64(0xF3F5F50EF706FBF5),
	SPH_C64(0xF9797996F280EF79), SPH_C64(0xA16F6F30DECE5F6F),
	SPH_C64(0x7E91916D3FEFFC91), SPH_C64(0x555252F8A407AA52),
	SPH_C64(0x9D606047C0FD2760), SPH_C64(0xCABCBC35657689BC),
	SPH_C64(0x569B9B372BCDAC9B), SPH_C64(0x028E8E8A018C048E),
	SPH_C64(0xB6A3A3D25B1571A3), SPH_C64(0x300C0C6C183C600C),
	SPH_C64(0xF17B7B84F68AFF7B), SPH_C64(0xD43535806AE1B535),
	SPH_C64(0x741D1DF53A69E81D), SPH_C64(0xA7E0E0B3DD4753E0),
	SPH_C64(0x7BD7D721B3ACF6D7), SPH_C64(0x2FC2C29C99ED5EC2),
	SPH_C64(0xB82E2E435C966D2E), SPH_C64(0x314B4B29967A624B),
	SPH_C64(0xDFFEFE5DE121A3FE), SPH_C64(0x415757D5AE168257),
	SPH_C64(0x541515BD2A41A815), SPH_C64(0xC17777E8EEB69F77),
	SPH_C64(0xDC3737926EEBA537), SPH_C64(0xB3E5E59ED7567BE5),
	SPH_C64(0x469F9F1323D98C9F), SPH_C64(0xE7F0F023FD17D3F0),
	SPH_C64(0x354A4A20947F6A4A), SPH_C64(0x4FDADA44A9959EDA),
	SPH_C64(0x7D5858A2B025FA58), SPH_C64(0x03C9C9CF8FCA06C9),
	SPH_C64(0xA429297C528D5529), SPH_C64(0x280A0A5A1422500A),
	SPH_C64(0xFEB1B1507F4FE1B1), SPH_C64(0xBAA0A0C95D1A69A0),
	SPH_C64(0xB16B6B14D6DA7F6B), SPH_C64(0x2E8585D917AB5C85),
	SPH_C64(0xCEBDBD3C677381BD), SPH_C64(0x695D5D8FBA34D25D),
	SPH_C64(0x4010109020508010), SPH_C64(0xF7F4F407F503F3F4),
	SPH_C64(0x0BCBCBDD8BC016CB), SPH_C64(0xF83E3ED37CC6ED3E),
	SPH_C64(0x1405052D0A112805), SPH_C64(0x81676778CEE61F67),
	SPH_C64(0xB7E4E497D55373E4), SPH_C64(0x9C2727024EBB2527),
	SPH_C64(0x1941417382583241), SPH_C64(0x168B8BA70B9D2C8B),
	SPH_C64(0xA6A7A7F6530151A7), SPH_C64(0xE97D7DB2FA94CF7D),
	SPH_C64(0x6E95954937FBDC95), SPH_C64(0x47D8D856AD9F8ED8),
	SPH_C64(0xCBFBFB70EB308BFB), SPH_C64(0x9FEEEECDC17123EE),
	SPH_C64(0xED7C7CBBF891C77C), SPH_C64(0x85666671CCE31766),
	SPH_C64(0x53DDDD7BA78EA6DD), SPH_C64(0x5C1717AF2E4BB817),
	SPH_C64(0x014747458E460247), SPH_C64(0x429E9E1A21DC849E),
	SPH_C64(0x0FCACAD489C51ECA), SPH_C64(0xB42D2D585A99752D),
	SPH_C64(0xC6BFBF2E637991BF), SPH_C64(0x1C07073F0E1B3807),
	SPH_C64(0x8EADADAC472301AD), SPH_C64(0x755A5AB0B42FEA5A),
	SPH_C64(0x368383EF1BB56C83), SPH_C64(0xCC3333B666FF8533),
	SPH_C64(0x9163635CC6F23F63), SPH_C64(0x08020212040A1002),
	SPH_C64(0x92AAAA93493839AA), SPH_C64(0xD97171DEE2A8AF71),
	SPH_C64(0x07C8C8C68DCF0EC8), SPH_C64(0x641919D1327DC819),
	SPH_C64(0x3949493B92707249), SPH_C64(0x43D9D95FAF9A86D9),
	SPH_C64(0xEFF2F231F91DC3F2), SPH_C64(0xABE3E3A8DB484BE3),
	SPH_C64(0x715B5BB9B62AE25B), SPH_C64(0x1A8888BC0D923488),
	SPH_C64(0x529A9A3E29C8A49A), SPH_C64(0x9826260B4CBE2D26),
	SPH_C64(0xC83232BF64FA8D32), SPH_C64(0xFAB0B0597D4AE9B0),
	SPH_C64(0x83E9E9F2CF6A1BE9), SPH_C64(0x3C0F0F771E33780F),
	SPH_C64(0x73D5D533B7A6E6D5), SPH_C64(0x3A8080F41DBA7480),
	SPH_C64(0xC2BEBE27617C99BE), SPH_C64(0x13CDCDEB87DE26CD),
	SPH_C64(0xD034348968E4BD34), SPH_C64(0x3D48483290757A48),
	SPH_C64(0xDBFFFF54E324ABFF), SPH_C64(0xF57A7A8DF48FF77A),
	SPH_C64(0x7A9090643DEAF490), SPH_C64(0x615F5F9DBE3EC25F),
	SPH_C64(0x8020203D40A01D20), SPH_C64(0xBD68680FD0D56768),
	SPH_C64(0x681A1ACA3472D01A), SPH_C64(0x82AEAEB7412C19AE),
	SPH_C64(0xEAB4B47D755EC9B4), SPH_C64(0x4D5454CEA8199A54),
	SPH_C64(0x7693937F3BE5EC93), SPH_C64(0x8822222F44AA0D22),
	SPH_C64(0x8D646463C8E90764), SPH_C64(0xE3F1F12AFF12DBF1),
	SPH_C64(0xD17373CCE6A2BF73), SPH_C64(0x48121282245A9012),
	SPH_C64(0x1D40407A805D3A40), SPH_C64(0x2008084810284008),
	SPH_C64(0x2BC3C3959BE856C3), SPH_C64(0x97ECECDFC57B33EC),
	SPH_C64(0x4BDBDB4DAB9096DB), SPH_C64(0xBEA1A1C05F1F61A1),
	SPH_C64(0x0E8D8D9107831C8D), SPH_C64(0xF43D3DC87AC9F53D),
	SPH_C64(0x6697975B33F1CC97), SPH_C64(0x0000000000000000),
	SPH_C64(0x1BCFCFF983D436CF), SPH_C64(0xAC2B2B6E5687452B),
	SPH_C64(0xC57676E1ECB39776), SPH_C64(0x328282E619B06482),
	SPH_C64(0x7FD6D628B1A9FED6), SPH_C64(0x6C1B1BC33677D81B),
	SPH_C64(0xEEB5B574775BC1B5), SPH_C64(0x86AFAFBE432911AF),
	SPH_C64(0xB56A6A1DD4DF776A), SPH_C64(0x5D5050EAA00DBA50),
	SPH_C64(0x094545578A4C1245), SPH_C64(0xEBF3F338FB18CBF3),
	SPH_C64(0xC03030AD60F09D30), SPH_C64(0x9BEFEFC4C3742BEF),
	SPH_C64(0xFC3F3FDA7EC3E53F), SPH_C64(0x495555C7AA1C9255),
	SPH_C64(0xB2A2A2DB591079A2), SPH_C64(0x8FEAEAE9C96503EA),
	SPH_C64(0x8965656ACAEC0F65), SPH_C64(0xD2BABA036968B9BA),
	SPH_C64(0xBC2F2F4A5E93652F), SPH_C64(0x27C0C08E9DE74EC0),
	SPH_C64(0x5FDEDE60A181BEDE), SPH_C64(0x701C1CFC386CE01C),
	SPH_C64(0xD3FDFD46E72EBBFD), SPH_C64(0x294D4D1F9A64524D),
	SPH_C64(0x7292927639E0E492), SPH_C64(0xC97575FAEABC8F75),
	SPH_C64(0x180606360C1E3006), SPH_C64(0x128A8AAE0998248A),
	SPH_C64(0xF2B2B24B7940F9B2), SPH_C64(0xBFE6E685D15963E6),
	SPH_C64(0x380E0E7E1C36700E), SPH_C64(0x7C1F1FE73E63F81F),
	SPH_C64(0x95626255C4F73762), SPH_C64(0x77D4D43AB5A3EED4),
	SPH_C64(0x9AA8A8814D3229A8), SPH_C64(0x6296965231F4C496),
	SPH_C64(0xC3F9F962EF3A9BF9), SPH_C64(0x33C5C5A397F666C5),
	SPH_C64(0x942525104AB13525), SPH_C64(0x795959ABB220F259),
	SPH_C64(0x2A8484D015AE5484), SPH_C64(0xD57272C5E4A7B772),
	SPH_C64(0xE43939EC72DDD539), SPH_C64(0x2D4C4C1698615A4C),
	SPH_C64(0x655E5E94BC3BCA5E), SPH_C64(0xFD78789FF085E778),
	SPH_C64(0xE03838E570D8DD38), SPH_C64(0x0A8C8C980586148C),
	SPH_C64(0x63D1D117BFB2C6D1), SPH_C64(0xAEA5A5E4570B41A5),
	SPH_C64(0xAFE2E2A1D94D43E2), SPH_C64(0x9961614EC2F82F61),
	SPH_C64(0xF6B3B3427B45F1B3), SPH_C64(0x8421213442A51521),
	SPH_C64(0x4A9C9C0825D6949C), SPH_C64(0x781E1EEE3C66F01E),
	SPH_C64(0x1143436186522243), SPH_C64(0x3BC7C7B193FC76C7),
	SPH_C64(0xD7FCFC4FE52BB3FC), SPH_C64(0x1004042408142004),
	SPH_C64(0x595151E3A208B251), SPH_C64(0x5E9999252FC7BC99),
	SPH_C64(0xA96D6D22DAC44F6D), SPH_C64(0x340D0D651A39680D),
	SPH_C64(0xCFFAFA79E93583FA), SPH_C64(0x5BDFDF69A384B6DF),
	SPH_C64(0xE57E7EA9FC9BD77E), SPH_C64(0x9024241948B43D24),
	SPH_C64(0xEC3B3BFE76D7C53B), SPH_C64(0x96ABAB9A4B3D31AB),
	SPH_C64(0x1FCECEF081D13ECE), SPH_C64(0x4411119922558811),
	SPH_C64(0x068F8F8303890C8F), SPH_C64(0x254E4E049C6B4A4E),
	SPH_C64(0xE6B7B7667351D1B7), SPH_C64(0x8BEBEBE0CB600BEB),
	SPH_C64(0xF03C3CC178CCFD3C), SPH_C64(0x3E8181FD1FBF7C81),
	SPH_C64(0x6A94944035FED494), SPH_C64(0xFBF7F71CF30CEBF7),
	SPH_C64(0xDEB9B9186F67A1B9), SPH_C64(0x4C13138B265F9813),
	SPH_C64(0xB02C2C51589C7D2C), SPH_C64(0x6BD3D305BBB8D6D3),
	SPH_C64(0xBBE7E78CD35C6BE7), SPH_C64(0xA56E6E39DCCB576E),
	SPH_C64(0x37C4C4AA95F36EC4), SPH_C64(0x0C03031B060F1803),
	SPH_C64(0x455656DCAC138A56), SPH_C64(0x0D44445E88491A44),
	SPH_C64(0xE17F7FA0FE9EDF7F), SPH_C64(0x9EA9A9884F3721A9),
	SPH_C64(0xA82A2A6754824D2A), SPH_C64(0xD6BBBB0A6B6DB1BB),
	SPH_C64(0x23C1C1879FE246C1), SPH_C64(0x515353F1A602A253),
	SPH_C64(0x57DCDC72A58BAEDC), SPH_C64(0x2C0B0B531627580B),
	SPH_C64(0x4E9D9D0127D39C9D), SPH_C64(0xAD6C6C2BD8C1476C),
	SPH_C64(0xC43131A462F59531), SPH_C64(0xCD7474F3E8B98774),
	SPH_C64(0xFFF6F615F109E3F6), SPH_C64(0x0546464C8C430A46),
	SPH_C64(0x8AACACA5452609AC), SPH_C64(0x1E8989B50F973C89),
	SPH_C64(0x501414B42844A014), SPH_C64(0xA3E1E1BADF425BE1),
	SPH_C64(0x581616A62C4EB016), SPH_C64(0xE83A3AF774D2CD3A),
	SPH_C64(0xB9696906D2D06F69), SPH_C64(0x24090941122D4809),
	SPH_C64(0xDD7070D7E0ADA770), SPH_C64(0xE2B6B66F7154D9B6),
	SPH_C64(0x67D0D01EBDB7CED0), SPH_C64(0x93EDEDD6C77E3BED),
	SPH_C64(0x17CCCCE285DB2ECC), SPH_C64(0x1542426884572A42),
	SPH_C64(0x5A98982C2DC2B498), SPH_C64(0xAAA4A4ED550E49A4),
	SPH_C64(0xA028287550885D28), SPH_C64(0x6D5C5C86B831DA5C),
	SPH_C64(0xC7F8F86BED3F93F8), SPH_C64(0x228686C211A44486)
};

static const sph_u64 plain_T6[256] = {
	SPH_C64(0x1818D83078C01860), SPH_C64(0x23232646AF05238C),
	SPH_C64(0xC6C6B891F97EC63F), SPH_C64(0xE8E8FBCD6F13E887),
	SPH_C64(0x8787CB13A14C8726), SPH_C64(0xB8B8116D62A9B8DA),
	SPH_C64(0x0101090205080104), SPH_C64(0x4F4F0D9E6E424F21),
	SPH_C64(0x36369B6CEEAD36D8), SPH_C64(0xA6A6FF510459A6A2),
	SPH_C64(0xD2D20CB9BDDED26F), SPH_C64(0xF5F50EF706FBF5F3),
	SPH_C64(0x797996F280EF79F9), SPH_C64(0x6F6F30DECE5F6FA1),
	SPH_C64(0x91916D3FEFFC917E), SPH_C64(0x5252F8A407AA5255),
	SPH_C64(0x606047C0FD27609D), SPH_C64(0xBCBC35657689BCCA),
	SPH_C64(0x9B9B372BCDAC9B56), SPH_C64(0x8E8E8A018C048E02),
	SPH_C64(0xA3A3D25B1571A3B6), SPH_C64(0x0C0C6C183C600C30),
	SPH_C64(0x7B7B84F68AFF7BF1), SPH_C64(0x3535806AE1B535D4),
	SPH_C64(0x1D1DF53A69E81D74), SPH_C64(0xE0E0B3DD4753E0A7),
	SPH_C64(0xD7D721B3ACF6D77B), SPH_C64(0xC2C29C99ED5EC22F),
	SPH_C64(0x2E2E435C966D2EB8), SPH_C64(0x4B4B29967A624B31),
	SPH_C64(0xFEFE5DE121A3FEDF), SPH_C64(0x5757D5AE16825741),
	SPH_C64(0x1515BD2A41A81554), SPH_C64(0x7777E8EEB69F77C1),
	SPH_C64(0x3737926EEBA537DC), SPH_C64(0xE5E59ED7567BE5B3),
	SPH_C64(0x9F9F1323D98C9F46), SPH_C64(0xF0F023FD17D3F0E7),
	SPH_C64(0x4A4A20947F6A4A35), SPH_C64(0xDADA44A9959EDA4F),
	SPH_C64(0x5858A2B025FA587D), SPH_C64(0xC9C9CF8FCA06C903),
	SPH_C64(0x29297C528D5529A4), SPH_C64(0x0A0A5A1422500A28),
	SPH_C64(0xB1B1507F4FE1B1FE), SPH_C64(0xA0A0C95D1A69A0BA),
	SPH_C64(0x6B6B14D6DA7F6BB1), SPH_C64(0x8585D917AB5C852E),
	SPH_C64(0xBDBD3C677381BDCE), SPH_C64(0x5D5D8FBA34D25D69),
	SPH_C64(0x1010902050801040), SPH_C64(0xF4F407F503F3F4F7),
	SPH_C64(0xCBCBDD8BC016CB0B), SPH_C64(0x3E3ED37CC6ED3EF8),
	SPH_C64(0x05052D0A11280514), SPH_C64(0x676778CEE61F6781),
	SPH_C64(0xE4E497D55373E4B7), SPH_C64(0x2727024EBB25279C),
	SPH_C64(0x4141738258324119), SPH_C64(0x8B8BA70B9D2C8B16),
	SPH_C64(0xA7A7F6530151A7A6), SPH_C64(0x7D7DB2FA94CF7DE9),
	SPH_C64(0x95954937FBDC956E), SPH_C64(0xD8D856AD9F8ED847),
	SPH_C64(0xFBFB70EB308BFBCB), SPH_C64(0xEEEECDC17123EE9F),
	SPH_C64(0x7C7CBBF891C77CED), SPH_C64(0x666671CCE3176685),
	SPH_C64(0xDDDD7BA78EA6DD53), SPH_C64(0x1717AF2E4BB8175C),
	SPH_C64(0x4747458E46024701), SPH_C64(0x9E9E1A21DC849E42),
	SPH_C64(0xCACAD489C51ECA0F), SPH_C64(0x2D2D585A99752DB4),
	SPH_C64(0xBFBF2E637991BFC6), SPH_C64(0x07073F0E1B38071C),
	SPH_C64(0xADADAC472301AD8E), SPH_C64(0x5A5AB0B42FEA5A75),
	SPH_C64(0x8383EF1BB56C8336), SPH_C64(0x3333B666FF8533CC),
	SPH_C64(0x63635CC6F23F6391), SPH_C64(0x020212040A100208),
	SPH_C64(0xAAAA93493839AA92), SPH_C64(0x7171DEE2A8AF71D9),
	SPH_C64(0xC8C8C68DCF0EC807), SPH_C64(0x1919D1327DC81964),
	SPH_C64(0x49493B9270724939), SPH_C64(0xD9D95FAF9A86D943),
	SPH_C64(0xF2F231F91DC3F2EF), SPH_C64(0xE3E3A8DB484BE3AB),
	SPH_C64(0x5B5BB9B62AE25B71), SPH_C64(0x8888BC0D9234881A),
	SPH_C64(0x9A9A3E29C8A49A52), SPH_C64(0x26260B4CBE2D2698),
	SPH_C64(0x3232BF64FA8D32C8), SPH_C64(0xB0B0597D4AE9B0FA),
	SPH_C64(0xE9E9F2CF6A1BE983), SPH_C64(0x0F0F771E33780F3C),
	SPH_C64(0xD5D533B7A6E6D573), SPH_C64(0x8080F41DBA74803A),
	SPH_C64(0xBEBE27617C99BEC2), SPH_C64(0xCDCDEB87DE26CD13),
	SPH_C64(0x34348968E4BD34D0), SPH_C64(0x48483290757A483D),
	SPH_C64(0xFFFF54E324ABFFDB), SPH_C64(0x7A7A8DF48FF77AF5),
	SPH_C64(0x9090643DEAF4907A), SPH_C64(0x5F5F9DBE3EC25F61),
	SPH_C64(0x20203D40A01D2080), SPH_C64(0x68680FD0D56768BD),
	SPH_C64(0x1A1ACA3472D01A68), SPH_C64(0xAEAEB7412C19AE82),
	SPH_C64(0xB4B47D755EC9B4EA), SPH_C64(0x5454CEA8199A544D),
	SPH_C64(0x93937F3BE5EC9376), SPH_C64(0x22222F44AA0D2288),
	SPH_C64(0x646463C8E907648D), SPH_C64(0xF1F12AFF12DBF1E3),
	SPH_C64(0x7373CCE6A2BF73D1), SPH_C64(0x121282245A901248),
	SPH_C64(0x40407A805D3A401D), SPH_C64(0x0808481028400820),
	SPH_C64(0xC3C3959BE856C32B), SPH_C64(0xECECDFC57B33EC97),
	SPH_C64(0xDBDB4DAB9096DB4B), SPH_C64(0xA1A1C05F1F61A1BE),
	SPH_C64(0x8D8D9107831C8D0E), SPH_C64(0x3D3DC87AC9F53DF4),
	SPH_C64(0x97975B33F1CC9766), SPH_C64(0x0000000000000000),
	SPH_C64(0xCFCFF983D436CF1B), SPH_C64(0x2B2B6E5687452BAC),
	SPH_C64(0x7676E1ECB39776C5), SPH_C64(0x8282E619B0648232),
	SPH_C64(0xD6D628B1A9FED67F), SPH_C64(0x1B1BC33677D81B6C),
	SPH_C64(0xB5B574775BC1B5EE), SPH_C64(0xAFAFBE432911AF86),
	SPH_C64(0x6A6A1DD4DF776AB5), SPH_C64(0x5050EAA00DBA505D),
	SPH_C64(0x4545578A4C124509), SPH_C64(0xF3F338FB18CBF3EB),
	SPH_C64(0x3030AD60F09D30C0), SPH_C64(0xEFEFC4C3742BEF9B),
	SPH_C64(0x3F3FDA7EC3E53FFC), SPH_C64(0x5555C7AA1C925549),
	SPH_C64(0xA2A2DB591079A2B2), SPH_C64(0xEAEAE9C96503EA8F),
	SPH_C64(0x65656ACAEC0F6589), SPH_C64(0xBABA036968B9BAD2),
	SPH_C64(0x2F2F4A5E93652FBC), SPH_C64(0xC0C08E9DE74EC027),
	SPH_C64(0xDEDE60A181BEDE5F), SPH_C64(0x1C1CFC386CE01C70),
	SPH_C64(0xFDFD46E72EBBFDD3), SPH_C64(0x4D4D1F9A64524D29),
	SPH_C64(0x92927639E0E49272), SPH_C64(0x7575FAEABC8F75C9),
	SPH_C64(0x0606360C1E300618), SPH_C64(0x8A8AAE0998248A12),
	SPH_C64(0xB2B24B7940F9B2F2), SPH_C64(0xE6E685D15963E6BF),
	SPH_C64(0x0E0E7E1C36700E38), SPH_C64(0x1F1FE73E63F81F7C),
	SPH_C64(0x626255C4F7376295), SPH_C64(0xD4D43AB5A3EED477),
	SPH_C64(0xA8A8814D3229A89A), SPH_C64(0x96965231F4C49662),
	SPH_C64(0xF9F962EF3A9BF9C3), SPH_C64(0xC5C5A397F666C533),
	SPH_C64(0x2525104AB1352594), SPH_C64(0x5959ABB220F25979),
	SPH_C64(0x8484D015AE54842A), SPH_C64(0x7272C5E4A7B772D5),
	SPH_C64(0x3939EC72DDD539E4), SPH_C64(0x4C4C1698615A4C2D),
	SPH_C64(0x5E5E94BC3BCA5E65), SPH_C64(0x78789FF085E778FD),
	SPH_C64(0x3838E570D8DD38E0), SPH_C64(0x8C8C980586148C0A),
	SPH_C64(0xD1D117BFB2C6D163), SPH_C64(0xA5A5E4570B41A5AE),
	SPH_C64(0xE2E2A1D94D43E2AF), SPH_C64(0x61614EC2F82F6199),
	SPH_C64(0xB3B3427B45F1B3F6), SPH_C64(0x21213442A5152184),
	SPH_C64(0x9C9C0825D6949C4A), SPH_C64(0x1E1EEE3C66F01E78),
	SPH_C64(0x4343618652224311), SPH_C64(0xC7C7B193FC76C73B),
	SPH_C64(0xFCFC4FE52BB3FCD7), SPH_C64(0x0404240814200410),
	SPH_C64(0x5151E3A208B25159), SPH_C64(0x9999252FC7BC995E),
	SPH_C64(0x6D6D22DAC44F6DA9), SPH_C64(0x0D0D651A39680D34),
	SPH_C64(0xFAFA79E93583FACF), SPH_C64(0xDFDF69A384B6DF5B),
	SPH_C64(0x7E7EA9FC9BD77EE5), SPH_C64(0x24241948B43D2490),
	SPH_C64(0x3B3BFE76D7C53BEC), SPH_C64(0xABAB9A4B3D31AB96),
	SPH_C64(0xCECEF081D13ECE1F), SPH_C64(0x1111992255881144),
	SPH_C64(0x8F8F8303890C8F06), SPH_C64(0x4E4E049C6B4A4E25),
	SPH_C64(0xB7B7667351D1B7E6), SPH_C64(0xEBEBE0CB600BEB8B),
	SPH_C64(0x3C3CC178CCFD3CF0), SPH_C64(0x8181FD1FBF7C813E),
	SPH_C64(0x94944035FED4946A), SPH_C64(0xF7F71CF30CEBF7FB),
	SPH_C64(0xB9B9186F67A1B9DE), SPH_C64(0x13138B265F98134C),
	SPH_C64(0x2C2C51589C7D2CB0), SPH_C64(0xD3D305BBB8D6D36B),
	SPH_C64(0xE7E78CD35C6BE7BB), SPH_C64(0x6E6E39DCCB576EA5),
	SPH_C64(0xC4C4AA95F36EC437), SPH_C64(0x03031B060F18030C),
	SPH_C64(0x5656DCAC138A5645), SPH_C64(0x44445E88491A440D),
	SPH_C64(0x7F7FA0FE9EDF7FE1), SPH_C64(0xA9A9884F3721A99E),
	SPH_C64(0x2A2A6754824D2AA8), SPH_C64(0xBBBB0A6B6DB1BBD6),
	SPH_C64(0xC1C1879FE246C123), SPH_C64(0x5353F1A602A25351),
	SPH_C64(0xDCDC72A58BAEDC57), SPH_C64(0x0B0B531627580B2C),
	SPH_C64(0x9D9D0127D39C9D4E), SPH_C64(0x6C6C2BD8C1476CAD),
	SPH_C64(0x3131A462F59531C4), SPH_C64(0x7474F3E8B98774CD),
	SPH_C64(0xF6F615F109E3F6FF), SPH_C64(0x46464C8C430A4605),
	SPH_C64(0xACACA5452609AC8A), SPH_C64(0x8989B50F973C891E),
	SPH_C64(0x1414B42844A01450), SPH_C64(0xE1E1BADF425BE1A3),
	SPH_C64(0x1616A62C4EB01658), SPH_C64(0x3A3AF774D2CD3AE8),
	SPH_C64(0x696906D2D06F69B9), SPH_C64(0x090941122D480924),
	SPH_C64(0x7070D7E0ADA770DD), SPH_C64(0xB6B66F7154D9B6E2),
	SPH_C64(0xD0D01EBDB7CED067), SPH_C64(0xEDEDD6C77E3BED93),
	SPH_C64(0xCCCCE285DB2ECC17), SPH_C64(0x42426884572A4215),
	SPH_C64(0x98982C2DC2B4985A), SPH_C64(0xA4A4ED550E49A4AA),
	SPH_C64(0x28287550885D28A0), SPH_C64(0x5C5C86B831DA5C6D),
	SPH_C64(0xF8F86BED3F93F8C7), SPH_C64(0x8686C211A4448622)
};

static const sph_u64 plain_T7[256] = {
	SPH_C64(0x18D83078C0186018), SPH_C64(0x232646AF05238C23),
	SPH_C64(0xC6B891F97EC63FC6), SPH_C64(0xE8FBCD6F13E887E8),
	SPH_C64(0x87CB13A14C872687), SPH_C64(0xB8116D62A9B8DAB8),
	SPH_C64(0x0109020508010401), SPH_C64(0x4F0D9E6E424F214F),
	SPH_C64(0x369B6CEEAD36D836), SPH_C64(0xA6FF510459A6A2A6),
	SPH_C64(0xD20CB9BDDED26FD2), SPH_C64(0xF50EF706FBF5F3F5),
	SPH_C64(0x7996F280EF79F979), SPH_C64(0x6F30DECE5F6FA16F),
	SPH_C64(0x916D3FEFFC917E91), SPH_C64(0x52F8A407AA525552),
	SPH_C64(0x6047C0FD27609D60), SPH_C64(0xBC35657689BCCABC),
	SPH_C64(0x9B372BCDAC9B569B), SPH_C64(0x8E8A018C048E028E),
	SPH_C64(0xA3D25B1571A3B6A3), SPH_C64(0x0C6C183C600C300C),
	SPH_C64(0x7B84F68AFF7BF17B), SPH_C64(0x35806AE1B535D435),
	SPH_C64(0x1DF53A69E81D741D), SPH_C64(0xE0B3DD4753E0A7E0),
	SPH_C64(0xD721B3ACF6D77BD7), SPH_C64(0xC29C99ED5EC22FC2),
	SPH_C64(0x2E435C966D2EB82E), SPH_C64(0x4B29967A624B314B),
	SPH_C64(0xFE5DE121A3FEDFFE), SPH_C64(0x57D5AE1682574157),
	SPH_C64(0x15BD2A41A8155415), SPH_C64(0x77E8EEB69F77C177),
	SPH_C64(0x37926EEBA537DC37), SPH_C64(0xE59ED7567BE5B3E5),
	SPH_C64(0x9F1323D98C9F469F), SPH_C64(0xF023FD17D3F0E7F0),
	SPH_C64(0x4A20947F6A4A354A), SPH_C64(0xDA44A9959EDA4FDA),
	SPH_C64(0x58A2B025FA587D58), SPH_C64(0xC9CF8FCA06C903C9),
	SPH_C64(0x297C528D5529A429), SPH_C64(0x0A5A1422500A280A),
	SPH_C64(0xB1507F4FE1B1FEB1), SPH_C64(0xA0C95D1A69A0BAA0),
	SPH_C64(0x6B14D6DA7F6BB16B), SPH_C64(0x85D917AB5C852E85),
	SPH_C64(0xBD3C677381BDCEBD), SPH_C64(0x5D8FBA34D25D695D),
	SPH_C64(0x1090205080104010), SPH_C64(0xF407F503F3F4F7F4),
	SPH_C64(0xCBDD8BC016CB0BCB), SPH_C64(0x3ED37CC6ED3EF83E),
	SPH_C64(0x052D0A1128051405), SPH_C64(0x6778CEE61F678167),
	SPH_C64(0xE497D55373E4B7E4), SPH_C64(0x27024EBB25279C27),
	SPH_C64(0x4173825832411941), SPH_C64(0x8BA70B9D2C8B168B),
	SPH_C64(0xA7F6530151A7A6A7), SPH_C64(0x7DB2FA94CF7DE97D),
	SPH_C64(0x954937FBDC956E95), SPH_C64(0xD856AD9F8ED847D8),
	SPH_C64(0xFB70EB308BFBCBFB), SPH_C64(0xEECDC17123EE9FEE),
	SPH_C64(0x7CBBF891C77CED7C), SPH_C64(0x6671CCE317668566),
	SPH_C64(0xDD7BA78EA6DD53DD), SPH_C64(0x17AF2E4BB8175C17),
	SPH_C64(0x47458E4602470147), SPH_C64(0x9E1A21DC849E429E),
	SPH_C64(0xCAD489C51ECA0FCA), SPH_C64(0x2D585A99752DB42D),
	SPH_C64(0xBF2E637991BFC6BF), SPH_C64(0x073F0E1B38071C07),
	SPH_C64(0xADAC472301AD8EAD), SPH_C64(0x5AB0B42FEA5A755A),
	SPH_C64(0x83EF1BB56C833683), SPH_C64(0x33B666FF8533CC33),
	SPH_C64(0x635CC6F23F639163), SPH_C64(0x0212040A10020802),
	SPH_C64(0xAA93493839AA92AA), SPH_C64(0x71DEE2A8AF71D971),
	SPH_C64(0xC8C68DCF0EC807C8), SPH_C64(0x19D1327DC8196419),
	SPH_C64(0x493B927072493949), SPH_C64(0xD95FAF9A86D943D9),
	SPH_C64(0xF231F91DC3F2EFF2), SPH_C64(0xE3A8DB484BE3ABE3),
	SPH_C64(0x5BB9B62AE25B715B), SPH_C64(0x88BC0D9234881A88),
	SPH_C64(0x9A3E29C8A49A529A), SPH_C64(0x260B4CBE2D269826),
	SPH_C64(0x32BF64FA8D32C832), SPH_C64(0xB0597D4AE9B0FAB0),
	SPH_C64(0xE9F2CF6A1BE983E9), SPH_C64(0x0F771E33780F3C0F),
	SPH_C64(0xD533B7A6E6D573D5), SPH_C64(0x80F41DBA74803A80),
	SPH_C64(0xBE27617C99BEC2BE), SPH_C64(0xCDEB87DE26CD13CD),
	SPH_C64(0x348968E4BD34D034), SPH_C64(0x483290757A483D48),
	SPH_C64(0xFF54E324ABFFDBFF), SPH_C64(0x7A8DF48FF77AF57A),
	SPH_C64(0x90643DEAF4907A90), SPH_C64(0x5F9DBE3EC25F615F),
	SPH_C64(0x203D40A01D208020), SPH_C64(0x680FD0D56768BD68),
	SPH_C64(0x1ACA3472D01A681A), SPH_C64(0xAEB7412C19AE82AE),
	SPH_C64(0xB47D755EC9B4EAB4), SPH_C64(0x54CEA8199A544D54),
	SPH_C64(0x937F3BE5EC937693), SPH_C64(0x222F44AA0D228822),
	SPH_C64(0x6463C8E907648D64), SPH_C64(0xF12AFF12DBF1E3F1),
	SPH_C64(0x73CCE6A2BF73D173), SPH_C64(0x1282245A90124812),
	SPH_C64(0x407A805D3A401D40), SPH_C64(0x0848102840082008),
	SPH_C64(0xC3959BE856C32BC3), SPH_C64(0xECDFC57B33EC97EC),
	SPH_C64(0xDB4DAB9096DB4BDB), SPH_C64(0xA1C05F1F61A1BEA1),
	SPH_C64(0x8D9107831C8D0E8D), SPH_C64(0x3DC87AC9F53DF43D),
	SPH_C64(0x975B33F1CC976697), SPH_C64(0x0000000000000000),
	SPH_C64(0xCFF983D436CF1BCF), SPH_C64(0x2B6E5687452BAC2B),
	SPH_C64(0x76E1ECB39776C576), SPH_C64(0x82E619B064823282),
	SPH_C64(0xD628B1A9FED67FD6), SPH_C64(0x1BC33677D81B6C1B),
	SPH_C64(0xB574775BC1B5EEB5), SPH_C64(0xAFBE432911AF86AF),
	SPH_C64(0x6A1DD4DF776AB56A), SPH_C64(0x50EAA00DBA505D50),
	SPH_C64(0x45578A4C12450945), SPH_C64(0xF338FB18CBF3EBF3),
	SPH_C64(0x30AD60F09D30C030), SPH_C64(0xEFC4C3742BEF9BEF),
	SPH_C64(0x3FDA7EC3E53FFC3F), SPH_C64(0x55C7AA1C92554955),
	SPH_C64(0xA2DB591079A2B2A2), SPH_C64(0xEAE9C96503EA8FEA),
	SPH_C64(0x656ACAEC0F658965), SPH_C64(0xBA036968B9BAD2BA),
	SPH_C64(0x2F4A5E93652FBC2F), SPH_C64(0xC08E9DE74EC027C0),
	SPH_C64(0xDE60A181BEDE5FDE), SPH_C64(0x1CFC386CE01C701C),
	SPH_C64(0xFD46E72EBBFDD3FD), SPH_C64(0x4D1F9A64524D294D),
	SPH_C64(0x927639E0E4927292), SPH_C64(0x75FAEABC8F75C975),
	SPH_C64(0x06360C1E30061806), SPH_C64(0x8AAE0998248A128A),
	SPH_C64(0xB24B7940F9B2F2B2), SPH_C64(0xE685D15963E6BFE6),
	SPH_C64(0x0E7E1C36700E380E), SPH_C64(0x1FE73E63F81F7C1F),
	SPH_C64(0x6255C4F737629562), SPH_C64(0xD43AB5A3EED477D4),
	SPH_C64(0xA8814D3229A89AA8), SPH_C64(0x965231F4C4966296),
	SPH_C64(0xF962EF3A9BF9C3F9), SPH_C64(0xC5A397F666C533C5),
	SPH_C64(0x25104AB135259425), SPH_C64(0x59ABB220F2597959),
	SPH_C64(0x84D015AE54842A84), SPH_C64(0x72C5E4A7B772D572),
	SPH_C64(0x39EC72DDD539E439), SPH_C64(0x4C1698615A4C2D4C),
	SPH_C64(0x5E94BC3BCA5E655E), SPH_C64(0x789FF085E778FD78),
	SPH_C64(0x38E570D8DD38E038), SPH_C64(0x8C980586148C0A8C),
	SPH_C64(0xD117BFB2C6D163D1), SPH_C64(0xA5E4570B41A5AEA5),
	SPH_C64(0xE2A1D94D43E2AFE2), SPH_C64(0x614EC2F82F619961),
	SPH_C64(0xB3427B45F1B3F6B3), SPH_C64(0x213442A515218421),
	SPH_C64(0x9C0825D6949C4A9C), SPH_C64(0x1EEE3C66F01E781E),
	SPH_C64(0x4361865222431143), SPH_C64(0xC7B193FC76C73BC7),
	SPH_C64(0xFC4FE52BB3FCD7FC), SPH_C64(0x0424081420041004),
	SPH_C64(0x51E3A208B2515951), SPH_C64(0x99252FC7BC995E99),
	SPH_C64(0x6D22DAC44F6DA96D), SPH_C64(0x0D651A39680D340D),
	SPH_C64(0xFA79E93583FACFFA), SPH_C64(0xDF69A384B6DF5BDF),
	SPH_C64(0x7EA9FC9BD77EE57E), SPH_C64(0x241948B43D249024),
	SPH_C64(0x3BFE76D7C53BEC3B), SPH_C64(0xAB9A4B3D31AB96AB),
	SPH_C64(0xCEF081D13ECE1FCE), SPH_C64(0x1199225588114411),
	SPH_C64(0x8F8303890C8F068F), SPH_C64(0x4E049C6B4A4E254E),
	SPH_C64(0xB7667351D1B7E6B7), SPH_C64(0xEBE0CB600BEB8BEB),
	SPH_C64(0x3CC178CCFD3CF03C), SPH_C64(0x81FD1FBF7C813E81),
	SPH_C64(0x944035FED4946A94), SPH_C64(0xF71CF30CEBF7FBF7),
	SPH_C64(0xB9186F67A1B9DEB9), SPH_C64(0x138B265F98134C13),
	SPH_C64(0x2C51589C7D2CB02C), SPH_C64(0xD305BBB8D6D36BD3),
	SPH_C64(0xE78CD35C6BE7BBE7), SPH_C64(0x6E39DCCB576EA56E),
	SPH_C64(0xC4AA95F36EC437C4), SPH_C64(0x031B060F18030C03),
	SPH_C64(0x56DCAC138A564556), SPH_C64(0x445E88491A440D44),
	SPH_C64(0x7FA0FE9EDF7FE17F), SPH_C64(0xA9884F3721A99EA9),
	SPH_C64(0x2A6754824D2AA82A), SPH_C64(0xBB0A6B6DB1BBD6BB),
	SPH_C64(0xC1879FE246C123C1), SPH_C64(0x53F1A602A2535153),
	SPH_C64(0xDC72A58BAEDC57DC), SPH_C64(0x0B531627580B2C0B),
	SPH_C64(0x9D0127D39C9D4E9D), SPH_C64(0x6C2BD8C1476CAD6C),
	SPH_C64(0x31A462F59531C431), SPH_C64(0x74F3E8B98774CD74),
	SPH_C64(0xF615F109E3F6FFF6), SPH_C64(0x464C8C430A460546),
	SPH_C64(0xACA5452609AC8AAC), SPH_C64(0x89B50F973C891E89),
	SPH_C64(0x14B42844A0145014), SPH_C64(0xE1BADF425BE1A3E1),
	SPH_C64(0x16A62C4EB0165816), SPH_C64(0x3AF774D2CD3AE83A),
	SPH_C64(0x6906D2D06F69B969), SPH_C64(0x0941122D48092409),
	SPH_C64(0x70D7E0ADA770DD70), SPH_C64(0xB66F7154D9B6E2B6),
	SPH_C64(0xD01EBDB7CED067D0), SPH_C64(0xEDD6C77E3BED93ED),
	SPH_C64(0xCCE285DB2ECC17CC), SPH_C64(0x426884572A421542),
	SPH_C64(0x982C2DC2B4985A98), SPH_C64(0xA4ED550E49A4AAA4),
	SPH_C64(0x287550885D28A028), SPH_C64(0x5C86B831DA5C6D5C),
	SPH_C64(0xF86BED3F93F8C7F8), SPH_C64(0x86C211A444862286)
};

#endif

/*
 * Round constants.
 */
static const sph_u64 plain_RC[10] = {
	SPH_C64(0x4F01B887E8C62318),
	SPH_C64(0x52916F79F5D2A636),
	SPH_C64(0x357B0CA38E9BBC60),
	SPH_C64(0x57FE4B2EC2D7E01D),
	SPH_C64(0xDA4AF09FE5377715),
	SPH_C64(0x856BA0B10A29C958),
	SPH_C64(0x67053ECBF4105DBD),
	SPH_C64(0xD8957DA78B4127E4),
	SPH_C64(0x9E4717DD667CEEFB),
	SPH_C64(0x33835AAD07BF2DCA)
};

/* ====================================================================== */
/*
 * Constants for plain WHIRLPOOL-0 (first version).
 */

static const sph_u64 old0_T0[256] = {
	SPH_C64(0xD50F67D568B86868), SPH_C64(0xB71ECEB7D06DD0D0),
	SPH_C64(0x60E00B60EB20EBEB), SPH_C64(0x876E45872B7D2B2B),
	SPH_C64(0x75327A7548D84848), SPH_C64(0xD3019CD39DBA9D9D),
	SPH_C64(0xDF1D77DF6ABE6A6A), SPH_C64(0x53977353E431E4E4),
	SPH_C64(0x48A84B48E338E3E3), SPH_C64(0x15D27115A3F8A3A3),
	SPH_C64(0x13DC8A1356FA5656), SPH_C64(0xBFFD7CBF819E8181),
	SPH_C64(0x94B2CF947D877D7D), SPH_C64(0x122ADB12F10EF1F1),
	SPH_C64(0xABD95CAB85928585), SPH_C64(0xDC1A84DC9EBF9E9E),
	SPH_C64(0x9C517D9C2C742C2C), SPH_C64(0x8C8A048C8E8F8E8E),
	SPH_C64(0x859FE78578887878), SPH_C64(0xC5D41EC5CA43CACA),
	SPH_C64(0x4BAFB84B17391717), SPH_C64(0x37882137A9E6A9A9),
	SPH_C64(0xF84E2FF861A36161), SPH_C64(0xA633E6A6D562D5D5),
	SPH_C64(0x348FD2345DE75D5D), SPH_C64(0x275358270B1D0B0B),
	SPH_C64(0x869814868C898C8C), SPH_C64(0xCCC1FDCC3C443C3C),
	SPH_C64(0xB6E89FB677997777), SPH_C64(0x08E3B20851F35151),
	SPH_C64(0xAA2F0DAA22662222), SPH_C64(0x57682A5742C64242),
	SPH_C64(0xC3DAE5C33F413F3F), SPH_C64(0x19CE9A1954FC5454),
	SPH_C64(0x5873325841C34141), SPH_C64(0xBAF474BA809D8080),
	SPH_C64(0xDBE22EDBCC49CCCC), SPH_C64(0xA4C244A486978686),
	SPH_C64(0x4542F145B3C8B3B3), SPH_C64(0x78D8C07818281818),
	SPH_C64(0x96436D962E722E2E), SPH_C64(0x16D5821657F95757),
	SPH_C64(0x1E36301E060A0606), SPH_C64(0xF75537F762A66262),
	SPH_C64(0x0307F303F401F4F4), SPH_C64(0xEE9BADEE365A3636),
	SPH_C64(0xB217C6B2D16ED1D1), SPH_C64(0xDA147FDA6BBD6B6B),
	SPH_C64(0x77C3D8771B2D1B1B), SPH_C64(0xEC6A0FEC65AF6565),
	SPH_C64(0xBCFA8FBC759F7575), SPH_C64(0x5090805010301010),
	SPH_C64(0x95449E95DA73DADA), SPH_C64(0x703B727049DB4949),
	SPH_C64(0xBE0B2DBE266A2626), SPH_C64(0x3A629B3AF916F9F9),
	SPH_C64(0xC0DD16C0CB40CBCB), SPH_C64(0xE37117E366AA6666),
	SPH_C64(0x5C8C6B5CE734E7E7), SPH_C64(0x6803B968BAD3BABA),
	SPH_C64(0x2CB7192CAEEFAEAE), SPH_C64(0x0DEABA0D50F05050),
	SPH_C64(0x07F8AA0752F65252), SPH_C64(0x3D9A313DABE0ABAB),
	SPH_C64(0x112D2811050F0505), SPH_C64(0x1723D317F00DF0F0),
	SPH_C64(0x396568390D170D0D), SPH_C64(0xA2CCBFA273957373),
	SPH_C64(0xD7FEC5D73B4D3B3B), SPH_C64(0x14242014040C0404),
	SPH_C64(0xA03D1DA020602020), SPH_C64(0x215DA321FE1FFEFE),
	SPH_C64(0x8E7BA68EDD7ADDDD), SPH_C64(0x060EFB06F502F5F5),
	SPH_C64(0x5E7DC95EB4C1B4B4), SPH_C64(0x3E9DC23E5FE15F5F),
	SPH_C64(0x225A50220A1E0A0A), SPH_C64(0x5B74C15BB5C2B5B5),
	SPH_C64(0xE78E4EE7C05DC0C0), SPH_C64(0x1AC9691AA0FDA0A0),
	SPH_C64(0xA8DEAFA871937171), SPH_C64(0x0BE4410BA5F2A5A5),
	SPH_C64(0x995875992D772D2D), SPH_C64(0xFD4727FD60A06060),
	SPH_C64(0xA7C5B7A772967272), SPH_C64(0xE57FECE593A89393),
	SPH_C64(0xDDECD5DD394B3939), SPH_C64(0x2848402808180808),
	SPH_C64(0xB5EF6CB583988383), SPH_C64(0xA53415A521632121),
	SPH_C64(0x3186DA315CE45C5C), SPH_C64(0xA1CB4CA187948787),
	SPH_C64(0x4F50E14FB1CEB1B1), SPH_C64(0x47B35347E03DE0E0),
	SPH_C64(0x0000000000000000), SPH_C64(0xE89556E8C358C3C3),
	SPH_C64(0x5A82905A12361212), SPH_C64(0xEF6DFCEF91AE9191),
	SPH_C64(0x98AE24988A838A8A), SPH_C64(0x0A12100A02060202),
	SPH_C64(0x6CFCE06C1C241C1C), SPH_C64(0x59856359E637E6E6),
	SPH_C64(0x4C57124C45CF4545), SPH_C64(0xED9C5EEDC25BC2C2),
	SPH_C64(0xF3AA6EF3C451C4C4), SPH_C64(0x2E46BB2EFD1AFDFD),
	SPH_C64(0x792E9179BFDCBFBF), SPH_C64(0x495E1A4944CC4444),
	SPH_C64(0x1FC0611FA1FEA1A1), SPH_C64(0x61165A614CD44C4C),
	SPH_C64(0xFFB685FF33553333), SPH_C64(0xF6A366F6C552C5C5),
	SPH_C64(0xAED054AE84918484), SPH_C64(0xAF2605AF23652323),
	SPH_C64(0x91BBC7917C847C7C), SPH_C64(0x4A59E94AB0CDB0B0),
	SPH_C64(0xB11035B1256F2525), SPH_C64(0x41BDA841153F1515),
	SPH_C64(0xE180B5E1355F3535), SPH_C64(0xD0066FD069BB6969),
	SPH_C64(0x2454AB24FF1CFFFF), SPH_C64(0xFE40D4FE94A19494),
	SPH_C64(0x641F52644DD74D4D), SPH_C64(0xADD7A7AD70907070),
	SPH_C64(0x10DB7910A2FBA2A2), SPH_C64(0x29BE1129AFECAFAF),
	SPH_C64(0xDEEB26DECD4ACDCD), SPH_C64(0xA928FEA9D667D6D6),
	SPH_C64(0xC12B47C16CB46C6C), SPH_C64(0x5166D151B7C4B7B7),
	SPH_C64(0x3F6B933FF815F8F8), SPH_C64(0x2D41482D091B0909),
	SPH_C64(0x1838CB18F308F3F3), SPH_C64(0xE6781FE667A96767),
	SPH_C64(0x0EED490EA4F1A4A4), SPH_C64(0x65E90365EA23EAEA),
	SPH_C64(0x7BDF337BEC29ECEC), SPH_C64(0x546FD954B6C7B6B6),
	SPH_C64(0xA33AEEA3D461D4D4), SPH_C64(0xBD0CDEBDD26BD2D2),
	SPH_C64(0x44B4A044143C1414), SPH_C64(0x66EEF0661E221E1E),
	SPH_C64(0x42BA5B42E13EE1E1), SPH_C64(0xB4193DB4246C2424),
	SPH_C64(0xD8E5DDD838483838), SPH_C64(0xF9B87EF9C657C6C6),
	SPH_C64(0x904D9690DB70DBDB), SPH_C64(0x7A29627A4BDD4B4B),
	SPH_C64(0x8F8DF78F7A8E7A7A), SPH_C64(0xD2F7CDD23A4E3A3A),
	SPH_C64(0x8160BE81DE7FDEDE), SPH_C64(0x3B94CA3B5EE25E5E),
	SPH_C64(0x8469B684DF7CDFDF), SPH_C64(0xFB49DCFB95A29595),
	SPH_C64(0x2B4FB32BFC19FCFC), SPH_C64(0x38933938AAE3AAAA),
	SPH_C64(0xAC21F6ACD764D7D7), SPH_C64(0xD1F03ED1CE4FCECE),
	SPH_C64(0x1B3F381B07090707), SPH_C64(0x337778330F110F0F),
	SPH_C64(0xC9C8F5C93D473D3D), SPH_C64(0x25A2FA2558E85858),
	SPH_C64(0xC83EA4C89AB39A9A), SPH_C64(0xC22CB4C298B59898),
	SPH_C64(0xD60894D69CB99C9C), SPH_C64(0x1D31C31DF20BF2F2),
	SPH_C64(0x01F65101A7F4A7A7), SPH_C64(0x5599885511331111),
	SPH_C64(0x9BA9D79B7E827E7E), SPH_C64(0x9DA72C9D8B808B8B),
	SPH_C64(0x5261225243C54343), SPH_C64(0x0F1B180F03050303),
	SPH_C64(0x4DA1434DE23BE2E2), SPH_C64(0x8B72AE8BDC79DCDC),
	SPH_C64(0x569E7B56E532E5E5), SPH_C64(0x404BF940B2CBB2B2),
	SPH_C64(0x6B044A6B4ED24E4E), SPH_C64(0xFCB176FCC754C7C7),
	SPH_C64(0xC4224FC46DB76D6D), SPH_C64(0x6AF21B6AE926E9E9),
	SPH_C64(0xBB0225BB27692727), SPH_C64(0x5D7A3A5D40C04040),
	SPH_C64(0x9F568E9FD875D8D8), SPH_C64(0xEB92A5EB37593737),
	SPH_C64(0xE076E4E092AB9292), SPH_C64(0x89830C898F8C8F8F),
	SPH_C64(0x0509080501030101), SPH_C64(0x69F5E8691D271D1D),
	SPH_C64(0x02F1A20253F55353), SPH_C64(0xC6D3EDC63E423E3E),
	SPH_C64(0x20ABF22059EB5959), SPH_C64(0xE28746E2C15EC1C1),
	SPH_C64(0x6E0D426E4FD14F4F), SPH_C64(0xFABF8DFA32563232),
	SPH_C64(0x4EA6B04E163A1616), SPH_C64(0x35798335FA13FAFA),
	SPH_C64(0xB9F387B9749C7474), SPH_C64(0x30708B30FB10FBFB),
	SPH_C64(0xF25C3FF263A56363), SPH_C64(0xD9138CD99FBC9F9F),
	SPH_C64(0xE489BDE4345C3434), SPH_C64(0x72CAD0721A2E1A1A),
	SPH_C64(0x82674D822A7E2A2A), SPH_C64(0x2FB0EA2F5AEE5A5A),
	SPH_C64(0x83911C838D8A8D8D), SPH_C64(0xCACF06CAC946C9C9),
	SPH_C64(0xD4F936D4CF4CCFCF), SPH_C64(0x0915E309F607F6F6),
	SPH_C64(0xEA64F4EA90AD9090), SPH_C64(0x88755D8828782828),
	SPH_C64(0x92BC349288858888), SPH_C64(0xCD37ACCD9BB09B9B),
	SPH_C64(0xF5A495F531533131), SPH_C64(0x367E70360E120E0E),
	SPH_C64(0x733C8173BDDABDBD), SPH_C64(0x7F206A7F4ADE4A4A),
	SPH_C64(0x6FFB136FE825E8E8), SPH_C64(0xF452C4F496A79696),
	SPH_C64(0x04FF5904A6F7A6A6), SPH_C64(0x3C6C603C0C140C0C),
	SPH_C64(0xCFC60ECFC845C8C8), SPH_C64(0x8096EF80798B7979),
	SPH_C64(0x76358976BCD9BCBC), SPH_C64(0x7C27997CBEDFBEBE),
	SPH_C64(0x74C42B74EF2CEFEF), SPH_C64(0xCB3957CB6EB26E6E),
	SPH_C64(0x434C0A4346CA4646), SPH_C64(0xF15BCCF197A49797),
	SPH_C64(0x2AB9E22A5BED5B5B), SPH_C64(0x7ED63B7EED2AEDED),
	SPH_C64(0x7DD1C87D192B1919), SPH_C64(0x9A5F869AD976D9D9),
	SPH_C64(0x26A50926ACE9ACAC), SPH_C64(0xC725BCC799B69999),
	SPH_C64(0x32812932A8E5A8A8), SPH_C64(0x8D7C558D297B2929),
	SPH_C64(0xE96307E964AC6464), SPH_C64(0x63E7F8631F211F1F),
	SPH_C64(0x23AC0123ADEAADAD), SPH_C64(0x1CC7921C55FF5555),
	SPH_C64(0x5F8B985F13351313), SPH_C64(0x6D0AB16DBBD0BBBB),
	SPH_C64(0x0C1CEB0CF704F7F7), SPH_C64(0xCE305FCE6FB16F6F),
	SPH_C64(0x6718A167B9D6B9B9), SPH_C64(0x4645024647C94747),
	SPH_C64(0x934A65932F712F2F), SPH_C64(0x71CD2371EE2FEEEE),
	SPH_C64(0x6211A962B8D5B8B8), SPH_C64(0x8A84FF8A7B8D7B7B),
	SPH_C64(0x97B53C9789868989), SPH_C64(0xF0AD9DF030503030),
	SPH_C64(0xB805D6B8D368D3D3), SPH_C64(0x9EA0DF9E7F817F7F),
	SPH_C64(0xB3E197B3769A7676), SPH_C64(0xB0E664B0829B8282)
};

#if !SPH_SMALL_FOOTPRINT_WHIRLPOOL

static const sph_u64 old0_T1[256] = {
	SPH_C64(0x0F67D568B86868D5), SPH_C64(0x1ECEB7D06DD0D0B7),
	SPH_C64(0xE00B60EB20EBEB60), SPH_C64(0x6E45872B7D2B2B87),
	SPH_C64(0x327A7548D8484875), SPH_C64(0x019CD39DBA9D9DD3),
	SPH_C64(0x1D77DF6ABE6A6ADF), SPH_C64(0x977353E431E4E453),
	SPH_C64(0xA84B48E338E3E348), SPH_C64(0xD27115A3F8A3A315),
	SPH_C64(0xDC8A1356FA565613), SPH_C64(0xFD7CBF819E8181BF),
	SPH_C64(0xB2CF947D877D7D94), SPH_C64(0x2ADB12F10EF1F112),
	SPH_C64(0xD95CAB85928585AB), SPH_C64(0x1A84DC9EBF9E9EDC),
	SPH_C64(0x517D9C2C742C2C9C), SPH_C64(0x8A048C8E8F8E8E8C),
	SPH_C64(0x9FE7857888787885), SPH_C64(0xD41EC5CA43CACAC5),
	SPH_C64(0xAFB84B173917174B), SPH_C64(0x882137A9E6A9A937),
	SPH_C64(0x4E2FF861A36161F8), SPH_C64(0x33E6A6D562D5D5A6),
	SPH_C64(0x8FD2345DE75D5D34), SPH_C64(0x5358270B1D0B0B27),
	SPH_C64(0x9814868C898C8C86), SPH_C64(0xC1FDCC3C443C3CCC),
	SPH_C64(0xE89FB677997777B6), SPH_C64(0xE3B20851F3515108),
	SPH_C64(0x2F0DAA22662222AA), SPH_C64(0x682A5742C6424257),
	SPH_C64(0xDAE5C33F413F3FC3), SPH_C64(0xCE9A1954FC545419),
	SPH_C64(0x73325841C3414158), SPH_C64(0xF474BA809D8080BA),
	SPH_C64(0xE22EDBCC49CCCCDB), SPH_C64(0xC244A486978686A4),
	SPH_C64(0x42F145B3C8B3B345), SPH_C64(0xD8C0781828181878),
	SPH_C64(0x436D962E722E2E96), SPH_C64(0xD5821657F9575716),
	SPH_C64(0x36301E060A06061E), SPH_C64(0x5537F762A66262F7),
	SPH_C64(0x07F303F401F4F403), SPH_C64(0x9BADEE365A3636EE),
	SPH_C64(0x17C6B2D16ED1D1B2), SPH_C64(0x147FDA6BBD6B6BDA),
	SPH_C64(0xC3D8771B2D1B1B77), SPH_C64(0x6A0FEC65AF6565EC),
	SPH_C64(0xFA8FBC759F7575BC), SPH_C64(0x9080501030101050),
	SPH_C64(0x449E95DA73DADA95), SPH_C64(0x3B727049DB494970),
	SPH_C64(0x0B2DBE266A2626BE), SPH_C64(0x629B3AF916F9F93A),
	SPH_C64(0xDD16C0CB40CBCBC0), SPH_C64(0x7117E366AA6666E3),
	SPH_C64(0x8C6B5CE734E7E75C), SPH_C64(0x03B968BAD3BABA68),
	SPH_C64(0xB7192CAEEFAEAE2C), SPH_C64(0xEABA0D50F050500D),
	SPH_C64(0xF8AA0752F6525207), SPH_C64(0x9A313DABE0ABAB3D),
	SPH_C64(0x2D2811050F050511), SPH_C64(0x23D317F00DF0F017),
	SPH_C64(0x6568390D170D0D39), SPH_C64(0xCCBFA273957373A2),
	SPH_C64(0xFEC5D73B4D3B3BD7), SPH_C64(0x242014040C040414),
	SPH_C64(0x3D1DA020602020A0), SPH_C64(0x5DA321FE1FFEFE21),
	SPH_C64(0x7BA68EDD7ADDDD8E), SPH_C64(0x0EFB06F502F5F506),
	SPH_C64(0x7DC95EB4C1B4B45E), SPH_C64(0x9DC23E5FE15F5F3E),
	SPH_C64(0x5A50220A1E0A0A22), SPH_C64(0x74C15BB5C2B5B55B),
	SPH_C64(0x8E4EE7C05DC0C0E7), SPH_C64(0xC9691AA0FDA0A01A),
	SPH_C64(0xDEAFA871937171A8), SPH_C64(0xE4410BA5F2A5A50B),
	SPH_C64(0x5875992D772D2D99), SPH_C64(0x4727FD60A06060FD),
	SPH_C64(0xC5B7A772967272A7), SPH_C64(0x7FECE593A89393E5),
	SPH_C64(0xECD5DD394B3939DD), SPH_C64(0x4840280818080828),
	SPH_C64(0xEF6CB583988383B5), SPH_C64(0x3415A521632121A5),
	SPH_C64(0x86DA315CE45C5C31), SPH_C64(0xCB4CA187948787A1),
	SPH_C64(0x50E14FB1CEB1B14F), SPH_C64(0xB35347E03DE0E047),
	SPH_C64(0x0000000000000000), SPH_C64(0x9556E8C358C3C3E8),
	SPH_C64(0x82905A123612125A), SPH_C64(0x6DFCEF91AE9191EF),
	SPH_C64(0xAE24988A838A8A98), SPH_C64(0x12100A020602020A),
	SPH_C64(0xFCE06C1C241C1C6C), SPH_C64(0x856359E637E6E659),
	SPH_C64(0x57124C45CF45454C), SPH_C64(0x9C5EEDC25BC2C2ED),
	SPH_C64(0xAA6EF3C451C4C4F3), SPH_C64(0x46BB2EFD1AFDFD2E),
	SPH_C64(0x2E9179BFDCBFBF79), SPH_C64(0x5E1A4944CC444449),
	SPH_C64(0xC0611FA1FEA1A11F), SPH_C64(0x165A614CD44C4C61),
	SPH_C64(0xB685FF33553333FF), SPH_C64(0xA366F6C552C5C5F6),
	SPH_C64(0xD054AE84918484AE), SPH_C64(0x2605AF23652323AF),
	SPH_C64(0xBBC7917C847C7C91), SPH_C64(0x59E94AB0CDB0B04A),
	SPH_C64(0x1035B1256F2525B1), SPH_C64(0xBDA841153F151541),
	SPH_C64(0x80B5E1355F3535E1), SPH_C64(0x066FD069BB6969D0),
	SPH_C64(0x54AB24FF1CFFFF24), SPH_C64(0x40D4FE94A19494FE),
	SPH_C64(0x1F52644DD74D4D64), SPH_C64(0xD7A7AD70907070AD),
	SPH_C64(0xDB7910A2FBA2A210), SPH_C64(0xBE1129AFECAFAF29),
	SPH_C64(0xEB26DECD4ACDCDDE), SPH_C64(0x28FEA9D667D6D6A9),
	SPH_C64(0x2B47C16CB46C6CC1), SPH_C64(0x66D151B7C4B7B751),
	SPH_C64(0x6B933FF815F8F83F), SPH_C64(0x41482D091B09092D),
	SPH_C64(0x38CB18F308F3F318), SPH_C64(0x781FE667A96767E6),
	SPH_C64(0xED490EA4F1A4A40E), SPH_C64(0xE90365EA23EAEA65),
	SPH_C64(0xDF337BEC29ECEC7B), SPH_C64(0x6FD954B6C7B6B654),
	SPH_C64(0x3AEEA3D461D4D4A3), SPH_C64(0x0CDEBDD26BD2D2BD),
	SPH_C64(0xB4A044143C141444), SPH_C64(0xEEF0661E221E1E66),
	SPH_C64(0xBA5B42E13EE1E142), SPH_C64(0x193DB4246C2424B4),
	SPH_C64(0xE5DDD838483838D8), SPH_C64(0xB87EF9C657C6C6F9),
	SPH_C64(0x4D9690DB70DBDB90), SPH_C64(0x29627A4BDD4B4B7A),
	SPH_C64(0x8DF78F7A8E7A7A8F), SPH_C64(0xF7CDD23A4E3A3AD2),
	SPH_C64(0x60BE81DE7FDEDE81), SPH_C64(0x94CA3B5EE25E5E3B),
	SPH_C64(0x69B684DF7CDFDF84), SPH_C64(0x49DCFB95A29595FB),
	SPH_C64(0x4FB32BFC19FCFC2B), SPH_C64(0x933938AAE3AAAA38),
	SPH_C64(0x21F6ACD764D7D7AC), SPH_C64(0xF03ED1CE4FCECED1),
	SPH_C64(0x3F381B070907071B), SPH_C64(0x7778330F110F0F33),
	SPH_C64(0xC8F5C93D473D3DC9), SPH_C64(0xA2FA2558E8585825),
	SPH_C64(0x3EA4C89AB39A9AC8), SPH_C64(0x2CB4C298B59898C2),
	SPH_C64(0x0894D69CB99C9CD6), SPH_C64(0x31C31DF20BF2F21D),
	SPH_C64(0xF65101A7F4A7A701), SPH_C64(0x9988551133111155),
	SPH_C64(0xA9D79B7E827E7E9B), SPH_C64(0xA72C9D8B808B8B9D),
	SPH_C64(0x61225243C5434352), SPH_C64(0x1B180F030503030F),
	SPH_C64(0xA1434DE23BE2E24D), SPH_C64(0x72AE8BDC79DCDC8B),
	SPH_C64(0x9E7B56E532E5E556), SPH_C64(0x4BF940B2CBB2B240),
	SPH_C64(0x044A6B4ED24E4E6B), SPH_C64(0xB176FCC754C7C7FC),
	SPH_C64(0x224FC46DB76D6DC4), SPH_C64(0xF21B6AE926E9E96A),
	SPH_C64(0x0225BB27692727BB), SPH_C64(0x7A3A5D40C040405D),
	SPH_C64(0x568E9FD875D8D89F), SPH_C64(0x92A5EB37593737EB),
	SPH_C64(0x76E4E092AB9292E0), SPH_C64(0x830C898F8C8F8F89),
	SPH_C64(0x0908050103010105), SPH_C64(0xF5E8691D271D1D69),
	SPH_C64(0xF1A20253F5535302), SPH_C64(0xD3EDC63E423E3EC6),
	SPH_C64(0xABF22059EB595920), SPH_C64(0x8746E2C15EC1C1E2),
	SPH_C64(0x0D426E4FD14F4F6E), SPH_C64(0xBF8DFA32563232FA),
	SPH_C64(0xA6B04E163A16164E), SPH_C64(0x798335FA13FAFA35),
	SPH_C64(0xF387B9749C7474B9), SPH_C64(0x708B30FB10FBFB30),
	SPH_C64(0x5C3FF263A56363F2), SPH_C64(0x138CD99FBC9F9FD9),
	SPH_C64(0x89BDE4345C3434E4), SPH_C64(0xCAD0721A2E1A1A72),
	SPH_C64(0x674D822A7E2A2A82), SPH_C64(0xB0EA2F5AEE5A5A2F),
	SPH_C64(0x911C838D8A8D8D83), SPH_C64(0xCF06CAC946C9C9CA),
	SPH_C64(0xF936D4CF4CCFCFD4), SPH_C64(0x15E309F607F6F609),
	SPH_C64(0x64F4EA90AD9090EA), SPH_C64(0x755D882878282888),
	SPH_C64(0xBC34928885888892), SPH_C64(0x37ACCD9BB09B9BCD),
	SPH_C64(0xA495F531533131F5), SPH_C64(0x7E70360E120E0E36),
	SPH_C64(0x3C8173BDDABDBD73), SPH_C64(0x206A7F4ADE4A4A7F),
	SPH_C64(0xFB136FE825E8E86F), SPH_C64(0x52C4F496A79696F4),
	SPH_C64(0xFF5904A6F7A6A604), SPH_C64(0x6C603C0C140C0C3C),
	SPH_C64(0xC60ECFC845C8C8CF), SPH_C64(0x96EF80798B797980),
	SPH_C64(0x358976BCD9BCBC76), SPH_C64(0x27997CBEDFBEBE7C),
	SPH_C64(0xC42B74EF2CEFEF74), SPH_C64(0x3957CB6EB26E6ECB),
	SPH_C64(0x4C0A4346CA464643), SPH_C64(0x5BCCF197A49797F1),
	SPH_C64(0xB9E22A5BED5B5B2A), SPH_C64(0xD63B7EED2AEDED7E),
	SPH_C64(0xD1C87D192B19197D), SPH_C64(0x5F869AD976D9D99A),
	SPH_C64(0xA50926ACE9ACAC26), SPH_C64(0x25BCC799B69999C7),
	SPH_C64(0x812932A8E5A8A832), SPH_C64(0x7C558D297B29298D),
	SPH_C64(0x6307E964AC6464E9), SPH_C64(0xE7F8631F211F1F63),
	SPH_C64(0xAC0123ADEAADAD23), SPH_C64(0xC7921C55FF55551C),
	SPH_C64(0x8B985F133513135F), SPH_C64(0x0AB16DBBD0BBBB6D),
	SPH_C64(0x1CEB0CF704F7F70C), SPH_C64(0x305FCE6FB16F6FCE),
	SPH_C64(0x18A167B9D6B9B967), SPH_C64(0x45024647C9474746),
	SPH_C64(0x4A65932F712F2F93), SPH_C64(0xCD2371EE2FEEEE71),
	SPH_C64(0x11A962B8D5B8B862), SPH_C64(0x84FF8A7B8D7B7B8A),
	SPH_C64(0xB53C978986898997), SPH_C64(0xAD9DF030503030F0),
	SPH_C64(0x05D6B8D368D3D3B8), SPH_C64(0xA0DF9E7F817F7F9E),
	SPH_C64(0xE197B3769A7676B3), SPH_C64(0xE664B0829B8282B0)
};

static const sph_u64 old0_T2[256] = {
	SPH_C64(0x67D568B86868D50F), SPH_C64(0xCEB7D06DD0D0B71E),
	SPH_C64(0x0B60EB20EBEB60E0), SPH_C64(0x45872B7D2B2B876E),
	SPH_C64(0x7A7548D848487532), SPH_C64(0x9CD39DBA9D9DD301),
	SPH_C64(0x77DF6ABE6A6ADF1D), SPH_C64(0x7353E431E4E45397),
	SPH_C64(0x4B48E338E3E348A8), SPH_C64(0x7115A3F8A3A315D2),
	SPH_C64(0x8A1356FA565613DC), SPH_C64(0x7CBF819E8181BFFD),
	SPH_C64(0xCF947D877D7D94B2), SPH_C64(0xDB12F10EF1F1122A),
	SPH_C64(0x5CAB85928585ABD9), SPH_C64(0x84DC9EBF9E9EDC1A),
	SPH_C64(0x7D9C2C742C2C9C51), SPH_C64(0x048C8E8F8E8E8C8A),
	SPH_C64(0xE78578887878859F), SPH_C64(0x1EC5CA43CACAC5D4),
	SPH_C64(0xB84B173917174BAF), SPH_C64(0x2137A9E6A9A93788),
	SPH_C64(0x2FF861A36161F84E), SPH_C64(0xE6A6D562D5D5A633),
	SPH_C64(0xD2345DE75D5D348F), SPH_C64(0x58270B1D0B0B2753),
	SPH_C64(0x14868C898C8C8698), SPH_C64(0xFDCC3C443C3CCCC1),
	SPH_C64(0x9FB677997777B6E8), SPH_C64(0xB20851F3515108E3),
	SPH_C64(0x0DAA22662222AA2F), SPH_C64(0x2A5742C642425768),
	SPH_C64(0xE5C33F413F3FC3DA), SPH_C64(0x9A1954FC545419CE),
	SPH_C64(0x325841C341415873), SPH_C64(0x74BA809D8080BAF4),
	SPH_C64(0x2EDBCC49CCCCDBE2), SPH_C64(0x44A486978686A4C2),
	SPH_C64(0xF145B3C8B3B34542), SPH_C64(0xC0781828181878D8),
	SPH_C64(0x6D962E722E2E9643), SPH_C64(0x821657F9575716D5),
	SPH_C64(0x301E060A06061E36), SPH_C64(0x37F762A66262F755),
	SPH_C64(0xF303F401F4F40307), SPH_C64(0xADEE365A3636EE9B),
	SPH_C64(0xC6B2D16ED1D1B217), SPH_C64(0x7FDA6BBD6B6BDA14),
	SPH_C64(0xD8771B2D1B1B77C3), SPH_C64(0x0FEC65AF6565EC6A),
	SPH_C64(0x8FBC759F7575BCFA), SPH_C64(0x8050103010105090),
	SPH_C64(0x9E95DA73DADA9544), SPH_C64(0x727049DB4949703B),
	SPH_C64(0x2DBE266A2626BE0B), SPH_C64(0x9B3AF916F9F93A62),
	SPH_C64(0x16C0CB40CBCBC0DD), SPH_C64(0x17E366AA6666E371),
	SPH_C64(0x6B5CE734E7E75C8C), SPH_C64(0xB968BAD3BABA6803),
	SPH_C64(0x192CAEEFAEAE2CB7), SPH_C64(0xBA0D50F050500DEA),
	SPH_C64(0xAA0752F6525207F8), SPH_C64(0x313DABE0ABAB3D9A),
	SPH_C64(0x2811050F0505112D), SPH_C64(0xD317F00DF0F01723),
	SPH_C64(0x68390D170D0D3965), SPH_C64(0xBFA273957373A2CC),
	SPH_C64(0xC5D73B4D3B3BD7FE), SPH_C64(0x2014040C04041424),
	SPH_C64(0x1DA020602020A03D), SPH_C64(0xA321FE1FFEFE215D),
	SPH_C64(0xA68EDD7ADDDD8E7B), SPH_C64(0xFB06F502F5F5060E),
	SPH_C64(0xC95EB4C1B4B45E7D), SPH_C64(0xC23E5FE15F5F3E9D),
	SPH_C64(0x50220A1E0A0A225A), SPH_C64(0xC15BB5C2B5B55B74),
	SPH_C64(0x4EE7C05DC0C0E78E), SPH_C64(0x691AA0FDA0A01AC9),
	SPH_C64(0xAFA871937171A8DE), SPH_C64(0x410BA5F2A5A50BE4),
	SPH_C64(0x75992D772D2D9958), SPH_C64(0x27FD60A06060FD47),
	SPH_C64(0xB7A772967272A7C5), SPH_C64(0xECE593A89393E57F),
	SPH_C64(0xD5DD394B3939DDEC), SPH_C64(0x4028081808082848),
	SPH_C64(0x6CB583988383B5EF), SPH_C64(0x15A521632121A534),
	SPH_C64(0xDA315CE45C5C3186), SPH_C64(0x4CA187948787A1CB),
	SPH_C64(0xE14FB1CEB1B14F50), SPH_C64(0x5347E03DE0E047B3),
	SPH_C64(0x0000000000000000), SPH_C64(0x56E8C358C3C3E895),
	SPH_C64(0x905A123612125A82), SPH_C64(0xFCEF91AE9191EF6D),
	SPH_C64(0x24988A838A8A98AE), SPH_C64(0x100A020602020A12),
	SPH_C64(0xE06C1C241C1C6CFC), SPH_C64(0x6359E637E6E65985),
	SPH_C64(0x124C45CF45454C57), SPH_C64(0x5EEDC25BC2C2ED9C),
	SPH_C64(0x6EF3C451C4C4F3AA), SPH_C64(0xBB2EFD1AFDFD2E46),
	SPH_C64(0x9179BFDCBFBF792E), SPH_C64(0x1A4944CC4444495E),
	SPH_C64(0x611FA1FEA1A11FC0), SPH_C64(0x5A614CD44C4C6116),
	SPH_C64(0x85FF33553333FFB6), SPH_C64(0x66F6C552C5C5F6A3),
	SPH_C64(0x54AE84918484AED0), SPH_C64(0x05AF23652323AF26),
	SPH_C64(0xC7917C847C7C91BB), SPH_C64(0xE94AB0CDB0B04A59),
	SPH_C64(0x35B1256F2525B110), SPH_C64(0xA841153F151541BD),
	SPH_C64(0xB5E1355F3535E180), SPH_C64(0x6FD069BB6969D006),
	SPH_C64(0xAB24FF1CFFFF2454), SPH_C64(0xD4FE94A19494FE40),
	SPH_C64(0x52644DD74D4D641F), SPH_C64(0xA7AD70907070ADD7),
	SPH_C64(0x7910A2FBA2A210DB), SPH_C64(0x1129AFECAFAF29BE),
	SPH_C64(0x26DECD4ACDCDDEEB), SPH_C64(0xFEA9D667D6D6A928),
	SPH_C64(0x47C16CB46C6CC12B), SPH_C64(0xD151B7C4B7B75166),
	SPH_C64(0x933FF815F8F83F6B), SPH_C64(0x482D091B09092D41),
	SPH_C64(0xCB18F308F3F31838), SPH_C64(0x1FE667A96767E678),
	SPH_C64(0x490EA4F1A4A40EED), SPH_C64(0x0365EA23EAEA65E9),
	SPH_C64(0x337BEC29ECEC7BDF), SPH_C64(0xD954B6C7B6B6546F),
	SPH_C64(0xEEA3D461D4D4A33A), SPH_C64(0xDEBDD26BD2D2BD0C),
	SPH_C64(0xA044143C141444B4), SPH_C64(0xF0661E221E1E66EE),
	SPH_C64(0x5B42E13EE1E142BA), SPH_C64(0x3DB4246C2424B419),
	SPH_C64(0xDDD838483838D8E5), SPH_C64(0x7EF9C657C6C6F9B8),
	SPH_C64(0x9690DB70DBDB904D), SPH_C64(0x627A4BDD4B4B7A29),
	SPH_C64(0xF78F7A8E7A7A8F8D), SPH_C64(0xCDD23A4E3A3AD2F7),
	SPH_C64(0xBE81DE7FDEDE8160), SPH_C64(0xCA3B5EE25E5E3B94),
	SPH_C64(0xB684DF7CDFDF8469), SPH_C64(0xDCFB95A29595FB49),
	SPH_C64(0xB32BFC19FCFC2B4F), SPH_C64(0x3938AAE3AAAA3893),
	SPH_C64(0xF6ACD764D7D7AC21), SPH_C64(0x3ED1CE4FCECED1F0),
	SPH_C64(0x381B070907071B3F), SPH_C64(0x78330F110F0F3377),
	SPH_C64(0xF5C93D473D3DC9C8), SPH_C64(0xFA2558E8585825A2),
	SPH_C64(0xA4C89AB39A9AC83E), SPH_C64(0xB4C298B59898C22C),
	SPH_C64(0x94D69CB99C9CD608), SPH_C64(0xC31DF20BF2F21D31),
	SPH_C64(0x5101A7F4A7A701F6), SPH_C64(0x8855113311115599),
	SPH_C64(0xD79B7E827E7E9BA9), SPH_C64(0x2C9D8B808B8B9DA7),
	SPH_C64(0x225243C543435261), SPH_C64(0x180F030503030F1B),
	SPH_C64(0x434DE23BE2E24DA1), SPH_C64(0xAE8BDC79DCDC8B72),
	SPH_C64(0x7B56E532E5E5569E), SPH_C64(0xF940B2CBB2B2404B),
	SPH_C64(0x4A6B4ED24E4E6B04), SPH_C64(0x76FCC754C7C7FCB1),
	SPH_C64(0x4FC46DB76D6DC422), SPH_C64(0x1B6AE926E9E96AF2),
	SPH_C64(0x25BB27692727BB02), SPH_C64(0x3A5D40C040405D7A),
	SPH_C64(0x8E9FD875D8D89F56), SPH_C64(0xA5EB37593737EB92),
	SPH_C64(0xE4E092AB9292E076), SPH_C64(0x0C898F8C8F8F8983),
	SPH_C64(0x0805010301010509), SPH_C64(0xE8691D271D1D69F5),
	SPH_C64(0xA20253F5535302F1), SPH_C64(0xEDC63E423E3EC6D3),
	SPH_C64(0xF22059EB595920AB), SPH_C64(0x46E2C15EC1C1E287),
	SPH_C64(0x426E4FD14F4F6E0D), SPH_C64(0x8DFA32563232FABF),
	SPH_C64(0xB04E163A16164EA6), SPH_C64(0x8335FA13FAFA3579),
	SPH_C64(0x87B9749C7474B9F3), SPH_C64(0x8B30FB10FBFB3070),
	SPH_C64(0x3FF263A56363F25C), SPH_C64(0x8CD99FBC9F9FD913),
	SPH_C64(0xBDE4345C3434E489), SPH_C64(0xD0721A2E1A1A72CA),
	SPH_C64(0x4D822A7E2A2A8267), SPH_C64(0xEA2F5AEE5A5A2FB0),
	SPH_C64(0x1C838D8A8D8D8391), SPH_C64(0x06CAC946C9C9CACF),
	SPH_C64(0x36D4CF4CCFCFD4F9), SPH_C64(0xE309F607F6F60915),
	SPH_C64(0xF4EA90AD9090EA64), SPH_C64(0x5D88287828288875),
	SPH_C64(0x34928885888892BC), SPH_C64(0xACCD9BB09B9BCD37),
	SPH_C64(0x95F531533131F5A4), SPH_C64(0x70360E120E0E367E),
	SPH_C64(0x8173BDDABDBD733C), SPH_C64(0x6A7F4ADE4A4A7F20),
	SPH_C64(0x136FE825E8E86FFB), SPH_C64(0xC4F496A79696F452),
	SPH_C64(0x5904A6F7A6A604FF), SPH_C64(0x603C0C140C0C3C6C),
	SPH_C64(0x0ECFC845C8C8CFC6), SPH_C64(0xEF80798B79798096),
	SPH_C64(0x8976BCD9BCBC7635), SPH_C64(0x997CBEDFBEBE7C27),
	SPH_C64(0x2B74EF2CEFEF74C4), SPH_C64(0x57CB6EB26E6ECB39),
	SPH_C64(0x0A4346CA4646434C), SPH_C64(0xCCF197A49797F15B),
	SPH_C64(0xE22A5BED5B5B2AB9), SPH_C64(0x3B7EED2AEDED7ED6),
	SPH_C64(0xC87D192B19197DD1), SPH_C64(0x869AD976D9D99A5F),
	SPH_C64(0x0926ACE9ACAC26A5), SPH_C64(0xBCC799B69999C725),
	SPH_C64(0x2932A8E5A8A83281), SPH_C64(0x558D297B29298D7C),
	SPH_C64(0x07E964AC6464E963), SPH_C64(0xF8631F211F1F63E7),
	SPH_C64(0x0123ADEAADAD23AC), SPH_C64(0x921C55FF55551CC7),
	SPH_C64(0x985F133513135F8B), SPH_C64(0xB16DBBD0BBBB6D0A),
	SPH_C64(0xEB0CF704F7F70C1C), SPH_C64(0x5FCE6FB16F6FCE30),
	SPH_C64(0xA167B9D6B9B96718), SPH_C64(0x024647C947474645),
	SPH_C64(0x65932F712F2F934A), SPH_C64(0x2371EE2FEEEE71CD),
	SPH_C64(0xA962B8D5B8B86211), SPH_C64(0xFF8A7B8D7B7B8A84),
	SPH_C64(0x3C978986898997B5), SPH_C64(0x9DF030503030F0AD),
	SPH_C64(0xD6B8D368D3D3B805), SPH_C64(0xDF9E7F817F7F9EA0),
	SPH_C64(0x97B3769A7676B3E1), SPH_C64(0x64B0829B8282B0E6)
};

static const sph_u64 old0_T3[256] = {
	SPH_C64(0xD568B86868D50F67), SPH_C64(0xB7D06DD0D0B71ECE),
	SPH_C64(0x60EB20EBEB60E00B), SPH_C64(0x872B7D2B2B876E45),
	SPH_C64(0x7548D8484875327A), SPH_C64(0xD39DBA9D9DD3019C),
	SPH_C64(0xDF6ABE6A6ADF1D77), SPH_C64(0x53E431E4E4539773),
	SPH_C64(0x48E338E3E348A84B), SPH_C64(0x15A3F8A3A315D271),
	SPH_C64(0x1356FA565613DC8A), SPH_C64(0xBF819E8181BFFD7C),
	SPH_C64(0x947D877D7D94B2CF), SPH_C64(0x12F10EF1F1122ADB),
	SPH_C64(0xAB85928585ABD95C), SPH_C64(0xDC9EBF9E9EDC1A84),
	SPH_C64(0x9C2C742C2C9C517D), SPH_C64(0x8C8E8F8E8E8C8A04),
	SPH_C64(0x8578887878859FE7), SPH_C64(0xC5CA43CACAC5D41E),
	SPH_C64(0x4B173917174BAFB8), SPH_C64(0x37A9E6A9A9378821),
	SPH_C64(0xF861A36161F84E2F), SPH_C64(0xA6D562D5D5A633E6),
	SPH_C64(0x345DE75D5D348FD2), SPH_C64(0x270B1D0B0B275358),
	SPH_C64(0x868C898C8C869814), SPH_C64(0xCC3C443C3CCCC1FD),
	SPH_C64(0xB677997777B6E89F), SPH_C64(0x0851F3515108E3B2),
	SPH_C64(0xAA22662222AA2F0D), SPH_C64(0x5742C6424257682A),
	SPH_C64(0xC33F413F3FC3DAE5), SPH_C64(0x1954FC545419CE9A),
	SPH_C64(0x5841C34141587332), SPH_C64(0xBA809D8080BAF474),
	SPH_C64(0xDBCC49CCCCDBE22E), SPH_C64(0xA486978686A4C244),
	SPH_C64(0x45B3C8B3B34542F1), SPH_C64(0x781828181878D8C0),
	SPH_C64(0x962E722E2E96436D), SPH_C64(0x1657F9575716D582),
	SPH_C64(0x1E060A06061E3630), SPH_C64(0xF762A66262F75537),
	SPH_C64(0x03F401F4F40307F3), SPH_C64(0xEE365A3636EE9BAD),
	SPH_C64(0xB2D16ED1D1B217C6), SPH_C64(0xDA6BBD6B6BDA147F),
	SPH_C64(0x771B2D1B1B77C3D8), SPH_C64(0xEC65AF6565EC6A0F),
	SPH_C64(0xBC759F7575BCFA8F), SPH_C64(0x5010301010509080),
	SPH_C64(0x95DA73DADA95449E), SPH_C64(0x7049DB4949703B72),
	SPH_C64(0xBE266A2626BE0B2D), SPH_C64(0x3AF916F9F93A629B),
	SPH_C64(0xC0CB40CBCBC0DD16), SPH_C64(0xE366AA6666E37117),
	SPH_C64(0x5CE734E7E75C8C6B), SPH_C64(0x68BAD3BABA6803B9),
	SPH_C64(0x2CAEEFAEAE2CB719), SPH_C64(0x0D50F050500DEABA),
	SPH_C64(0x0752F6525207F8AA), SPH_C64(0x3DABE0ABAB3D9A31),
	SPH_C64(0x11050F0505112D28), SPH_C64(0x17F00DF0F01723D3),
	SPH_C64(0x390D170D0D396568), SPH_C64(0xA273957373A2CCBF),
	SPH_C64(0xD73B4D3B3BD7FEC5), SPH_C64(0x14040C0404142420),
	SPH_C64(0xA020602020A03D1D), SPH_C64(0x21FE1FFEFE215DA3),
	SPH_C64(0x8EDD7ADDDD8E7BA6), SPH_C64(0x06F502F5F5060EFB),
	SPH_C64(0x5EB4C1B4B45E7DC9), SPH_C64(0x3E5FE15F5F3E9DC2),
	SPH_C64(0x220A1E0A0A225A50), SPH_C64(0x5BB5C2B5B55B74C1),
	SPH_C64(0xE7C05DC0C0E78E4E), SPH_C64(0x1AA0FDA0A01AC969),
	SPH_C64(0xA871937171A8DEAF), SPH_C64(0x0BA5F2A5A50BE441),
	SPH_C64(0x992D772D2D995875), SPH_C64(0xFD60A06060FD4727),
	SPH_C64(0xA772967272A7C5B7), SPH_C64(0xE593A89393E57FEC),
	SPH_C64(0xDD394B3939DDECD5), SPH_C64(0x2808180808284840),
	SPH_C64(0xB583988383B5EF6C), SPH_C64(0xA521632121A53415),
	SPH_C64(0x315CE45C5C3186DA), SPH_C64(0xA187948787A1CB4C),
	SPH_C64(0x4FB1CEB1B14F50E1), SPH_C64(0x47E03DE0E047B353),
	SPH_C64(0x0000000000000000), SPH_C64(0xE8C358C3C3E89556),
	SPH_C64(0x5A123612125A8290), SPH_C64(0xEF91AE9191EF6DFC),
	SPH_C64(0x988A838A8A98AE24), SPH_C64(0x0A020602020A1210),
	SPH_C64(0x6C1C241C1C6CFCE0), SPH_C64(0x59E637E6E6598563),
	SPH_C64(0x4C45CF45454C5712), SPH_C64(0xEDC25BC2C2ED9C5E),
	SPH_C64(0xF3C451C4C4F3AA6E), SPH_C64(0x2EFD1AFDFD2E46BB),
	SPH_C64(0x79BFDCBFBF792E91), SPH_C64(0x4944CC4444495E1A),
	SPH_C64(0x1FA1FEA1A11FC061), SPH_C64(0x614CD44C4C61165A),
	SPH_C64(0xFF33553333FFB685), SPH_C64(0xF6C552C5C5F6A366),
	SPH_C64(0xAE84918484AED054), SPH_C64(0xAF23652323AF2605),
	SPH_C64(0x917C847C7C91BBC7), SPH_C64(0x4AB0CDB0B04A59E9),
	SPH_C64(0xB1256F2525B11035), SPH_C64(0x41153F151541BDA8),
	SPH_C64(0xE1355F3535E180B5), SPH_C64(0xD069BB6969D0066F),
	SPH_C64(0x24FF1CFFFF2454AB), SPH_C64(0xFE94A19494FE40D4),
	SPH_C64(0x644DD74D4D641F52), SPH_C64(0xAD70907070ADD7A7),
	SPH_C64(0x10A2FBA2A210DB79), SPH_C64(0x29AFECAFAF29BE11),
	SPH_C64(0xDECD4ACDCDDEEB26), SPH_C64(0xA9D667D6D6A928FE),
	SPH_C64(0xC16CB46C6CC12B47), SPH_C64(0x51B7C4B7B75166D1),
	SPH_C64(0x3FF815F8F83F6B93), SPH_C64(0x2D091B09092D4148),
	SPH_C64(0x18F308F3F31838CB), SPH_C64(0xE667A96767E6781F),
	SPH_C64(0x0EA4F1A4A40EED49), SPH_C64(0x65EA23EAEA65E903),
	SPH_C64(0x7BEC29ECEC7BDF33), SPH_C64(0x54B6C7B6B6546FD9),
	SPH_C64(0xA3D461D4D4A33AEE), SPH_C64(0xBDD26BD2D2BD0CDE),
	SPH_C64(0x44143C141444B4A0), SPH_C64(0x661E221E1E66EEF0),
	SPH_C64(0x42E13EE1E142BA5B), SPH_C64(0xB4246C2424B4193D),
	SPH_C64(0xD838483838D8E5DD), SPH_C64(0xF9C657C6C6F9B87E),
	SPH_C64(0x90DB70DBDB904D96), SPH_C64(0x7A4BDD4B4B7A2962),
	SPH_C64(0x8F7A8E7A7A8F8DF7), SPH_C64(0xD23A4E3A3AD2F7CD),
	SPH_C64(0x81DE7FDEDE8160BE), SPH_C64(0x3B5EE25E5E3B94CA),
	SPH_C64(0x84DF7CDFDF8469B6), SPH_C64(0xFB95A29595FB49DC),
	SPH_C64(0x2BFC19FCFC2B4FB3), SPH_C64(0x38AAE3AAAA389339),
	SPH_C64(0xACD764D7D7AC21F6), SPH_C64(0xD1CE4FCECED1F03E),
	SPH_C64(0x1B070907071B3F38), SPH_C64(0x330F110F0F337778),
	SPH_C64(0xC93D473D3DC9C8F5), SPH_C64(0x2558E8585825A2FA),
	SPH_C64(0xC89AB39A9AC83EA4), SPH_C64(0xC298B59898C22CB4),
	SPH_C64(0xD69CB99C9CD60894), SPH_C64(0x1DF20BF2F21D31C3),
	SPH_C64(0x01A7F4A7A701F651), SPH_C64(0x5511331111559988),
	SPH_C64(0x9B7E827E7E9BA9D7), SPH_C64(0x9D8B808B8B9DA72C),
	SPH_C64(0x5243C54343526122), SPH_C64(0x0F030503030F1B18),
	SPH_C64(0x4DE23BE2E24DA143), SPH_C64(0x8BDC79DCDC8B72AE),
	SPH_C64(0x56E532E5E5569E7B), SPH_C64(0x40B2CBB2B2404BF9),
	SPH_C64(0x6B4ED24E4E6B044A), SPH_C64(0xFCC754C7C7FCB176),
	SPH_C64(0xC46DB76D6DC4224F), SPH_C64(0x6AE926E9E96AF21B),
	SPH_C64(0xBB27692727BB0225), SPH_C64(0x5D40C040405D7A3A),
	SPH_C64(0x9FD875D8D89F568E), SPH_C64(0xEB37593737EB92A5),
	SPH_C64(0xE092AB9292E076E4), SPH_C64(0x898F8C8F8F89830C),
	SPH_C64(0x0501030101050908), SPH_C64(0x691D271D1D69F5E8),
	SPH_C64(0x0253F5535302F1A2), SPH_C64(0xC63E423E3EC6D3ED),
	SPH_C64(0x2059EB595920ABF2), SPH_C64(0xE2C15EC1C1E28746),
	SPH_C64(0x6E4FD14F4F6E0D42), SPH_C64(0xFA32563232FABF8D),
	SPH_C64(0x4E163A16164EA6B0), SPH_C64(0x35FA13FAFA357983),
	SPH_C64(0xB9749C7474B9F387), SPH_C64(0x30FB10FBFB30708B),
	SPH_C64(0xF263A56363F25C3F), SPH_C64(0xD99FBC9F9FD9138C),
	SPH_C64(0xE4345C3434E489BD), SPH_C64(0x721A2E1A1A72CAD0),
	SPH_C64(0x822A7E2A2A82674D), SPH_C64(0x2F5AEE5A5A2FB0EA),
	SPH_C64(0x838D8A8D8D83911C), SPH_C64(0xCAC946C9C9CACF06),
	SPH_C64(0xD4CF4CCFCFD4F936), SPH_C64(0x09F607F6F60915E3),
	SPH_C64(0xEA90AD9090EA64F4), SPH_C64(0x882878282888755D),
	SPH_C64(0x928885888892BC34), SPH_C64(0xCD9BB09B9BCD37AC),
	SPH_C64(0xF531533131F5A495), SPH_C64(0x360E120E0E367E70),
	SPH_C64(0x73BDDABDBD733C81), SPH_C64(0x7F4ADE4A4A7F206A),
	SPH_C64(0x6FE825E8E86FFB13), SPH_C64(0xF496A79696F452C4),
	SPH_C64(0x04A6F7A6A604FF59), SPH_C64(0x3C0C140C0C3C6C60),
	SPH_C64(0xCFC845C8C8CFC60E), SPH_C64(0x80798B79798096EF),
	SPH_C64(0x76BCD9BCBC763589), SPH_C64(0x7CBEDFBEBE7C2799),
	SPH_C64(0x74EF2CEFEF74C42B), SPH_C64(0xCB6EB26E6ECB3957),
	SPH_C64(0x4346CA4646434C0A), SPH_C64(0xF197A49797F15BCC),
	SPH_C64(0x2A5BED5B5B2AB9E2), SPH_C64(0x7EED2AEDED7ED63B),
	SPH_C64(0x7D192B19197DD1C8), SPH_C64(0x9AD976D9D99A5F86),
	SPH_C64(0x26ACE9ACAC26A509), SPH_C64(0xC799B69999C725BC),
	SPH_C64(0x32A8E5A8A8328129), SPH_C64(0x8D297B29298D7C55),
	SPH_C64(0xE964AC6464E96307), SPH_C64(0x631F211F1F63E7F8),
	SPH_C64(0x23ADEAADAD23AC01), SPH_C64(0x1C55FF55551CC792),
	SPH_C64(0x5F133513135F8B98), SPH_C64(0x6DBBD0BBBB6D0AB1),
	SPH_C64(0x0CF704F7F70C1CEB), SPH_C64(0xCE6FB16F6FCE305F),
	SPH_C64(0x67B9D6B9B96718A1), SPH_C64(0x4647C94747464502),
	SPH_C64(0x932F712F2F934A65), SPH_C64(0x71EE2FEEEE71CD23),
	SPH_C64(0x62B8D5B8B86211A9), SPH_C64(0x8A7B8D7B7B8A84FF),
	SPH_C64(0x978986898997B53C), SPH_C64(0xF030503030F0AD9D),
	SPH_C64(0xB8D368D3D3B805D6), SPH_C64(0x9E7F817F7F9EA0DF),
	SPH_C64(0xB3769A7676B3E197), SPH_C64(0xB0829B8282B0E664)
};

static const sph_u64 old0_T4[256] = {
	SPH_C64(0x68B86868D50F67D5), SPH_C64(0xD06DD0D0B71ECEB7),
	SPH_C64(0xEB20EBEB60E00B60), SPH_C64(0x2B7D2B2B876E4587),
	SPH_C64(0x48D8484875327A75), SPH_C64(0x9DBA9D9DD3019CD3),
	SPH_C64(0x6ABE6A6ADF1D77DF), SPH_C64(0xE431E4E453977353),
	SPH_C64(0xE338E3E348A84B48), SPH_C64(0xA3F8A3A315D27115),
	SPH_C64(0x56FA565613DC8A13), SPH_C64(0x819E8181BFFD7CBF),
	SPH_C64(0x7D877D7D94B2CF94), SPH_C64(0xF10EF1F1122ADB12),
	SPH_C64(0x85928585ABD95CAB), SPH_C64(0x9EBF9E9EDC1A84DC),
	SPH_C64(0x2C742C2C9C517D9C), SPH_C64(0x8E8F8E8E8C8A048C),
	SPH_C64(0x78887878859FE785), SPH_C64(0xCA43CACAC5D41EC5),
	SPH_C64(0x173917174BAFB84B), SPH_C64(0xA9E6A9A937882137),
	SPH_C64(0x61A36161F84E2FF8), SPH_C64(0xD562D5D5A633E6A6),
	SPH_C64(0x5DE75D5D348FD234), SPH_C64(0x0B1D0B0B27535827),
	SPH_C64(0x8C898C8C86981486), SPH_C64(0x3C443C3CCCC1FDCC),
	SPH_C64(0x77997777B6E89FB6), SPH_C64(0x51F3515108E3B208),
	SPH_C64(0x22662222AA2F0DAA), SPH_C64(0x42C6424257682A57),
	SPH_C64(0x3F413F3FC3DAE5C3), SPH_C64(0x54FC545419CE9A19),
	SPH_C64(0x41C3414158733258), SPH_C64(0x809D8080BAF474BA),
	SPH_C64(0xCC49CCCCDBE22EDB), SPH_C64(0x86978686A4C244A4),
	SPH_C64(0xB3C8B3B34542F145), SPH_C64(0x1828181878D8C078),
	SPH_C64(0x2E722E2E96436D96), SPH_C64(0x57F9575716D58216),
	SPH_C64(0x060A06061E36301E), SPH_C64(0x62A66262F75537F7),
	SPH_C64(0xF401F4F40307F303), SPH_C64(0x365A3636EE9BADEE),
	SPH_C64(0xD16ED1D1B217C6B2), SPH_C64(0x6BBD6B6BDA147FDA),
	SPH_C64(0x1B2D1B1B77C3D877), SPH_C64(0x65AF6565EC6A0FEC),
	SPH_C64(0x759F7575BCFA8FBC), SPH_C64(0x1030101050908050),
	SPH_C64(0xDA73DADA95449E95), SPH_C64(0x49DB4949703B7270),
	SPH_C64(0x266A2626BE0B2DBE), SPH_C64(0xF916F9F93A629B3A),
	SPH_C64(0xCB40CBCBC0DD16C0), SPH_C64(0x66AA6666E37117E3),
	SPH_C64(0xE734E7E75C8C6B5C), SPH_C64(0xBAD3BABA6803B968),
	SPH_C64(0xAEEFAEAE2CB7192C), SPH_C64(0x50F050500DEABA0D),
	SPH_C64(0x52F6525207F8AA07), SPH_C64(0xABE0ABAB3D9A313D),
	SPH_C64(0x050F0505112D2811), SPH_C64(0xF00DF0F01723D317),
	SPH_C64(0x0D170D0D39656839), SPH_C64(0x73957373A2CCBFA2),
	SPH_C64(0x3B4D3B3BD7FEC5D7), SPH_C64(0x040C040414242014),
	SPH_C64(0x20602020A03D1DA0), SPH_C64(0xFE1FFEFE215DA321),
	SPH_C64(0xDD7ADDDD8E7BA68E), SPH_C64(0xF502F5F5060EFB06),
	SPH_C64(0xB4C1B4B45E7DC95E), SPH_C64(0x5FE15F5F3E9DC23E),
	SPH_C64(0x0A1E0A0A225A5022), SPH_C64(0xB5C2B5B55B74C15B),
	SPH_C64(0xC05DC0C0E78E4EE7), SPH_C64(0xA0FDA0A01AC9691A),
	SPH_C64(0x71937171A8DEAFA8), SPH_C64(0xA5F2A5A50BE4410B),
	SPH_C64(0x2D772D2D99587599), SPH_C64(0x60A06060FD4727FD),
	SPH_C64(0x72967272A7C5B7A7), SPH_C64(0x93A89393E57FECE5),
	SPH_C64(0x394B3939DDECD5DD), SPH_C64(0x0818080828484028),
	SPH_C64(0x83988383B5EF6CB5), SPH_C64(0x21632121A53415A5),
	SPH_C64(0x5CE45C5C3186DA31), SPH_C64(0x87948787A1CB4CA1),
	SPH_C64(0xB1CEB1B14F50E14F), SPH_C64(0xE03DE0E047B35347),
	SPH_C64(0x0000000000000000), SPH_C64(0xC358C3C3E89556E8),
	SPH_C64(0x123612125A82905A), SPH_C64(0x91AE9191EF6DFCEF),
	SPH_C64(0x8A838A8A98AE2498), SPH_C64(0x020602020A12100A),
	SPH_C64(0x1C241C1C6CFCE06C), SPH_C64(0xE637E6E659856359),
	SPH_C64(0x45CF45454C57124C), SPH_C64(0xC25BC2C2ED9C5EED),
	SPH_C64(0xC451C4C4F3AA6EF3), SPH_C64(0xFD1AFDFD2E46BB2E),
	SPH_C64(0xBFDCBFBF792E9179), SPH_C64(0x44CC4444495E1A49),
	SPH_C64(0xA1FEA1A11FC0611F), SPH_C64(0x4CD44C4C61165A61),
	SPH_C64(0x33553333FFB685FF), SPH_C64(0xC552C5C5F6A366F6),
	SPH_C64(0x84918484AED054AE), SPH_C64(0x23652323AF2605AF),
	SPH_C64(0x7C847C7C91BBC791), SPH_C64(0xB0CDB0B04A59E94A),
	SPH_C64(0x256F2525B11035B1), SPH_C64(0x153F151541BDA841),
	SPH_C64(0x355F3535E180B5E1), SPH_C64(0x69BB6969D0066FD0),
	SPH_C64(0xFF1CFFFF2454AB24), SPH_C64(0x94A19494FE40D4FE),
	SPH_C64(0x4DD74D4D641F5264), SPH_C64(0x70907070ADD7A7AD),
	SPH_C64(0xA2FBA2A210DB7910), SPH_C64(0xAFECAFAF29BE1129),
	SPH_C64(0xCD4ACDCDDEEB26DE), SPH_C64(0xD667D6D6A928FEA9),
	SPH_C64(0x6CB46C6CC12B47C1), SPH_C64(0xB7C4B7B75166D151),
	SPH_C64(0xF815F8F83F6B933F), SPH_C64(0x091B09092D41482D),
	SPH_C64(0xF308F3F31838CB18), SPH_C64(0x67A96767E6781FE6),
	SPH_C64(0xA4F1A4A40EED490E), SPH_C64(0xEA23EAEA65E90365),
	SPH_C64(0xEC29ECEC7BDF337B), SPH_C64(0xB6C7B6B6546FD954),
	SPH_C64(0xD461D4D4A33AEEA3), SPH_C64(0xD26BD2D2BD0CDEBD),
	SPH_C64(0x143C141444B4A044), SPH_C64(0x1E221E1E66EEF066),
	SPH_C64(0xE13EE1E142BA5B42), SPH_C64(0x246C2424B4193DB4),
	SPH_C64(0x38483838D8E5DDD8), SPH_C64(0xC657C6C6F9B87EF9),
	SPH_C64(0xDB70DBDB904D9690), SPH_C64(0x4BDD4B4B7A29627A),
	SPH_C64(0x7A8E7A7A8F8DF78F), SPH_C64(0x3A4E3A3AD2F7CDD2),
	SPH_C64(0xDE7FDEDE8160BE81), SPH_C64(0x5EE25E5E3B94CA3B),
	SPH_C64(0xDF7CDFDF8469B684), SPH_C64(0x95A29595FB49DCFB),
	SPH_C64(0xFC19FCFC2B4FB32B), SPH_C64(0xAAE3AAAA38933938),
	SPH_C64(0xD764D7D7AC21F6AC), SPH_C64(0xCE4FCECED1F03ED1),
	SPH_C64(0x070907071B3F381B), SPH_C64(0x0F110F0F33777833),
	SPH_C64(0x3D473D3DC9C8F5C9), SPH_C64(0x58E8585825A2FA25),
	SPH_C64(0x9AB39A9AC83EA4C8), SPH_C64(0x98B59898C22CB4C2),
	SPH_C64(0x9CB99C9CD60894D6), SPH_C64(0xF20BF2F21D31C31D),
	SPH_C64(0xA7F4A7A701F65101), SPH_C64(0x1133111155998855),
	SPH_C64(0x7E827E7E9BA9D79B), SPH_C64(0x8B808B8B9DA72C9D),
	SPH_C64(0x43C5434352612252), SPH_C64(0x030503030F1B180F),
	SPH_C64(0xE23BE2E24DA1434D), SPH_C64(0xDC79DCDC8B72AE8B),
	SPH_C64(0xE532E5E5569E7B56), SPH_C64(0xB2CBB2B2404BF940),
	SPH_C64(0x4ED24E4E6B044A6B), SPH_C64(0xC754C7C7FCB176FC),
	SPH_C64(0x6DB76D6DC4224FC4), SPH_C64(0xE926E9E96AF21B6A),
	SPH_C64(0x27692727BB0225BB), SPH_C64(0x40C040405D7A3A5D),
	SPH_C64(0xD875D8D89F568E9F), SPH_C64(0x37593737EB92A5EB),
	SPH_C64(0x92AB9292E076E4E0), SPH_C64(0x8F8C8F8F89830C89),
	SPH_C64(0x0103010105090805), SPH_C64(0x1D271D1D69F5E869),
	SPH_C64(0x53F5535302F1A202), SPH_C64(0x3E423E3EC6D3EDC6),
	SPH_C64(0x59EB595920ABF220), SPH_C64(0xC15EC1C1E28746E2),
	SPH_C64(0x4FD14F4F6E0D426E), SPH_C64(0x32563232FABF8DFA),
	SPH_C64(0x163A16164EA6B04E), SPH_C64(0xFA13FAFA35798335),
	SPH_C64(0x749C7474B9F387B9), SPH_C64(0xFB10FBFB30708B30),
	SPH_C64(0x63A56363F25C3FF2), SPH_C64(0x9FBC9F9FD9138CD9),
	SPH_C64(0x345C3434E489BDE4), SPH_C64(0x1A2E1A1A72CAD072),
	SPH_C64(0x2A7E2A2A82674D82), SPH_C64(0x5AEE5A5A2FB0EA2F),
	SPH_C64(0x8D8A8D8D83911C83), SPH_C64(0xC946C9C9CACF06CA),
	SPH_C64(0xCF4CCFCFD4F936D4), SPH_C64(0xF607F6F60915E309),
	SPH_C64(0x90AD9090EA64F4EA), SPH_C64(0x2878282888755D88),
	SPH_C64(0x8885888892BC3492), SPH_C64(0x9BB09B9BCD37ACCD),
	SPH_C64(0x31533131F5A495F5), SPH_C64(0x0E120E0E367E7036),
	SPH_C64(0xBDDABDBD733C8173), SPH_C64(0x4ADE4A4A7F206A7F),
	SPH_C64(0xE825E8E86FFB136F), SPH_C64(0x96A79696F452C4F4),
	SPH_C64(0xA6F7A6A604FF5904), SPH_C64(0x0C140C0C3C6C603C),
	SPH_C64(0xC845C8C8CFC60ECF), SPH_C64(0x798B79798096EF80),
	SPH_C64(0xBCD9BCBC76358976), SPH_C64(0xBEDFBEBE7C27997C),
	SPH_C64(0xEF2CEFEF74C42B74), SPH_C64(0x6EB26E6ECB3957CB),
	SPH_C64(0x46CA4646434C0A43), SPH_C64(0x97A49797F15BCCF1),
	SPH_C64(0x5BED5B5B2AB9E22A), SPH_C64(0xED2AEDED7ED63B7E),
	SPH_C64(0x192B19197DD1C87D), SPH_C64(0xD976D9D99A5F869A),
	SPH_C64(0xACE9ACAC26A50926), SPH_C64(0x99B69999C725BCC7),
	SPH_C64(0xA8E5A8A832812932), SPH_C64(0x297B29298D7C558D),
	SPH_C64(0x64AC6464E96307E9), SPH_C64(0x1F211F1F63E7F863),
	SPH_C64(0xADEAADAD23AC0123), SPH_C64(0x55FF55551CC7921C),
	SPH_C64(0x133513135F8B985F), SPH_C64(0xBBD0BBBB6D0AB16D),
	SPH_C64(0xF704F7F70C1CEB0C), SPH_C64(0x6FB16F6FCE305FCE),
	SPH_C64(0xB9D6B9B96718A167), SPH_C64(0x47C9474746450246),
	SPH_C64(0x2F712F2F934A6593), SPH_C64(0xEE2FEEEE71CD2371),
	SPH_C64(0xB8D5B8B86211A962), SPH_C64(0x7B8D7B7B8A84FF8A),
	SPH_C64(0x8986898997B53C97), SPH_C64(0x30503030F0AD9DF0),
	SPH_C64(0xD368D3D3B805D6B8), SPH_C64(0x7F817F7F9EA0DF9E),
	SPH_C64(0x769A7676B3E197B3), SPH_C64(0x829B8282B0E664B0)
};

static const sph_u64 old0_T5[256] = {
	SPH_C64(0xB86868D50F67D568), SPH_C64(0x6DD0D0B71ECEB7D0),
	SPH_C64(0x20EBEB60E00B60EB), SPH_C64(0x7D2B2B876E45872B),
	SPH_C64(0xD8484875327A7548), SPH_C64(0xBA9D9DD3019CD39D),
	SPH_C64(0xBE6A6ADF1D77DF6A), SPH_C64(0x31E4E453977353E4),
	SPH_C64(0x38E3E348A84B48E3), SPH_C64(0xF8A3A315D27115A3),
	SPH_C64(0xFA565613DC8A1356), SPH_C64(0x9E8181BFFD7CBF81),
	SPH_C64(0x877D7D94B2CF947D), SPH_C64(0x0EF1F1122ADB12F1),
	SPH_C64(0x928585ABD95CAB85), SPH_C64(0xBF9E9EDC1A84DC9E),
	SPH_C64(0x742C2C9C517D9C2C), SPH_C64(0x8F8E8E8C8A048C8E),
	SPH_C64(0x887878859FE78578), SPH_C64(0x43CACAC5D41EC5CA),
	SPH_C64(0x3917174BAFB84B17), SPH_C64(0xE6A9A937882137A9),
	SPH_C64(0xA36161F84E2FF861), SPH_C64(0x62D5D5A633E6A6D5),
	SPH_C64(0xE75D5D348FD2345D), SPH_C64(0x1D0B0B275358270B),
	SPH_C64(0x898C8C869814868C), SPH_C64(0x443C3CCCC1FDCC3C),
	SPH_C64(0x997777B6E89FB677), SPH_C64(0xF3515108E3B20851),
	SPH_C64(0x662222AA2F0DAA22), SPH_C64(0xC6424257682A5742),
	SPH_C64(0x413F3FC3DAE5C33F), SPH_C64(0xFC545419CE9A1954),
	SPH_C64(0xC341415873325841), SPH_C64(0x9D8080BAF474BA80),
	SPH_C64(0x49CCCCDBE22EDBCC), SPH_C64(0x978686A4C244A486),
	SPH_C64(0xC8B3B34542F145B3), SPH_C64(0x28181878D8C07818),
	SPH_C64(0x722E2E96436D962E), SPH_C64(0xF9575716D5821657),
	SPH_C64(0x0A06061E36301E06), SPH_C64(0xA66262F75537F762),
	SPH_C64(0x01F4F40307F303F4), SPH_C64(0x5A3636EE9BADEE36),
	SPH_C64(0x6ED1D1B217C6B2D1), SPH_C64(0xBD6B6BDA147FDA6B),
	SPH_C64(0x2D1B1B77C3D8771B), SPH_C64(0xAF6565EC6A0FEC65),
	SPH_C64(0x9F7575BCFA8FBC75), SPH_C64(0x3010105090805010),
	SPH_C64(0x73DADA95449E95DA), SPH_C64(0xDB4949703B727049),
	SPH_C64(0x6A2626BE0B2DBE26), SPH_C64(0x16F9F93A629B3AF9),
	SPH_C64(0x40CBCBC0DD16C0CB), SPH_C64(0xAA6666E37117E366),
	SPH_C64(0x34E7E75C8C6B5CE7), SPH_C64(0xD3BABA6803B968BA),
	SPH_C64(0xEFAEAE2CB7192CAE), SPH_C64(0xF050500DEABA0D50),
	SPH_C64(0xF6525207F8AA0752), SPH_C64(0xE0ABAB3D9A313DAB),
	SPH_C64(0x0F0505112D281105), SPH_C64(0x0DF0F01723D317F0),
	SPH_C64(0x170D0D396568390D), SPH_C64(0x957373A2CCBFA273),
	SPH_C64(0x4D3B3BD7FEC5D73B), SPH_C64(0x0C04041424201404),
	SPH_C64(0x602020A03D1DA020), SPH_C64(0x1FFEFE215DA321FE),
	SPH_C64(0x7ADDDD8E7BA68EDD), SPH_C64(0x02F5F5060EFB06F5),
	SPH_C64(0xC1B4B45E7DC95EB4), SPH_C64(0xE15F5F3E9DC23E5F),
	SPH_C64(0x1E0A0A225A50220A), SPH_C64(0xC2B5B55B74C15BB5),
	SPH_C64(0x5DC0C0E78E4EE7C0), SPH_C64(0xFDA0A01AC9691AA0),
	SPH_C64(0x937171A8DEAFA871), SPH_C64(0xF2A5A50BE4410BA5),
	SPH_C64(0x772D2D995875992D), SPH_C64(0xA06060FD4727FD60),
	SPH_C64(0x967272A7C5B7A772), SPH_C64(0xA89393E57FECE593),
	SPH_C64(0x4B3939DDECD5DD39), SPH_C64(0x1808082848402808),
	SPH_C64(0x988383B5EF6CB583), SPH_C64(0x632121A53415A521),
	SPH_C64(0xE45C5C3186DA315C), SPH_C64(0x948787A1CB4CA187),
	SPH_C64(0xCEB1B14F50E14FB1), SPH_C64(0x3DE0E047B35347E0),
	SPH_C64(0x0000000000000000), SPH_C64(0x58C3C3E89556E8C3),
	SPH_C64(0x3612125A82905A12), SPH_C64(0xAE9191EF6DFCEF91),
	SPH_C64(0x838A8A98AE24988A), SPH_C64(0x0602020A12100A02),
	SPH_C64(0x241C1C6CFCE06C1C), SPH_C64(0x37E6E659856359E6),
	SPH_C64(0xCF45454C57124C45), SPH_C64(0x5BC2C2ED9C5EEDC2),
	SPH_C64(0x51C4C4F3AA6EF3C4), SPH_C64(0x1AFDFD2E46BB2EFD),
	SPH_C64(0xDCBFBF792E9179BF), SPH_C64(0xCC4444495E1A4944),
	SPH_C64(0xFEA1A11FC0611FA1), SPH_C64(0xD44C4C61165A614C),
	SPH_C64(0x553333FFB685FF33), SPH_C64(0x52C5C5F6A366F6C5),
	SPH_C64(0x918484AED054AE84), SPH_C64(0x652323AF2605AF23),
	SPH_C64(0x847C7C91BBC7917C), SPH_C64(0xCDB0B04A59E94AB0),
	SPH_C64(0x6F2525B11035B125), SPH_C64(0x3F151541BDA84115),
	SPH_C64(0x5F3535E180B5E135), SPH_C64(0xBB6969D0066FD069),
	SPH_C64(0x1CFFFF2454AB24FF), SPH_C64(0xA19494FE40D4FE94),
	SPH_C64(0xD74D4D641F52644D), SPH_C64(0x907070ADD7A7AD70),
	SPH_C64(0xFBA2A210DB7910A2), SPH_C64(0xECAFAF29BE1129AF),
	SPH_C64(0x4ACDCDDEEB26DECD), SPH_C64(0x67D6D6A928FEA9D6),
	SPH_C64(0xB46C6CC12B47C16C), SPH_C64(0xC4B7B75166D151B7),
	SPH_C64(0x15F8F83F6B933FF8), SPH_C64(0x1B09092D41482D09),
	SPH_C64(0x08F3F31838CB18F3), SPH_C64(0xA96767E6781FE667),
	SPH_C64(0xF1A4A40EED490EA4), SPH_C64(0x23EAEA65E90365EA),
	SPH_C64(0x29ECEC7BDF337BEC), SPH_C64(0xC7B6B6546FD954B6),
	SPH_C64(0x61D4D4A33AEEA3D4), SPH_C64(0x6BD2D2BD0CDEBDD2),
	SPH_C64(0x3C141444B4A04414), SPH_C64(0x221E1E66EEF0661E),
	SPH_C64(0x3EE1E142BA5B42E1), SPH_C64(0x6C2424B4193DB424),
	SPH_C64(0x483838D8E5DDD838), SPH_C64(0x57C6C6F9B87EF9C6),
	SPH_C64(0x70DBDB904D9690DB), SPH_C64(0xDD4B4B7A29627A4B),
	SPH_C64(0x8E7A7A8F8DF78F7A), SPH_C64(0x4E3A3AD2F7CDD23A),
	SPH_C64(0x7FDEDE8160BE81DE), SPH_C64(0xE25E5E3B94CA3B5E),
	SPH_C64(0x7CDFDF8469B684DF), SPH_C64(0xA29595FB49DCFB95),
	SPH_C64(0x19FCFC2B4FB32BFC), SPH_C64(0xE3AAAA38933938AA),
	SPH_C64(0x64D7D7AC21F6ACD7), SPH_C64(0x4FCECED1F03ED1CE),
	SPH_C64(0x0907071B3F381B07), SPH_C64(0x110F0F337778330F),
	SPH_C64(0x473D3DC9C8F5C93D), SPH_C64(0xE8585825A2FA2558),
	SPH_C64(0xB39A9AC83EA4C89A), SPH_C64(0xB59898C22CB4C298),
	SPH_C64(0xB99C9CD60894D69C), SPH_C64(0x0BF2F21D31C31DF2),
	SPH_C64(0xF4A7A701F65101A7), SPH_C64(0x3311115599885511),
	SPH_C64(0x827E7E9BA9D79B7E), SPH_C64(0x808B8B9DA72C9D8B),
	SPH_C64(0xC543435261225243), SPH_C64(0x0503030F1B180F03),
	SPH_C64(0x3BE2E24DA1434DE2), SPH_C64(0x79DCDC8B72AE8BDC),
	SPH_C64(0x32E5E5569E7B56E5), SPH_C64(0xCBB2B2404BF940B2),
	SPH_C64(0xD24E4E6B044A6B4E), SPH_C64(0x54C7C7FCB176FCC7),
	SPH_C64(0xB76D6DC4224FC46D), SPH_C64(0x26E9E96AF21B6AE9),
	SPH_C64(0x692727BB0225BB27), SPH_C64(0xC040405D7A3A5D40),
	SPH_C64(0x75D8D89F568E9FD8), SPH_C64(0x593737EB92A5EB37),
	SPH_C64(0xAB9292E076E4E092), SPH_C64(0x8C8F8F89830C898F),
	SPH_C64(0x0301010509080501), SPH_C64(0x271D1D69F5E8691D),
	SPH_C64(0xF5535302F1A20253), SPH_C64(0x423E3EC6D3EDC63E),
	SPH_C64(0xEB595920ABF22059), SPH_C64(0x5EC1C1E28746E2C1),
	SPH_C64(0xD14F4F6E0D426E4F), SPH_C64(0x563232FABF8DFA32),
	SPH_C64(0x3A16164EA6B04E16), SPH_C64(0x13FAFA35798335FA),
	SPH_C64(0x9C7474B9F387B974), SPH_C64(0x10FBFB30708B30FB),
	SPH_C64(0xA56363F25C3FF263), SPH_C64(0xBC9F9FD9138CD99F),
	SPH_C64(0x5C3434E489BDE434), SPH_C64(0x2E1A1A72CAD0721A),
	SPH_C64(0x7E2A2A82674D822A), SPH_C64(0xEE5A5A2FB0EA2F5A),
	SPH_C64(0x8A8D8D83911C838D), SPH_C64(0x46C9C9CACF06CAC9),
	SPH_C64(0x4CCFCFD4F936D4CF), SPH_C64(0x07F6F60915E309F6),
	SPH_C64(0xAD9090EA64F4EA90), SPH_C64(0x78282888755D8828),
	SPH_C64(0x85888892BC349288), SPH_C64(0xB09B9BCD37ACCD9B),
	SPH_C64(0x533131F5A495F531), SPH_C64(0x120E0E367E70360E),
	SPH_C64(0xDABDBD733C8173BD), SPH_C64(0xDE4A4A7F206A7F4A),
	SPH_C64(0x25E8E86FFB136FE8), SPH_C64(0xA79696F452C4F496),
	SPH_C64(0xF7A6A604FF5904A6), SPH_C64(0x140C0C3C6C603C0C),
	SPH_C64(0x45C8C8CFC60ECFC8), SPH_C64(0x8B79798096EF8079),
	SPH_C64(0xD9BCBC76358976BC), SPH_C64(0xDFBEBE7C27997CBE),
	SPH_C64(0x2CEFEF74C42B74EF), SPH_C64(0xB26E6ECB3957CB6E),
	SPH_C64(0xCA4646434C0A4346), SPH_C64(0xA49797F15BCCF197),
	SPH_C64(0xED5B5B2AB9E22A5B), SPH_C64(0x2AEDED7ED63B7EED),
	SPH_C64(0x2B19197DD1C87D19), SPH_C64(0x76D9D99A5F869AD9),
	SPH_C64(0xE9ACAC26A50926AC), SPH_C64(0xB69999C725BCC799),
	SPH_C64(0xE5A8A832812932A8), SPH_C64(0x7B29298D7C558D29),
	SPH_C64(0xAC6464E96307E964), SPH_C64(0x211F1F63E7F8631F),
	SPH_C64(0xEAADAD23AC0123AD), SPH_C64(0xFF55551CC7921C55),
	SPH_C64(0x3513135F8B985F13), SPH_C64(0xD0BBBB6D0AB16DBB),
	SPH_C64(0x04F7F70C1CEB0CF7), SPH_C64(0xB16F6FCE305FCE6F),
	SPH_C64(0xD6B9B96718A167B9), SPH_C64(0xC947474645024647),
	SPH_C64(0x712F2F934A65932F), SPH_C64(0x2FEEEE71CD2371EE),
	SPH_C64(0xD5B8B86211A962B8), SPH_C64(0x8D7B7B8A84FF8A7B),
	SPH_C64(0x86898997B53C9789), SPH_C64(0x503030F0AD9DF030),
	SPH_C64(0x68D3D3B805D6B8D3), SPH_C64(0x817F7F9EA0DF9E7F),
	SPH_C64(0x9A7676B3E197B376), SPH_C64(0x9B8282B0E664B082)
};

static const sph_u64 old0_T6[256] = {
	SPH_C64(0x6868D50F67D568B8), SPH_C64(0xD0D0B71ECEB7D06D),
	SPH_C64(0xEBEB60E00B60EB20), SPH_C64(0x2B2B876E45872B7D),
	SPH_C64(0x484875327A7548D8), SPH_C64(0x9D9DD3019CD39DBA),
	SPH_C64(0x6A6ADF1D77DF6ABE), SPH_C64(0xE4E453977353E431),
	SPH_C64(0xE3E348A84B48E338), SPH_C64(0xA3A315D27115A3F8),
	SPH_C64(0x565613DC8A1356FA), SPH_C64(0x8181BFFD7CBF819E),
	SPH_C64(0x7D7D94B2CF947D87), SPH_C64(0xF1F1122ADB12F10E),
	SPH_C64(0x8585ABD95CAB8592), SPH_C64(0x9E9EDC1A84DC9EBF),
	SPH_C64(0x2C2C9C517D9C2C74), SPH_C64(0x8E8E8C8A048C8E8F),
	SPH_C64(0x7878859FE7857888), SPH_C64(0xCACAC5D41EC5CA43),
	SPH_C64(0x17174BAFB84B1739), SPH_C64(0xA9A937882137A9E6),
	SPH_C64(0x6161F84E2FF861A3), SPH_C64(0xD5D5A633E6A6D562),
	SPH_C64(0x5D5D348FD2345DE7), SPH_C64(0x0B0B275358270B1D),
	SPH_C64(0x8C8C869814868C89), SPH_C64(0x3C3CCCC1FDCC3C44),
	SPH_C64(0x7777B6E89FB67799), SPH_C64(0x515108E3B20851F3),
	SPH_C64(0x2222AA2F0DAA2266), SPH_C64(0x424257682A5742C6),
	SPH_C64(0x3F3FC3DAE5C33F41), SPH_C64(0x545419CE9A1954FC),
	SPH_C64(0x41415873325841C3), SPH_C64(0x8080BAF474BA809D),
	SPH_C64(0xCCCCDBE22EDBCC49), SPH_C64(0x8686A4C244A48697),
	SPH_C64(0xB3B34542F145B3C8), SPH_C64(0x181878D8C0781828),
	SPH_C64(0x2E2E96436D962E72), SPH_C64(0x575716D5821657F9),
	SPH_C64(0x06061E36301E060A), SPH_C64(0x6262F75537F762A6),
	SPH_C64(0xF4F40307F303F401), SPH_C64(0x3636EE9BADEE365A),
	SPH_C64(0xD1D1B217C6B2D16E), SPH_C64(0x6B6BDA147FDA6BBD),
	SPH_C64(0x1B1B77C3D8771B2D), SPH_C64(0x6565EC6A0FEC65AF),
	SPH_C64(0x7575BCFA8FBC759F), SPH_C64(0x1010509080501030),
	SPH_C64(0xDADA95449E95DA73), SPH_C64(0x4949703B727049DB),
	SPH_C64(0x2626BE0B2DBE266A), SPH_C64(0xF9F93A629B3AF916),
	SPH_C64(0xCBCBC0DD16C0CB40), SPH_C64(0x6666E37117E366AA),
	SPH_C64(0xE7E75C8C6B5CE734), SPH_C64(0xBABA6803B968BAD3),
	SPH_C64(0xAEAE2CB7192CAEEF), SPH_C64(0x50500DEABA0D50F0),
	SPH_C64(0x525207F8AA0752F6), SPH_C64(0xABAB3D9A313DABE0),
	SPH_C64(0x0505112D2811050F), SPH_C64(0xF0F01723D317F00D),
	SPH_C64(0x0D0D396568390D17), SPH_C64(0x7373A2CCBFA27395),
	SPH_C64(0x3B3BD7FEC5D73B4D), SPH_C64(0x040414242014040C),
	SPH_C64(0x2020A03D1DA02060), SPH_C64(0xFEFE215DA321FE1F),
	SPH_C64(0xDDDD8E7BA68EDD7A), SPH_C64(0xF5F5060EFB06F502),
	SPH_C64(0xB4B45E7DC95EB4C1), SPH_C64(0x5F5F3E9DC23E5FE1),
	SPH_C64(0x0A0A225A50220A1E), SPH_C64(0xB5B55B74C15BB5C2),
	SPH_C64(0xC0C0E78E4EE7C05D), SPH_C64(0xA0A01AC9691AA0FD),
	SPH_C64(0x7171A8DEAFA87193), SPH_C64(0xA5A50BE4410BA5F2),
	SPH_C64(0x2D2D995875992D77), SPH_C64(0x6060FD4727FD60A0),
	SPH_C64(0x7272A7C5B7A77296), SPH_C64(0x9393E57FECE593A8),
	SPH_C64(0x3939DDECD5DD394B), SPH_C64(0x0808284840280818),
	SPH_C64(0x8383B5EF6CB58398), SPH_C64(0x2121A53415A52163),
	SPH_C64(0x5C5C3186DA315CE4), SPH_C64(0x8787A1CB4CA18794),
	SPH_C64(0xB1B14F50E14FB1CE), SPH_C64(0xE0E047B35347E03D),
	SPH_C64(0x0000000000000000), SPH_C64(0xC3C3E89556E8C358),
	SPH_C64(0x12125A82905A1236), SPH_C64(0x9191EF6DFCEF91AE),
	SPH_C64(0x8A8A98AE24988A83), SPH_C64(0x02020A12100A0206),
	SPH_C64(0x1C1C6CFCE06C1C24), SPH_C64(0xE6E659856359E637),
	SPH_C64(0x45454C57124C45CF), SPH_C64(0xC2C2ED9C5EEDC25B),
	SPH_C64(0xC4C4F3AA6EF3C451), SPH_C64(0xFDFD2E46BB2EFD1A),
	SPH_C64(0xBFBF792E9179BFDC), SPH_C64(0x4444495E1A4944CC),
	SPH_C64(0xA1A11FC0611FA1FE), SPH_C64(0x4C4C61165A614CD4),
	SPH_C64(0x3333FFB685FF3355), SPH_C64(0xC5C5F6A366F6C552),
	SPH_C64(0x8484AED054AE8491), SPH_C64(0x2323AF2605AF2365),
	SPH_C64(0x7C7C91BBC7917C84), SPH_C64(0xB0B04A59E94AB0CD),
	SPH_C64(0x2525B11035B1256F), SPH_C64(0x151541BDA841153F),
	SPH_C64(0x3535E180B5E1355F), SPH_C64(0x6969D0066FD069BB),
	SPH_C64(0xFFFF2454AB24FF1C), SPH_C64(0x9494FE40D4FE94A1),
	SPH_C64(0x4D4D641F52644DD7), SPH_C64(0x7070ADD7A7AD7090),
	SPH_C64(0xA2A210DB7910A2FB), SPH_C64(0xAFAF29BE1129AFEC),
	SPH_C64(0xCDCDDEEB26DECD4A), SPH_C64(0xD6D6A928FEA9D667),
	SPH_C64(0x6C6CC12B47C16CB4), SPH_C64(0xB7B75166D151B7C4),
	SPH_C64(0xF8F83F6B933FF815), SPH_C64(0x09092D41482D091B),
	SPH_C64(0xF3F31838CB18F308), SPH_C64(0x6767E6781FE667A9),
	SPH_C64(0xA4A40EED490EA4F1), SPH_C64(0xEAEA65E90365EA23),
	SPH_C64(0xECEC7BDF337BEC29), SPH_C64(0xB6B6546FD954B6C7),
	SPH_C64(0xD4D4A33AEEA3D461), SPH_C64(0xD2D2BD0CDEBDD26B),
	SPH_C64(0x141444B4A044143C), SPH_C64(0x1E1E66EEF0661E22),
	SPH_C64(0xE1E142BA5B42E13E), SPH_C64(0x2424B4193DB4246C),
	SPH_C64(0x3838D8E5DDD83848), SPH_C64(0xC6C6F9B87EF9C657),
	SPH_C64(0xDBDB904D9690DB70), SPH_C64(0x4B4B7A29627A4BDD),
	SPH_C64(0x7A7A8F8DF78F7A8E), SPH_C64(0x3A3AD2F7CDD23A4E),
	SPH_C64(0xDEDE8160BE81DE7F), SPH_C64(0x5E5E3B94CA3B5EE2),
	SPH_C64(0xDFDF8469B684DF7C), SPH_C64(0x9595FB49DCFB95A2),
	SPH_C64(0xFCFC2B4FB32BFC19), SPH_C64(0xAAAA38933938AAE3),
	SPH_C64(0xD7D7AC21F6ACD764), SPH_C64(0xCECED1F03ED1CE4F),
	SPH_C64(0x07071B3F381B0709), SPH_C64(0x0F0F337778330F11),
	SPH_C64(0x3D3DC9C8F5C93D47), SPH_C64(0x585825A2FA2558E8),
	SPH_C64(0x9A9AC83EA4C89AB3), SPH_C64(0x9898C22CB4C298B5),
	SPH_C64(0x9C9CD60894D69CB9), SPH_C64(0xF2F21D31C31DF20B),
	SPH_C64(0xA7A701F65101A7F4), SPH_C64(0x1111559988551133),
	SPH_C64(0x7E7E9BA9D79B7E82), SPH_C64(0x8B8B9DA72C9D8B80),
	SPH_C64(0x43435261225243C5), SPH_C64(0x03030F1B180F0305),
	SPH_C64(0xE2E24DA1434DE23B), SPH_C64(0xDCDC8B72AE8BDC79),
	SPH_C64(0xE5E5569E7B56E532), SPH_C64(0xB2B2404BF940B2CB),
	SPH_C64(0x4E4E6B044A6B4ED2), SPH_C64(0xC7C7FCB176FCC754),
	SPH_C64(0x6D6DC4224FC46DB7), SPH_C64(0xE9E96AF21B6AE926),
	SPH_C64(0x2727BB0225BB2769), SPH_C64(0x40405D7A3A5D40C0),
	SPH_C64(0xD8D89F568E9FD875), SPH_C64(0x3737EB92A5EB3759),
	SPH_C64(0x9292E076E4E092AB), SPH_C64(0x8F8F89830C898F8C),
	SPH_C64(0x0101050908050103), SPH_C64(0x1D1D69F5E8691D27),
	SPH_C64(0x535302F1A20253F5), SPH_C64(0x3E3EC6D3EDC63E42),
	SPH_C64(0x595920ABF22059EB), SPH_C64(0xC1C1E28746E2C15E),
	SPH_C64(0x4F4F6E0D426E4FD1), SPH_C64(0x3232FABF8DFA3256),
	SPH_C64(0x16164EA6B04E163A), SPH_C64(0xFAFA35798335FA13),
	SPH_C64(0x7474B9F387B9749C), SPH_C64(0xFBFB30708B30FB10),
	SPH_C64(0x6363F25C3FF263A5), SPH_C64(0x9F9FD9138CD99FBC),
	SPH_C64(0x3434E489BDE4345C), SPH_C64(0x1A1A72CAD0721A2E),
	SPH_C64(0x2A2A82674D822A7E), SPH_C64(0x5A5A2FB0EA2F5AEE),
	SPH_C64(0x8D8D83911C838D8A), SPH_C64(0xC9C9CACF06CAC946),
	SPH_C64(0xCFCFD4F936D4CF4C), SPH_C64(0xF6F60915E309F607),
	SPH_C64(0x9090EA64F4EA90AD), SPH_C64(0x282888755D882878),
	SPH_C64(0x888892BC34928885), SPH_C64(0x9B9BCD37ACCD9BB0),
	SPH_C64(0x3131F5A495F53153), SPH_C64(0x0E0E367E70360E12),
	SPH_C64(0xBDBD733C8173BDDA), SPH_C64(0x4A4A7F206A7F4ADE),
	SPH_C64(0xE8E86FFB136FE825), SPH_C64(0x9696F452C4F496A7),
	SPH_C64(0xA6A604FF5904A6F7), SPH_C64(0x0C0C3C6C603C0C14),
	SPH_C64(0xC8C8CFC60ECFC845), SPH_C64(0x79798096EF80798B),
	SPH_C64(0xBCBC76358976BCD9), SPH_C64(0xBEBE7C27997CBEDF),
	SPH_C64(0xEFEF74C42B74EF2C), SPH_C64(0x6E6ECB3957CB6EB2),
	SPH_C64(0x4646434C0A4346CA), SPH_C64(0x9797F15BCCF197A4),
	SPH_C64(0x5B5B2AB9E22A5BED), SPH_C64(0xEDED7ED63B7EED2A),
	SPH_C64(0x19197DD1C87D192B), SPH_C64(0xD9D99A5F869AD976),
	SPH_C64(0xACAC26A50926ACE9), SPH_C64(0x9999C725BCC799B6),
	SPH_C64(0xA8A832812932A8E5), SPH_C64(0x29298D7C558D297B),
	SPH_C64(0x6464E96307E964AC), SPH_C64(0x1F1F63E7F8631F21),
	SPH_C64(0xADAD23AC0123ADEA), SPH_C64(0x55551CC7921C55FF),
	SPH_C64(0x13135F8B985F1335), SPH_C64(0xBBBB6D0AB16DBBD0),
	SPH_C64(0xF7F70C1CEB0CF704), SPH_C64(0x6F6FCE305FCE6FB1),
	SPH_C64(0xB9B96718A167B9D6), SPH_C64(0x47474645024647C9),
	SPH_C64(0x2F2F934A65932F71), SPH_C64(0xEEEE71CD2371EE2F),
	SPH_C64(0xB8B86211A962B8D5), SPH_C64(0x7B7B8A84FF8A7B8D),
	SPH_C64(0x898997B53C978986), SPH_C64(0x3030F0AD9DF03050),
	SPH_C64(0xD3D3B805D6B8D368), SPH_C64(0x7F7F9EA0DF9E7F81),
	SPH_C64(0x7676B3E197B3769A), SPH_C64(0x8282B0E664B0829B)
};

static const sph_u64 old0_T7[256] = {
	SPH_C64(0x68D50F67D568B868), SPH_C64(0xD0B71ECEB7D06DD0),
	SPH_C64(0xEB60E00B60EB20EB), SPH_C64(0x2B876E45872B7D2B),
	SPH_C64(0x4875327A7548D848), SPH_C64(0x9DD3019CD39DBA9D),
	SPH_C64(0x6ADF1D77DF6ABE6A), SPH_C64(0xE453977353E431E4),
	SPH_C64(0xE348A84B48E338E3), SPH_C64(0xA315D27115A3F8A3),
	SPH_C64(0x5613DC8A1356FA56), SPH_C64(0x81BFFD7CBF819E81),
	SPH_C64(0x7D94B2CF947D877D), SPH_C64(0xF1122ADB12F10EF1),
	SPH_C64(0x85ABD95CAB859285), SPH_C64(0x9EDC1A84DC9EBF9E),
	SPH_C64(0x2C9C517D9C2C742C), SPH_C64(0x8E8C8A048C8E8F8E),
	SPH_C64(0x78859FE785788878), SPH_C64(0xCAC5D41EC5CA43CA),
	SPH_C64(0x174BAFB84B173917), SPH_C64(0xA937882137A9E6A9),
	SPH_C64(0x61F84E2FF861A361), SPH_C64(0xD5A633E6A6D562D5),
	SPH_C64(0x5D348FD2345DE75D), SPH_C64(0x0B275358270B1D0B),
	SPH_C64(0x8C869814868C898C), SPH_C64(0x3CCCC1FDCC3C443C),
	SPH_C64(0x77B6E89FB6779977), SPH_C64(0x5108E3B20851F351),
	SPH_C64(0x22AA2F0DAA226622), SPH_C64(0x4257682A5742C642),
	SPH_C64(0x3FC3DAE5C33F413F), SPH_C64(0x5419CE9A1954FC54),
	SPH_C64(0x415873325841C341), SPH_C64(0x80BAF474BA809D80),
	SPH_C64(0xCCDBE22EDBCC49CC), SPH_C64(0x86A4C244A4869786),
	SPH_C64(0xB34542F145B3C8B3), SPH_C64(0x1878D8C078182818),
	SPH_C64(0x2E96436D962E722E), SPH_C64(0x5716D5821657F957),
	SPH_C64(0x061E36301E060A06), SPH_C64(0x62F75537F762A662),
	SPH_C64(0xF40307F303F401F4), SPH_C64(0x36EE9BADEE365A36),
	SPH_C64(0xD1B217C6B2D16ED1), SPH_C64(0x6BDA147FDA6BBD6B),
	SPH_C64(0x1B77C3D8771B2D1B), SPH_C64(0x65EC6A0FEC65AF65),
	SPH_C64(0x75BCFA8FBC759F75), SPH_C64(0x1050908050103010),
	SPH_C64(0xDA95449E95DA73DA), SPH_C64(0x49703B727049DB49),
	SPH_C64(0x26BE0B2DBE266A26), SPH_C64(0xF93A629B3AF916F9),
	SPH_C64(0xCBC0DD16C0CB40CB), SPH_C64(0x66E37117E366AA66),
	SPH_C64(0xE75C8C6B5CE734E7), SPH_C64(0xBA6803B968BAD3BA),
	SPH_C64(0xAE2CB7192CAEEFAE), SPH_C64(0x500DEABA0D50F050),
	SPH_C64(0x5207F8AA0752F652), SPH_C64(0xAB3D9A313DABE0AB),
	SPH_C64(0x05112D2811050F05), SPH_C64(0xF01723D317F00DF0),
	SPH_C64(0x0D396568390D170D), SPH_C64(0x73A2CCBFA2739573),
	SPH_C64(0x3BD7FEC5D73B4D3B), SPH_C64(0x0414242014040C04),
	SPH_C64(0x20A03D1DA0206020), SPH_C64(0xFE215DA321FE1FFE),
	SPH_C64(0xDD8E7BA68EDD7ADD), SPH_C64(0xF5060EFB06F502F5),
	SPH_C64(0xB45E7DC95EB4C1B4), SPH_C64(0x5F3E9DC23E5FE15F),
	SPH_C64(0x0A225A50220A1E0A), SPH_C64(0xB55B74C15BB5C2B5),
	SPH_C64(0xC0E78E4EE7C05DC0), SPH_C64(0xA01AC9691AA0FDA0),
	SPH_C64(0x71A8DEAFA8719371), SPH_C64(0xA50BE4410BA5F2A5),
	SPH_C64(0x2D995875992D772D), SPH_C64(0x60FD4727FD60A060),
	SPH_C64(0x72A7C5B7A7729672), SPH_C64(0x93E57FECE593A893),
	SPH_C64(0x39DDECD5DD394B39), SPH_C64(0x0828484028081808),
	SPH_C64(0x83B5EF6CB5839883), SPH_C64(0x21A53415A5216321),
	SPH_C64(0x5C3186DA315CE45C), SPH_C64(0x87A1CB4CA1879487),
	SPH_C64(0xB14F50E14FB1CEB1), SPH_C64(0xE047B35347E03DE0),
	SPH_C64(0x0000000000000000), SPH_C64(0xC3E89556E8C358C3),
	SPH_C64(0x125A82905A123612), SPH_C64(0x91EF6DFCEF91AE91),
	SPH_C64(0x8A98AE24988A838A), SPH_C64(0x020A12100A020602),
	SPH_C64(0x1C6CFCE06C1C241C), SPH_C64(0xE659856359E637E6),
	SPH_C64(0x454C57124C45CF45), SPH_C64(0xC2ED9C5EEDC25BC2),
	SPH_C64(0xC4F3AA6EF3C451C4), SPH_C64(0xFD2E46BB2EFD1AFD),
	SPH_C64(0xBF792E9179BFDCBF), SPH_C64(0x44495E1A4944CC44),
	SPH_C64(0xA11FC0611FA1FEA1), SPH_C64(0x4C61165A614CD44C),
	SPH_C64(0x33FFB685FF335533), SPH_C64(0xC5F6A366F6C552C5),
	SPH_C64(0x84AED054AE849184), SPH_C64(0x23AF2605AF236523),
	SPH_C64(0x7C91BBC7917C847C), SPH_C64(0xB04A59E94AB0CDB0),
	SPH_C64(0x25B11035B1256F25), SPH_C64(0x1541BDA841153F15),
	SPH_C64(0x35E180B5E1355F35), SPH_C64(0x69D0066FD069BB69),
	SPH_C64(0xFF2454AB24FF1CFF), SPH_C64(0x94FE40D4FE94A194),
	SPH_C64(0x4D641F52644DD74D), SPH_C64(0x70ADD7A7AD709070),
	SPH_C64(0xA210DB7910A2FBA2), SPH_C64(0xAF29BE1129AFECAF),
	SPH_C64(0xCDDEEB26DECD4ACD), SPH_C64(0xD6A928FEA9D667D6),
	SPH_C64(0x6CC12B47C16CB46C), SPH_C64(0xB75166D151B7C4B7),
	SPH_C64(0xF83F6B933FF815F8), SPH_C64(0x092D41482D091B09),
	SPH_C64(0xF31838CB18F308F3), SPH_C64(0x67E6781FE667A967),
	SPH_C64(0xA40EED490EA4F1A4), SPH_C64(0xEA65E90365EA23EA),
	SPH_C64(0xEC7BDF337BEC29EC), SPH_C64(0xB6546FD954B6C7B6),
	SPH_C64(0xD4A33AEEA3D461D4), SPH_C64(0xD2BD0CDEBDD26BD2),
	SPH_C64(0x1444B4A044143C14), SPH_C64(0x1E66EEF0661E221E),
	SPH_C64(0xE142BA5B42E13EE1), SPH_C64(0x24B4193DB4246C24),
	SPH_C64(0x38D8E5DDD8384838), SPH_C64(0xC6F9B87EF9C657C6),
	SPH_C64(0xDB904D9690DB70DB), SPH_C64(0x4B7A29627A4BDD4B),
	SPH_C64(0x7A8F8DF78F7A8E7A), SPH_C64(0x3AD2F7CDD23A4E3A),
	SPH_C64(0xDE8160BE81DE7FDE), SPH_C64(0x5E3B94CA3B5EE25E),
	SPH_C64(0xDF8469B684DF7CDF), SPH_C64(0x95FB49DCFB95A295),
	SPH_C64(0xFC2B4FB32BFC19FC), SPH_C64(0xAA38933938AAE3AA),
	SPH_C64(0xD7AC21F6ACD764D7), SPH_C64(0xCED1F03ED1CE4FCE),
	SPH_C64(0x071B3F381B070907), SPH_C64(0x0F337778330F110F),
	SPH_C64(0x3DC9C8F5C93D473D), SPH_C64(0x5825A2FA2558E858),
	SPH_C64(0x9AC83EA4C89AB39A), SPH_C64(0x98C22CB4C298B598),
	SPH_C64(0x9CD60894D69CB99C), SPH_C64(0xF21D31C31DF20BF2),
	SPH_C64(0xA701F65101A7F4A7), SPH_C64(0x1155998855113311),
	SPH_C64(0x7E9BA9D79B7E827E), SPH_C64(0x8B9DA72C9D8B808B),
	SPH_C64(0x435261225243C543), SPH_C64(0x030F1B180F030503),
	SPH_C64(0xE24DA1434DE23BE2), SPH_C64(0xDC8B72AE8BDC79DC),
	SPH_C64(0xE5569E7B56E532E5), SPH_C64(0xB2404BF940B2CBB2),
	SPH_C64(0x4E6B044A6B4ED24E), SPH_C64(0xC7FCB176FCC754C7),
	SPH_C64(0x6DC4224FC46DB76D), SPH_C64(0xE96AF21B6AE926E9),
	SPH_C64(0x27BB0225BB276927), SPH_C64(0x405D7A3A5D40C040),
	SPH_C64(0xD89F568E9FD875D8), SPH_C64(0x37EB92A5EB375937),
	SPH_C64(0x92E076E4E092AB92), SPH_C64(0x8F89830C898F8C8F),
	SPH_C64(0x0105090805010301), SPH_C64(0x1D69F5E8691D271D),
	SPH_C64(0x5302F1A20253F553), SPH_C64(0x3EC6D3EDC63E423E),
	SPH_C64(0x5920ABF22059EB59), SPH_C64(0xC1E28746E2C15EC1),
	SPH_C64(0x4F6E0D426E4FD14F), SPH_C64(0x32FABF8DFA325632),
	SPH_C64(0x164EA6B04E163A16), SPH_C64(0xFA35798335FA13FA),
	SPH_C64(0x74B9F387B9749C74), SPH_C64(0xFB30708B30FB10FB),
	SPH_C64(0x63F25C3FF263A563), SPH_C64(0x9FD9138CD99FBC9F),
	SPH_C64(0x34E489BDE4345C34), SPH_C64(0x1A72CAD0721A2E1A),
	SPH_C64(0x2A82674D822A7E2A), SPH_C64(0x5A2FB0EA2F5AEE5A),
	SPH_C64(0x8D83911C838D8A8D), SPH_C64(0xC9CACF06CAC946C9),
	SPH_C64(0xCFD4F936D4CF4CCF), SPH_C64(0xF60915E309F607F6),
	SPH_C64(0x90EA64F4EA90AD90), SPH_C64(0x2888755D88287828),
	SPH_C64(0x8892BC3492888588), SPH_C64(0x9BCD37ACCD9BB09B),
	SPH_C64(0x31F5A495F5315331), SPH_C64(0x0E367E70360E120E),
	SPH_C64(0xBD733C8173BDDABD), SPH_C64(0x4A7F206A7F4ADE4A),
	SPH_C64(0xE86FFB136FE825E8), SPH_C64(0x96F452C4F496A796),
	SPH_C64(0xA604FF5904A6F7A6), SPH_C64(0x0C3C6C603C0C140C),
	SPH_C64(0xC8CFC60ECFC845C8), SPH_C64(0x798096EF80798B79),
	SPH_C64(0xBC76358976BCD9BC), SPH_C64(0xBE7C27997CBEDFBE),
	SPH_C64(0xEF74C42B74EF2CEF), SPH_C64(0x6ECB3957CB6EB26E),
	SPH_C64(0x46434C0A4346CA46), SPH_C64(0x97F15BCCF197A497),
	SPH_C64(0x5B2AB9E22A5BED5B), SPH_C64(0xED7ED63B7EED2AED),
	SPH_C64(0x197DD1C87D192B19), SPH_C64(0xD99A5F869AD976D9),
	SPH_C64(0xAC26A50926ACE9AC), SPH_C64(0x99C725BCC799B699),
	SPH_C64(0xA832812932A8E5A8), SPH_C64(0x298D7C558D297B29),
	SPH_C64(0x64E96307E964AC64), SPH_C64(0x1F63E7F8631F211F),
	SPH_C64(0xAD23AC0123ADEAAD), SPH_C64(0x551CC7921C55FF55),
	SPH_C64(0x135F8B985F133513), SPH_C64(0xBB6D0AB16DBBD0BB),
	SPH_C64(0xF70C1CEB0CF704F7), SPH_C64(0x6FCE305FCE6FB16F),
	SPH_C64(0xB96718A167B9D6B9), SPH_C64(0x474645024647C947),
	SPH_C64(0x2F934A65932F712F), SPH_C64(0xEE71CD2371EE2FEE),
	SPH_C64(0xB86211A962B8D5B8), SPH_C64(0x7B8A84FF8A7B8D7B),
	SPH_C64(0x8997B53C97898689), SPH_C64(0x30F0AD9DF0305030),
	SPH_C64(0xD3B805D6B8D368D3), SPH_C64(0x7F9EA0DF9E7F817F),
	SPH_C64(0x76B3E197B3769A76), SPH_C64(0x82B0E664B0829B82)
};

#endif

static const sph_u64 old0_RC[10] = {
	SPH_C64(0xE46A9D482BEBD068),
	SPH_C64(0x9E85F17D8156A3E3),
	SPH_C64(0xD561A917CA788E2C),
	SPH_C64(0x422251773C8C0B5D),
	SPH_C64(0x18B386CC8041543F),
	SPH_C64(0x6BD136F46206572E),
	SPH_C64(0xF92649DA1075651B),
	SPH_C64(0xAB5250AEBAE766CB),
	SPH_C64(0xFE20043B730DF005),
	SPH_C64(0xA0C0B50A5FB4F5DD)
};

/* ====================================================================== */
/*
 * Constants for plain WHIRLPOOL-1 (second version).
 */

static const sph_u64 old1_T0[256] = {
	SPH_C64(0x78D8C07818281818), SPH_C64(0xAF2605AF23652323),
	SPH_C64(0xF9B87EF9C657C6C6), SPH_C64(0x6FFB136FE825E8E8),
	SPH_C64(0xA1CB4CA187948787), SPH_C64(0x6211A962B8D5B8B8),
	SPH_C64(0x0509080501030101), SPH_C64(0x6E0D426E4FD14F4F),
	SPH_C64(0xEE9BADEE365A3636), SPH_C64(0x04FF5904A6F7A6A6),
	SPH_C64(0xBD0CDEBDD26BD2D2), SPH_C64(0x060EFB06F502F5F5),
	SPH_C64(0x8096EF80798B7979), SPH_C64(0xCE305FCE6FB16F6F),
	SPH_C64(0xEF6DFCEF91AE9191), SPH_C64(0x07F8AA0752F65252),
	SPH_C64(0xFD4727FD60A06060), SPH_C64(0x76358976BCD9BCBC),
	SPH_C64(0xCD37ACCD9BB09B9B), SPH_C64(0x8C8A048C8E8F8E8E),
	SPH_C64(0x15D27115A3F8A3A3), SPH_C64(0x3C6C603C0C140C0C),
	SPH_C64(0x8A84FF8A7B8D7B7B), SPH_C64(0xE180B5E1355F3535),
	SPH_C64(0x69F5E8691D271D1D), SPH_C64(0x47B35347E03DE0E0),
	SPH_C64(0xAC21F6ACD764D7D7), SPH_C64(0xED9C5EEDC25BC2C2),
	SPH_C64(0x96436D962E722E2E), SPH_C64(0x7A29627A4BDD4B4B),
	SPH_C64(0x215DA321FE1FFEFE), SPH_C64(0x16D5821657F95757),
	SPH_C64(0x41BDA841153F1515), SPH_C64(0xB6E89FB677997777),
	SPH_C64(0xEB92A5EB37593737), SPH_C64(0x569E7B56E532E5E5),
	SPH_C64(0xD9138CD99FBC9F9F), SPH_C64(0x1723D317F00DF0F0),
	SPH_C64(0x7F206A7F4ADE4A4A), SPH_C64(0x95449E95DA73DADA),
	SPH_C64(0x25A2FA2558E85858), SPH_C64(0xCACF06CAC946C9C9),
	SPH_C64(0x8D7C558D297B2929), SPH_C64(0x225A50220A1E0A0A),
	SPH_C64(0x4F50E14FB1CEB1B1), SPH_C64(0x1AC9691AA0FDA0A0),
	SPH_C64(0xDA147FDA6BBD6B6B), SPH_C64(0xABD95CAB85928585),
	SPH_C64(0x733C8173BDDABDBD), SPH_C64(0x348FD2345DE75D5D),
	SPH_C64(0x5090805010301010), SPH_C64(0x0307F303F401F4F4),
	SPH_C64(0xC0DD16C0CB40CBCB), SPH_C64(0xC6D3EDC63E423E3E),
	SPH_C64(0x112D2811050F0505), SPH_C64(0xE6781FE667A96767),
	SPH_C64(0x53977353E431E4E4), SPH_C64(0xBB0225BB27692727),
	SPH_C64(0x5873325841C34141), SPH_C64(0x9DA72C9D8B808B8B),
	SPH_C64(0x01F65101A7F4A7A7), SPH_C64(0x94B2CF947D877D7D),
	SPH_C64(0xFB49DCFB95A29595), SPH_C64(0x9F568E9FD875D8D8),
	SPH_C64(0x30708B30FB10FBFB), SPH_C64(0x71CD2371EE2FEEEE),
	SPH_C64(0x91BBC7917C847C7C), SPH_C64(0xE37117E366AA6666),
	SPH_C64(0x8E7BA68EDD7ADDDD), SPH_C64(0x4BAFB84B17391717),
	SPH_C64(0x4645024647C94747), SPH_C64(0xDC1A84DC9EBF9E9E),
	SPH_C64(0xC5D41EC5CA43CACA), SPH_C64(0x995875992D772D2D),
	SPH_C64(0x792E9179BFDCBFBF), SPH_C64(0x1B3F381B07090707),
	SPH_C64(0x23AC0123ADEAADAD), SPH_C64(0x2FB0EA2F5AEE5A5A),
	SPH_C64(0xB5EF6CB583988383), SPH_C64(0xFFB685FF33553333),
	SPH_C64(0xF25C3FF263A56363), SPH_C64(0x0A12100A02060202),
	SPH_C64(0x38933938AAE3AAAA), SPH_C64(0xA8DEAFA871937171),
	SPH_C64(0xCFC60ECFC845C8C8), SPH_C64(0x7DD1C87D192B1919),
	SPH_C64(0x703B727049DB4949), SPH_C64(0x9A5F869AD976D9D9),
	SPH_C64(0x1D31C31DF20BF2F2), SPH_C64(0x48A84B48E338E3E3),
	SPH_C64(0x2AB9E22A5BED5B5B), SPH_C64(0x92BC349288858888),
	SPH_C64(0xC83EA4C89AB39A9A), SPH_C64(0xBE0B2DBE266A2626),
	SPH_C64(0xFABF8DFA32563232), SPH_C64(0x4A59E94AB0CDB0B0),
	SPH_C64(0x6AF21B6AE926E9E9), SPH_C64(0x337778330F110F0F),
	SPH_C64(0xA633E6A6D562D5D5), SPH_C64(0xBAF474BA809D8080),
	SPH_C64(0x7C27997CBEDFBEBE), SPH_C64(0xDEEB26DECD4ACDCD),
	SPH_C64(0xE489BDE4345C3434), SPH_C64(0x75327A7548D84848),
	SPH_C64(0x2454AB24FF1CFFFF), SPH_C64(0x8F8DF78F7A8E7A7A),
	SPH_C64(0xEA64F4EA90AD9090), SPH_C64(0x3E9DC23E5FE15F5F),
	SPH_C64(0xA03D1DA020602020), SPH_C64(0xD50F67D568B86868),
	SPH_C64(0x72CAD0721A2E1A1A), SPH_C64(0x2CB7192CAEEFAEAE),
	SPH_C64(0x5E7DC95EB4C1B4B4), SPH_C64(0x19CE9A1954FC5454),
	SPH_C64(0xE57FECE593A89393), SPH_C64(0xAA2F0DAA22662222),
	SPH_C64(0xE96307E964AC6464), SPH_C64(0x122ADB12F10EF1F1),
	SPH_C64(0xA2CCBFA273957373), SPH_C64(0x5A82905A12361212),
	SPH_C64(0x5D7A3A5D40C04040), SPH_C64(0x2848402808180808),
	SPH_C64(0xE89556E8C358C3C3), SPH_C64(0x7BDF337BEC29ECEC),
	SPH_C64(0x904D9690DB70DBDB), SPH_C64(0x1FC0611FA1FEA1A1),
	SPH_C64(0x83911C838D8A8D8D), SPH_C64(0xC9C8F5C93D473D3D),
	SPH_C64(0xF15BCCF197A49797), SPH_C64(0x0000000000000000),
	SPH_C64(0xD4F936D4CF4CCFCF), SPH_C64(0x876E45872B7D2B2B),
	SPH_C64(0xB3E197B3769A7676), SPH_C64(0xB0E664B0829B8282),
	SPH_C64(0xA928FEA9D667D6D6), SPH_C64(0x77C3D8771B2D1B1B),
	SPH_C64(0x5B74C15BB5C2B5B5), SPH_C64(0x29BE1129AFECAFAF),
	SPH_C64(0xDF1D77DF6ABE6A6A), SPH_C64(0x0DEABA0D50F05050),
	SPH_C64(0x4C57124C45CF4545), SPH_C64(0x1838CB18F308F3F3),
	SPH_C64(0xF0AD9DF030503030), SPH_C64(0x74C42B74EF2CEFEF),
	SPH_C64(0xC3DAE5C33F413F3F), SPH_C64(0x1CC7921C55FF5555),
	SPH_C64(0x10DB7910A2FBA2A2), SPH_C64(0x65E90365EA23EAEA),
	SPH_C64(0xEC6A0FEC65AF6565), SPH_C64(0x6803B968BAD3BABA),
	SPH_C64(0x934A65932F712F2F), SPH_C64(0xE78E4EE7C05DC0C0),
	SPH_C64(0x8160BE81DE7FDEDE), SPH_C64(0x6CFCE06C1C241C1C),
	SPH_C64(0x2E46BB2EFD1AFDFD), SPH_C64(0x641F52644DD74D4D),
	SPH_C64(0xE076E4E092AB9292), SPH_C64(0xBCFA8FBC759F7575),
	SPH_C64(0x1E36301E060A0606), SPH_C64(0x98AE24988A838A8A),
	SPH_C64(0x404BF940B2CBB2B2), SPH_C64(0x59856359E637E6E6),
	SPH_C64(0x367E70360E120E0E), SPH_C64(0x63E7F8631F211F1F),
	SPH_C64(0xF75537F762A66262), SPH_C64(0xA33AEEA3D461D4D4),
	SPH_C64(0x32812932A8E5A8A8), SPH_C64(0xF452C4F496A79696),
	SPH_C64(0x3A629B3AF916F9F9), SPH_C64(0xF6A366F6C552C5C5),
	SPH_C64(0xB11035B1256F2525), SPH_C64(0x20ABF22059EB5959),
	SPH_C64(0xAED054AE84918484), SPH_C64(0xA7C5B7A772967272),
	SPH_C64(0xDDECD5DD394B3939), SPH_C64(0x61165A614CD44C4C),
	SPH_C64(0x3B94CA3B5EE25E5E), SPH_C64(0x859FE78578887878),
	SPH_C64(0xD8E5DDD838483838), SPH_C64(0x869814868C898C8C),
	SPH_C64(0xB217C6B2D16ED1D1), SPH_C64(0x0BE4410BA5F2A5A5),
	SPH_C64(0x4DA1434DE23BE2E2), SPH_C64(0xF84E2FF861A36161),
	SPH_C64(0x4542F145B3C8B3B3), SPH_C64(0xA53415A521632121),
	SPH_C64(0xD60894D69CB99C9C), SPH_C64(0x66EEF0661E221E1E),
	SPH_C64(0x5261225243C54343), SPH_C64(0xFCB176FCC754C7C7),
	SPH_C64(0x2B4FB32BFC19FCFC), SPH_C64(0x14242014040C0404),
	SPH_C64(0x08E3B20851F35151), SPH_C64(0xC725BCC799B69999),
	SPH_C64(0xC4224FC46DB76D6D), SPH_C64(0x396568390D170D0D),
	SPH_C64(0x35798335FA13FAFA), SPH_C64(0x8469B684DF7CDFDF),
	SPH_C64(0x9BA9D79B7E827E7E), SPH_C64(0xB4193DB4246C2424),
	SPH_C64(0xD7FEC5D73B4D3B3B), SPH_C64(0x3D9A313DABE0ABAB),
	SPH_C64(0xD1F03ED1CE4FCECE), SPH_C64(0x5599885511331111),
	SPH_C64(0x89830C898F8C8F8F), SPH_C64(0x6B044A6B4ED24E4E),
	SPH_C64(0x5166D151B7C4B7B7), SPH_C64(0x60E00B60EB20EBEB),
	SPH_C64(0xCCC1FDCC3C443C3C), SPH_C64(0xBFFD7CBF819E8181),
	SPH_C64(0xFE40D4FE94A19494), SPH_C64(0x0C1CEB0CF704F7F7),
	SPH_C64(0x6718A167B9D6B9B9), SPH_C64(0x5F8B985F13351313),
	SPH_C64(0x9C517D9C2C742C2C), SPH_C64(0xB805D6B8D368D3D3),
	SPH_C64(0x5C8C6B5CE734E7E7), SPH_C64(0xCB3957CB6EB26E6E),
	SPH_C64(0xF3AA6EF3C451C4C4), SPH_C64(0x0F1B180F03050303),
	SPH_C64(0x13DC8A1356FA5656), SPH_C64(0x495E1A4944CC4444),
	SPH_C64(0x9EA0DF9E7F817F7F), SPH_C64(0x37882137A9E6A9A9),
	SPH_C64(0x82674D822A7E2A2A), SPH_C64(0x6D0AB16DBBD0BBBB),
	SPH_C64(0xE28746E2C15EC1C1), SPH_C64(0x02F1A20253F55353),
	SPH_C64(0x8B72AE8BDC79DCDC), SPH_C64(0x275358270B1D0B0B),
	SPH_C64(0xD3019CD39DBA9D9D), SPH_C64(0xC12B47C16CB46C6C),
	SPH_C64(0xF5A495F531533131), SPH_C64(0xB9F387B9749C7474),
	SPH_C64(0x0915E309F607F6F6), SPH_C64(0x434C0A4346CA4646),
	SPH_C64(0x26A50926ACE9ACAC), SPH_C64(0x97B53C9789868989),
	SPH_C64(0x44B4A044143C1414), SPH_C64(0x42BA5B42E13EE1E1),
	SPH_C64(0x4EA6B04E163A1616), SPH_C64(0xD2F7CDD23A4E3A3A),
	SPH_C64(0xD0066FD069BB6969), SPH_C64(0x2D41482D091B0909),
	SPH_C64(0xADD7A7AD70907070), SPH_C64(0x546FD954B6C7B6B6),
	SPH_C64(0xB71ECEB7D06DD0D0), SPH_C64(0x7ED63B7EED2AEDED),
	SPH_C64(0xDBE22EDBCC49CCCC), SPH_C64(0x57682A5742C64242),
	SPH_C64(0xC22CB4C298B59898), SPH_C64(0x0EED490EA4F1A4A4),
	SPH_C64(0x88755D8828782828), SPH_C64(0x3186DA315CE45C5C),
	SPH_C64(0x3F6B933FF815F8F8), SPH_C64(0xA4C244A486978686)
};

#if !SPH_SMALL_FOOTPRINT_WHIRLPOOL

static const sph_u64 old1_T1[256] = {
	SPH_C64(0xD8C0781828181878), SPH_C64(0x2605AF23652323AF),
	SPH_C64(0xB87EF9C657C6C6F9), SPH_C64(0xFB136FE825E8E86F),
	SPH_C64(0xCB4CA187948787A1), SPH_C64(0x11A962B8D5B8B862),
	SPH_C64(0x0908050103010105), SPH_C64(0x0D426E4FD14F4F6E),
	SPH_C64(0x9BADEE365A3636EE), SPH_C64(0xFF5904A6F7A6A604),
	SPH_C64(0x0CDEBDD26BD2D2BD), SPH_C64(0x0EFB06F502F5F506),
	SPH_C64(0x96EF80798B797980), SPH_C64(0x305FCE6FB16F6FCE),
	SPH_C64(0x6DFCEF91AE9191EF), SPH_C64(0xF8AA0752F6525207),
	SPH_C64(0x4727FD60A06060FD), SPH_C64(0x358976BCD9BCBC76),
	SPH_C64(0x37ACCD9BB09B9BCD), SPH_C64(0x8A048C8E8F8E8E8C),
	SPH_C64(0xD27115A3F8A3A315), SPH_C64(0x6C603C0C140C0C3C),
	SPH_C64(0x84FF8A7B8D7B7B8A), SPH_C64(0x80B5E1355F3535E1),
	SPH_C64(0xF5E8691D271D1D69), SPH_C64(0xB35347E03DE0E047),
	SPH_C64(0x21F6ACD764D7D7AC), SPH_C64(0x9C5EEDC25BC2C2ED),
	SPH_C64(0x436D962E722E2E96), SPH_C64(0x29627A4BDD4B4B7A),
	SPH_C64(0x5DA321FE1FFEFE21), SPH_C64(0xD5821657F9575716),
	SPH_C64(0xBDA841153F151541), SPH_C64(0xE89FB677997777B6),
	SPH_C64(0x92A5EB37593737EB), SPH_C64(0x9E7B56E532E5E556),
	SPH_C64(0x138CD99FBC9F9FD9), SPH_C64(0x23D317F00DF0F017),
	SPH_C64(0x206A7F4ADE4A4A7F), SPH_C64(0x449E95DA73DADA95),
	SPH_C64(0xA2FA2558E8585825), SPH_C64(0xCF06CAC946C9C9CA),
	SPH_C64(0x7C558D297B29298D), SPH_C64(0x5A50220A1E0A0A22),
	SPH_C64(0x50E14FB1CEB1B14F), SPH_C64(0xC9691AA0FDA0A01A),
	SPH_C64(0x147FDA6BBD6B6BDA), SPH_C64(0xD95CAB85928585AB),
	SPH_C64(0x3C8173BDDABDBD73), SPH_C64(0x8FD2345DE75D5D34),
	SPH_C64(0x9080501030101050), SPH_C64(0x07F303F401F4F403),
	SPH_C64(0xDD16C0CB40CBCBC0), SPH_C64(0xD3EDC63E423E3EC6),
	SPH_C64(0x2D2811050F050511), SPH_C64(0x781FE667A96767E6),
	SPH_C64(0x977353E431E4E453), SPH_C64(0x0225BB27692727BB),
	SPH_C64(0x73325841C3414158), SPH_C64(0xA72C9D8B808B8B9D),
	SPH_C64(0xF65101A7F4A7A701), SPH_C64(0xB2CF947D877D7D94),
	SPH_C64(0x49DCFB95A29595FB), SPH_C64(0x568E9FD875D8D89F),
	SPH_C64(0x708B30FB10FBFB30), SPH_C64(0xCD2371EE2FEEEE71),
	SPH_C64(0xBBC7917C847C7C91), SPH_C64(0x7117E366AA6666E3),
	SPH_C64(0x7BA68EDD7ADDDD8E), SPH_C64(0xAFB84B173917174B),
	SPH_C64(0x45024647C9474746), SPH_C64(0x1A84DC9EBF9E9EDC),
	SPH_C64(0xD41EC5CA43CACAC5), SPH_C64(0x5875992D772D2D99),
	SPH_C64(0x2E9179BFDCBFBF79), SPH_C64(0x3F381B070907071B),
	SPH_C64(0xAC0123ADEAADAD23), SPH_C64(0xB0EA2F5AEE5A5A2F),
	SPH_C64(0xEF6CB583988383B5), SPH_C64(0xB685FF33553333FF),
	SPH_C64(0x5C3FF263A56363F2), SPH_C64(0x12100A020602020A),
	SPH_C64(0x933938AAE3AAAA38), SPH_C64(0xDEAFA871937171A8),
	SPH_C64(0xC60ECFC845C8C8CF), SPH_C64(0xD1C87D192B19197D),
	SPH_C64(0x3B727049DB494970), SPH_C64(0x5F869AD976D9D99A),
	SPH_C64(0x31C31DF20BF2F21D), SPH_C64(0xA84B48E338E3E348),
	SPH_C64(0xB9E22A5BED5B5B2A), SPH_C64(0xBC34928885888892),
	SPH_C64(0x3EA4C89AB39A9AC8), SPH_C64(0x0B2DBE266A2626BE),
	SPH_C64(0xBF8DFA32563232FA), SPH_C64(0x59E94AB0CDB0B04A),
	SPH_C64(0xF21B6AE926E9E96A), SPH_C64(0x7778330F110F0F33),
	SPH_C64(0x33E6A6D562D5D5A6), SPH_C64(0xF474BA809D8080BA),
	SPH_C64(0x27997CBEDFBEBE7C), SPH_C64(0xEB26DECD4ACDCDDE),
	SPH_C64(0x89BDE4345C3434E4), SPH_C64(0x327A7548D8484875),
	SPH_C64(0x54AB24FF1CFFFF24), SPH_C64(0x8DF78F7A8E7A7A8F),
	SPH_C64(0x64F4EA90AD9090EA), SPH_C64(0x9DC23E5FE15F5F3E),
	SPH_C64(0x3D1DA020602020A0), SPH_C64(0x0F67D568B86868D5),
	SPH_C64(0xCAD0721A2E1A1A72), SPH_C64(0xB7192CAEEFAEAE2C),
	SPH_C64(0x7DC95EB4C1B4B45E), SPH_C64(0xCE9A1954FC545419),
	SPH_C64(0x7FECE593A89393E5), SPH_C64(0x2F0DAA22662222AA),
	SPH_C64(0x6307E964AC6464E9), SPH_C64(0x2ADB12F10EF1F112),
	SPH_C64(0xCCBFA273957373A2), SPH_C64(0x82905A123612125A),
	SPH_C64(0x7A3A5D40C040405D), SPH_C64(0x4840280818080828),
	SPH_C64(0x9556E8C358C3C3E8), SPH_C64(0xDF337BEC29ECEC7B),
	SPH_C64(0x4D9690DB70DBDB90), SPH_C64(0xC0611FA1FEA1A11F),
	SPH_C64(0x911C838D8A8D8D83), SPH_C64(0xC8F5C93D473D3DC9),
	SPH_C64(0x5BCCF197A49797F1), SPH_C64(0x0000000000000000),
	SPH_C64(0xF936D4CF4CCFCFD4), SPH_C64(0x6E45872B7D2B2B87),
	SPH_C64(0xE197B3769A7676B3), SPH_C64(0xE664B0829B8282B0),
	SPH_C64(0x28FEA9D667D6D6A9), SPH_C64(0xC3D8771B2D1B1B77),
	SPH_C64(0x74C15BB5C2B5B55B), SPH_C64(0xBE1129AFECAFAF29),
	SPH_C64(0x1D77DF6ABE6A6ADF), SPH_C64(0xEABA0D50F050500D),
	SPH_C64(0x57124C45CF45454C), SPH_C64(0x38CB18F308F3F318),
	SPH_C64(0xAD9DF030503030F0), SPH_C64(0xC42B74EF2CEFEF74),
	SPH_C64(0xDAE5C33F413F3FC3), SPH_C64(0xC7921C55FF55551C),
	SPH_C64(0xDB7910A2FBA2A210), SPH_C64(0xE90365EA23EAEA65),
	SPH_C64(0x6A0FEC65AF6565EC), SPH_C64(0x03B968BAD3BABA68),
	SPH_C64(0x4A65932F712F2F93), SPH_C64(0x8E4EE7C05DC0C0E7),
	SPH_C64(0x60BE81DE7FDEDE81), SPH_C64(0xFCE06C1C241C1C6C),
	SPH_C64(0x46BB2EFD1AFDFD2E), SPH_C64(0x1F52644DD74D4D64),
	SPH_C64(0x76E4E092AB9292E0), SPH_C64(0xFA8FBC759F7575BC),
	SPH_C64(0x36301E060A06061E), SPH_C64(0xAE24988A838A8A98),
	SPH_C64(0x4BF940B2CBB2B240), SPH_C64(0x856359E637E6E659),
	SPH_C64(0x7E70360E120E0E36), SPH_C64(0xE7F8631F211F1F63),
	SPH_C64(0x5537F762A66262F7), SPH_C64(0x3AEEA3D461D4D4A3),
	SPH_C64(0x812932A8E5A8A832), SPH_C64(0x52C4F496A79696F4),
	SPH_C64(0x629B3AF916F9F93A), SPH_C64(0xA366F6C552C5C5F6),
	SPH_C64(0x1035B1256F2525B1), SPH_C64(0xABF22059EB595920),
	SPH_C64(0xD054AE84918484AE), SPH_C64(0xC5B7A772967272A7),
	SPH_C64(0xECD5DD394B3939DD), SPH_C64(0x165A614CD44C4C61),
	SPH_C64(0x94CA3B5EE25E5E3B), SPH_C64(0x9FE7857888787885),
	SPH_C64(0xE5DDD838483838D8), SPH_C64(0x9814868C898C8C86),
	SPH_C64(0x17C6B2D16ED1D1B2), SPH_C64(0xE4410BA5F2A5A50B),
	SPH_C64(0xA1434DE23BE2E24D), SPH_C64(0x4E2FF861A36161F8),
	SPH_C64(0x42F145B3C8B3B345), SPH_C64(0x3415A521632121A5),
	SPH_C64(0x0894D69CB99C9CD6), SPH_C64(0xEEF0661E221E1E66),
	SPH_C64(0x61225243C5434352), SPH_C64(0xB176FCC754C7C7FC),
	SPH_C64(0x4FB32BFC19FCFC2B), SPH_C64(0x242014040C040414),
	SPH_C64(0xE3B20851F3515108), SPH_C64(0x25BCC799B69999C7),
	SPH_C64(0x224FC46DB76D6DC4), SPH_C64(0x6568390D170D0D39),
	SPH_C64(0x798335FA13FAFA35), SPH_C64(0x69B684DF7CDFDF84),
	SPH_C64(0xA9D79B7E827E7E9B), SPH_C64(0x193DB4246C2424B4),
	SPH_C64(0xFEC5D73B4D3B3BD7), SPH_C64(0x9A313DABE0ABAB3D),
	SPH_C64(0xF03ED1CE4FCECED1), SPH_C64(0x9988551133111155),
	SPH_C64(0x830C898F8C8F8F89), SPH_C64(0x044A6B4ED24E4E6B),
	SPH_C64(0x66D151B7C4B7B751), SPH_C64(0xE00B60EB20EBEB60),
	SPH_C64(0xC1FDCC3C443C3CCC), SPH_C64(0xFD7CBF819E8181BF),
	SPH_C64(0x40D4FE94A19494FE), SPH_C64(0x1CEB0CF704F7F70C),
	SPH_C64(0x18A167B9D6B9B967), SPH_C64(0x8B985F133513135F),
	SPH_C64(0x517D9C2C742C2C9C), SPH_C64(0x05D6B8D368D3D3B8),
	SPH_C64(0x8C6B5CE734E7E75C), SPH_C64(0x3957CB6EB26E6ECB),
	SPH_C64(0xAA6EF3C451C4C4F3), SPH_C64(0x1B180F030503030F),
	SPH_C64(0xDC8A1356FA565613), SPH_C64(0x5E1A4944CC444449),
	SPH_C64(0xA0DF9E7F817F7F9E), SPH_C64(0x882137A9E6A9A937),
	SPH_C64(0x674D822A7E2A2A82), SPH_C64(0x0AB16DBBD0BBBB6D),
	SPH_C64(0x8746E2C15EC1C1E2), SPH_C64(0xF1A20253F5535302),
	SPH_C64(0x72AE8BDC79DCDC8B), SPH_C64(0x5358270B1D0B0B27),
	SPH_C64(0x019CD39DBA9D9DD3), SPH_C64(0x2B47C16CB46C6CC1),
	SPH_C64(0xA495F531533131F5), SPH_C64(0xF387B9749C7474B9),
	SPH_C64(0x15E309F607F6F609), SPH_C64(0x4C0A4346CA464643),
	SPH_C64(0xA50926ACE9ACAC26), SPH_C64(0xB53C978986898997),
	SPH_C64(0xB4A044143C141444), SPH_C64(0xBA5B42E13EE1E142),
	SPH_C64(0xA6B04E163A16164E), SPH_C64(0xF7CDD23A4E3A3AD2),
	SPH_C64(0x066FD069BB6969D0), SPH_C64(0x41482D091B09092D),
	SPH_C64(0xD7A7AD70907070AD), SPH_C64(0x6FD954B6C7B6B654),
	SPH_C64(0x1ECEB7D06DD0D0B7), SPH_C64(0xD63B7EED2AEDED7E),
	SPH_C64(0xE22EDBCC49CCCCDB), SPH_C64(0x682A5742C6424257),
	SPH_C64(0x2CB4C298B59898C2), SPH_C64(0xED490EA4F1A4A40E),
	SPH_C64(0x755D882878282888), SPH_C64(0x86DA315CE45C5C31),
	SPH_C64(0x6B933FF815F8F83F), SPH_C64(0xC244A486978686A4)
};

static const sph_u64 old1_T2[256] = {
	SPH_C64(0xC0781828181878D8), SPH_C64(0x05AF23652323AF26),
	SPH_C64(0x7EF9C657C6C6F9B8), SPH_C64(0x136FE825E8E86FFB),
	SPH_C64(0x4CA187948787A1CB), SPH_C64(0xA962B8D5B8B86211),
	SPH_C64(0x0805010301010509), SPH_C64(0x426E4FD14F4F6E0D),
	SPH_C64(0xADEE365A3636EE9B), SPH_C64(0x5904A6F7A6A604FF),
	SPH_C64(0xDEBDD26BD2D2BD0C), SPH_C64(0xFB06F502F5F5060E),
	SPH_C64(0xEF80798B79798096), SPH_C64(0x5FCE6FB16F6FCE30),
	SPH_C64(0xFCEF91AE9191EF6D), SPH_C64(0xAA0752F6525207F8),
	SPH_C64(0x27FD60A06060FD47), SPH_C64(0x8976BCD9BCBC7635),
	SPH_C64(0xACCD9BB09B9BCD37), SPH_C64(0x048C8E8F8E8E8C8A),
	SPH_C64(0x7115A3F8A3A315D2), SPH_C64(0x603C0C140C0C3C6C),
	SPH_C64(0xFF8A7B8D7B7B8A84), SPH_C64(0xB5E1355F3535E180),
	SPH_C64(0xE8691D271D1D69F5), SPH_C64(0x5347E03DE0E047B3),
	SPH_C64(0xF6ACD764D7D7AC21), SPH_C64(0x5EEDC25BC2C2ED9C),
	SPH_C64(0x6D962E722E2E9643), SPH_C64(0x627A4BDD4B4B7A29),
	SPH_C64(0xA321FE1FFEFE215D), SPH_C64(0x821657F9575716D5),
	SPH_C64(0xA841153F151541BD), SPH_C64(0x9FB677997777B6E8),
	SPH_C64(0xA5EB37593737EB92), SPH_C64(0x7B56E532E5E5569E),
	SPH_C64(0x8CD99FBC9F9FD913), SPH_C64(0xD317F00DF0F01723),
	SPH_C64(0x6A7F4ADE4A4A7F20), SPH_C64(0x9E95DA73DADA9544),
	SPH_C64(0xFA2558E8585825A2), SPH_C64(0x06CAC946C9C9CACF),
	SPH_C64(0x558D297B29298D7C), SPH_C64(0x50220A1E0A0A225A),
	SPH_C64(0xE14FB1CEB1B14F50), SPH_C64(0x691AA0FDA0A01AC9),
	SPH_C64(0x7FDA6BBD6B6BDA14), SPH_C64(0x5CAB85928585ABD9),
	SPH_C64(0x8173BDDABDBD733C), SPH_C64(0xD2345DE75D5D348F),
	SPH_C64(0x8050103010105090), SPH_C64(0xF303F401F4F40307),
	SPH_C64(0x16C0CB40CBCBC0DD), SPH_C64(0xEDC63E423E3EC6D3),
	SPH_C64(0x2811050F0505112D), SPH_C64(0x1FE667A96767E678),
	SPH_C64(0x7353E431E4E45397), SPH_C64(0x25BB27692727BB02),
	SPH_C64(0x325841C341415873), SPH_C64(0x2C9D8B808B8B9DA7),
	SPH_C64(0x5101A7F4A7A701F6), SPH_C64(0xCF947D877D7D94B2),
	SPH_C64(0xDCFB95A29595FB49), SPH_C64(0x8E9FD875D8D89F56),
	SPH_C64(0x8B30FB10FBFB3070), SPH_C64(0x2371EE2FEEEE71CD),
	SPH_C64(0xC7917C847C7C91BB), SPH_C64(0x17E366AA6666E371),
	SPH_C64(0xA68EDD7ADDDD8E7B), SPH_C64(0xB84B173917174BAF),
	SPH_C64(0x024647C947474645), SPH_C64(0x84DC9EBF9E9EDC1A),
	SPH_C64(0x1EC5CA43CACAC5D4), SPH_C64(0x75992D772D2D9958),
	SPH_C64(0x9179BFDCBFBF792E), SPH_C64(0x381B070907071B3F),
	SPH_C64(0x0123ADEAADAD23AC), SPH_C64(0xEA2F5AEE5A5A2FB0),
	SPH_C64(0x6CB583988383B5EF), SPH_C64(0x85FF33553333FFB6),
	SPH_C64(0x3FF263A56363F25C), SPH_C64(0x100A020602020A12),
	SPH_C64(0x3938AAE3AAAA3893), SPH_C64(0xAFA871937171A8DE),
	SPH_C64(0x0ECFC845C8C8CFC6), SPH_C64(0xC87D192B19197DD1),
	SPH_C64(0x727049DB4949703B), SPH_C64(0x869AD976D9D99A5F),
	SPH_C64(0xC31DF20BF2F21D31), SPH_C64(0x4B48E338E3E348A8),
	SPH_C64(0xE22A5BED5B5B2AB9), SPH_C64(0x34928885888892BC),
	SPH_C64(0xA4C89AB39A9AC83E), SPH_C64(0x2DBE266A2626BE0B),
	SPH_C64(0x8DFA32563232FABF), SPH_C64(0xE94AB0CDB0B04A59),
	SPH_C64(0x1B6AE926E9E96AF2), SPH_C64(0x78330F110F0F3377),
	SPH_C64(0xE6A6D562D5D5A633), SPH_C64(0x74BA809D8080BAF4),
	SPH_C64(0x997CBEDFBEBE7C27), SPH_C64(0x26DECD4ACDCDDEEB),
	SPH_C64(0xBDE4345C3434E489), SPH_C64(0x7A7548D848487532),
	SPH_C64(0xAB24FF1CFFFF2454), SPH_C64(0xF78F7A8E7A7A8F8D),
	SPH_C64(0xF4EA90AD9090EA64), SPH_C64(0xC23E5FE15F5F3E9D),
	SPH_C64(0x1DA020602020A03D), SPH_C64(0x67D568B86868D50F),
	SPH_C64(0xD0721A2E1A1A72CA), SPH_C64(0x192CAEEFAEAE2CB7),
	SPH_C64(0xC95EB4C1B4B45E7D), SPH_C64(0x9A1954FC545419CE),
	SPH_C64(0xECE593A89393E57F), SPH_C64(0x0DAA22662222AA2F),
	SPH_C64(0x07E964AC6464E963), SPH_C64(0xDB12F10EF1F1122A),
	SPH_C64(0xBFA273957373A2CC), SPH_C64(0x905A123612125A82),
	SPH_C64(0x3A5D40C040405D7A), SPH_C64(0x4028081808082848),
	SPH_C64(0x56E8C358C3C3E895), SPH_C64(0x337BEC29ECEC7BDF),
	SPH_C64(0x9690DB70DBDB904D), SPH_C64(0x611FA1FEA1A11FC0),
	SPH_C64(0x1C838D8A8D8D8391), SPH_C64(0xF5C93D473D3DC9C8),
	SPH_C64(0xCCF197A49797F15B), SPH_C64(0x0000000000000000),
	SPH_C64(0x36D4CF4CCFCFD4F9), SPH_C64(0x45872B7D2B2B876E),
	SPH_C64(0x97B3769A7676B3E1), SPH_C64(0x64B0829B8282B0E6),
	SPH_C64(0xFEA9D667D6D6A928), SPH_C64(0xD8771B2D1B1B77C3),
	SPH_C64(0xC15BB5C2B5B55B74), SPH_C64(0x1129AFECAFAF29BE),
	SPH_C64(0x77DF6ABE6A6ADF1D), SPH_C64(0xBA0D50F050500DEA),
	SPH_C64(0x124C45CF45454C57), SPH_C64(0xCB18F308F3F31838),
	SPH_C64(0x9DF030503030F0AD), SPH_C64(0x2B74EF2CEFEF74C4),
	SPH_C64(0xE5C33F413F3FC3DA), SPH_C64(0x921C55FF55551CC7),
	SPH_C64(0x7910A2FBA2A210DB), SPH_C64(0x0365EA23EAEA65E9),
	SPH_C64(0x0FEC65AF6565EC6A), SPH_C64(0xB968BAD3BABA6803),
	SPH_C64(0x65932F712F2F934A), SPH_C64(0x4EE7C05DC0C0E78E),
	SPH_C64(0xBE81DE7FDEDE8160), SPH_C64(0xE06C1C241C1C6CFC),
	SPH_C64(0xBB2EFD1AFDFD2E46), SPH_C64(0x52644DD74D4D641F),
	SPH_C64(0xE4E092AB9292E076), SPH_C64(0x8FBC759F7575BCFA),
	SPH_C64(0x301E060A06061E36), SPH_C64(0x24988A838A8A98AE),
	SPH_C64(0xF940B2CBB2B2404B), SPH_C64(0x6359E637E6E65985),
	SPH_C64(0x70360E120E0E367E), SPH_C64(0xF8631F211F1F63E7),
	SPH_C64(0x37F762A66262F755), SPH_C64(0xEEA3D461D4D4A33A),
	SPH_C64(0x2932A8E5A8A83281), SPH_C64(0xC4F496A79696F452),
	SPH_C64(0x9B3AF916F9F93A62), SPH_C64(0x66F6C552C5C5F6A3),
	SPH_C64(0x35B1256F2525B110), SPH_C64(0xF22059EB595920AB),
	SPH_C64(0x54AE84918484AED0), SPH_C64(0xB7A772967272A7C5),
	SPH_C64(0xD5DD394B3939DDEC), SPH_C64(0x5A614CD44C4C6116),
	SPH_C64(0xCA3B5EE25E5E3B94), SPH_C64(0xE78578887878859F),
	SPH_C64(0xDDD838483838D8E5), SPH_C64(0x14868C898C8C8698),
	SPH_C64(0xC6B2D16ED1D1B217), SPH_C64(0x410BA5F2A5A50BE4),
	SPH_C64(0x434DE23BE2E24DA1), SPH_C64(0x2FF861A36161F84E),
	SPH_C64(0xF145B3C8B3B34542), SPH_C64(0x15A521632121A534),
	SPH_C64(0x94D69CB99C9CD608), SPH_C64(0xF0661E221E1E66EE),
	SPH_C64(0x225243C543435261), SPH_C64(0x76FCC754C7C7FCB1),
	SPH_C64(0xB32BFC19FCFC2B4F), SPH_C64(0x2014040C04041424),
	SPH_C64(0xB20851F3515108E3), SPH_C64(0xBCC799B69999C725),
	SPH_C64(0x4FC46DB76D6DC422), SPH_C64(0x68390D170D0D3965),
	SPH_C64(0x8335FA13FAFA3579), SPH_C64(0xB684DF7CDFDF8469),
	SPH_C64(0xD79B7E827E7E9BA9), SPH_C64(0x3DB4246C2424B419),
	SPH_C64(0xC5D73B4D3B3BD7FE), SPH_C64(0x313DABE0ABAB3D9A),
	SPH_C64(0x3ED1CE4FCECED1F0), SPH_C64(0x8855113311115599),
	SPH_C64(0x0C898F8C8F8F8983), SPH_C64(0x4A6B4ED24E4E6B04),
	SPH_C64(0xD151B7C4B7B75166), SPH_C64(0x0B60EB20EBEB60E0),
	SPH_C64(0xFDCC3C443C3CCCC1), SPH_C64(0x7CBF819E8181BFFD),
	SPH_C64(0xD4FE94A19494FE40), SPH_C64(0xEB0CF704F7F70C1C),
	SPH_C64(0xA167B9D6B9B96718), SPH_C64(0x985F133513135F8B),
	SPH_C64(0x7D9C2C742C2C9C51), SPH_C64(0xD6B8D368D3D3B805),
	SPH_C64(0x6B5CE734E7E75C8C), SPH_C64(0x57CB6EB26E6ECB39),
	SPH_C64(0x6EF3C451C4C4F3AA), SPH_C64(0x180F030503030F1B),
	SPH_C64(0x8A1356FA565613DC), SPH_C64(0x1A4944CC4444495E),
	SPH_C64(0xDF9E7F817F7F9EA0), SPH_C64(0x2137A9E6A9A93788),
	SPH_C64(0x4D822A7E2A2A8267), SPH_C64(0xB16DBBD0BBBB6D0A),
	SPH_C64(0x46E2C15EC1C1E287), SPH_C64(0xA20253F5535302F1),
	SPH_C64(0xAE8BDC79DCDC8B72), SPH_C64(0x58270B1D0B0B2753),
	SPH_C64(0x9CD39DBA9D9DD301), SPH_C64(0x47C16CB46C6CC12B),
	SPH_C64(0x95F531533131F5A4), SPH_C64(0x87B9749C7474B9F3),
	SPH_C64(0xE309F607F6F60915), SPH_C64(0x0A4346CA4646434C),
	SPH_C64(0x0926ACE9ACAC26A5), SPH_C64(0x3C978986898997B5),
	SPH_C64(0xA044143C141444B4), SPH_C64(0x5B42E13EE1E142BA),
	SPH_C64(0xB04E163A16164EA6), SPH_C64(0xCDD23A4E3A3AD2F7),
	SPH_C64(0x6FD069BB6969D006), SPH_C64(0x482D091B09092D41),
	SPH_C64(0xA7AD70907070ADD7), SPH_C64(0xD954B6C7B6B6546F),
	SPH_C64(0xCEB7D06DD0D0B71E), SPH_C64(0x3B7EED2AEDED7ED6),
	SPH_C64(0x2EDBCC49CCCCDBE2), SPH_C64(0x2A5742C642425768),
	SPH_C64(0xB4C298B59898C22C), SPH_C64(0x490EA4F1A4A40EED),
	SPH_C64(0x5D88287828288875), SPH_C64(0xDA315CE45C5C3186),
	SPH_C64(0x933FF815F8F83F6B), SPH_C64(0x44A486978686A4C2)
};

static const sph_u64 old1_T3[256] = {
	SPH_C64(0x781828181878D8C0), SPH_C64(0xAF23652323AF2605),
	SPH_C64(0xF9C657C6C6F9B87E), SPH_C64(0x6FE825E8E86FFB13),
	SPH_C64(0xA187948787A1CB4C), SPH_C64(0x62B8D5B8B86211A9),
	SPH_C64(0x0501030101050908), SPH_C64(0x6E4FD14F4F6E0D42),
	SPH_C64(0xEE365A3636EE9BAD), SPH_C64(0x04A6F7A6A604FF59),
	SPH_C64(0xBDD26BD2D2BD0CDE), SPH_C64(0x06F502F5F5060EFB),
	SPH_C64(0x80798B79798096EF), SPH_C64(0xCE6FB16F6FCE305F),
	SPH_C64(0xEF91AE9191EF6DFC), SPH_C64(0x0752F6525207F8AA),
	SPH_C64(0xFD60A06060FD4727), SPH_C64(0x76BCD9BCBC763589),
	SPH_C64(0xCD9BB09B9BCD37AC), SPH_C64(0x8C8E8F8E8E8C8A04),
	SPH_C64(0x15A3F8A3A315D271), SPH_C64(0x3C0C140C0C3C6C60),
	SPH_C64(0x8A7B8D7B7B8A84FF), SPH_C64(0xE1355F3535E180B5),
	SPH_C64(0x691D271D1D69F5E8), SPH_C64(0x47E03DE0E047B353),
	SPH_C64(0xACD764D7D7AC21F6), SPH_C64(0xEDC25BC2C2ED9C5E),
	SPH_C64(0x962E722E2E96436D), SPH_C64(0x7A4BDD4B4B7A2962),
	SPH_C64(0x21FE1FFEFE215DA3), SPH_C64(0x1657F9575716D582),
	SPH_C64(0x41153F151541BDA8), SPH_C64(0xB677997777B6E89F),
	SPH_C64(0xEB37593737EB92A5), SPH_C64(0x56E532E5E5569E7B),
	SPH_C64(0xD99FBC9F9FD9138C), SPH_C64(0x17F00DF0F01723D3),
	SPH_C64(0x7F4ADE4A4A7F206A), SPH_C64(0x95DA73DADA95449E),
	SPH_C64(0x2558E8585825A2FA), SPH_C64(0xCAC946C9C9CACF06),
	SPH_C64(0x8D297B29298D7C55), SPH_C64(0x220A1E0A0A225A50),
	SPH_C64(0x4FB1CEB1B14F50E1), SPH_C64(0x1AA0FDA0A01AC969),
	SPH_C64(0xDA6BBD6B6BDA147F), SPH_C64(0xAB85928585ABD95C),
	SPH_C64(0x73BDDABDBD733C81), SPH_C64(0x345DE75D5D348FD2),
	SPH_C64(0x5010301010509080), SPH_C64(0x03F401F4F40307F3),
	SPH_C64(0xC0CB40CBCBC0DD16), SPH_C64(0xC63E423E3EC6D3ED),
	SPH_C64(0x11050F0505112D28), SPH_C64(0xE667A96767E6781F),
	SPH_C64(0x53E431E4E4539773), SPH_C64(0xBB27692727BB0225),
	SPH_C64(0x5841C34141587332), SPH_C64(0x9D8B808B8B9DA72C),
	SPH_C64(0x01A7F4A7A701F651), SPH_C64(0x947D877D7D94B2CF),
	SPH_C64(0xFB95A29595FB49DC), SPH_C64(0x9FD875D8D89F568E),
	SPH_C64(0x30FB10FBFB30708B), SPH_C64(0x71EE2FEEEE71CD23),
	SPH_C64(0x917C847C7C91BBC7), SPH_C64(0xE366AA6666E37117),
	SPH_C64(0x8EDD7ADDDD8E7BA6), SPH_C64(0x4B173917174BAFB8),
	SPH_C64(0x4647C94747464502), SPH_C64(0xDC9EBF9E9EDC1A84),
	SPH_C64(0xC5CA43CACAC5D41E), SPH_C64(0x992D772D2D995875),
	SPH_C64(0x79BFDCBFBF792E91), SPH_C64(0x1B070907071B3F38),
	SPH_C64(0x23ADEAADAD23AC01), SPH_C64(0x2F5AEE5A5A2FB0EA),
	SPH_C64(0xB583988383B5EF6C), SPH_C64(0xFF33553333FFB685),
	SPH_C64(0xF263A56363F25C3F), SPH_C64(0x0A020602020A1210),
	SPH_C64(0x38AAE3AAAA389339), SPH_C64(0xA871937171A8DEAF),
	SPH_C64(0xCFC845C8C8CFC60E), SPH_C64(0x7D192B19197DD1C8),
	SPH_C64(0x7049DB4949703B72), SPH_C64(0x9AD976D9D99A5F86),
	SPH_C64(0x1DF20BF2F21D31C3), SPH_C64(0x48E338E3E348A84B),
	SPH_C64(0x2A5BED5B5B2AB9E2), SPH_C64(0x928885888892BC34),
	SPH_C64(0xC89AB39A9AC83EA4), SPH_C64(0xBE266A2626BE0B2D),
	SPH_C64(0xFA32563232FABF8D), SPH_C64(0x4AB0CDB0B04A59E9),
	SPH_C64(0x6AE926E9E96AF21B), SPH_C64(0x330F110F0F337778),
	SPH_C64(0xA6D562D5D5A633E6), SPH_C64(0xBA809D8080BAF474),
	SPH_C64(0x7CBEDFBEBE7C2799), SPH_C64(0xDECD4ACDCDDEEB26),
	SPH_C64(0xE4345C3434E489BD), SPH_C64(0x7548D8484875327A),
	SPH_C64(0x24FF1CFFFF2454AB), SPH_C64(0x8F7A8E7A7A8F8DF7),
	SPH_C64(0xEA90AD9090EA64F4), SPH_C64(0x3E5FE15F5F3E9DC2),
	SPH_C64(0xA020602020A03D1D), SPH_C64(0xD568B86868D50F67),
	SPH_C64(0x721A2E1A1A72CAD0), SPH_C64(0x2CAEEFAEAE2CB719),
	SPH_C64(0x5EB4C1B4B45E7DC9), SPH_C64(0x1954FC545419CE9A),
	SPH_C64(0xE593A89393E57FEC), SPH_C64(0xAA22662222AA2F0D),
	SPH_C64(0xE964AC6464E96307), SPH_C64(0x12F10EF1F1122ADB),
	SPH_C64(0xA273957373A2CCBF), SPH_C64(0x5A123612125A8290),
	SPH_C64(0x5D40C040405D7A3A), SPH_C64(0x2808180808284840),
	SPH_C64(0xE8C358C3C3E89556), SPH_C64(0x7BEC29ECEC7BDF33),
	SPH_C64(0x90DB70DBDB904D96), SPH_C64(0x1FA1FEA1A11FC061),
	SPH_C64(0x838D8A8D8D83911C), SPH_C64(0xC93D473D3DC9C8F5),
	SPH_C64(0xF197A49797F15BCC), SPH_C64(0x0000000000000000),
	SPH_C64(0xD4CF4CCFCFD4F936), SPH_C64(0x872B7D2B2B876E45),
	SPH_C64(0xB3769A7676B3E197), SPH_C64(0xB0829B8282B0E664),
	SPH_C64(0xA9D667D6D6A928FE), SPH_C64(0x771B2D1B1B77C3D8),
	SPH_C64(0x5BB5C2B5B55B74C1), SPH_C64(0x29AFECAFAF29BE11),
	SPH_C64(0xDF6ABE6A6ADF1D77), SPH_C64(0x0D50F050500DEABA),
	SPH_C64(0x4C45CF45454C5712), SPH_C64(0x18F308F3F31838CB),
	SPH_C64(0xF030503030F0AD9D), SPH_C64(0x74EF2CEFEF74C42B),
	SPH_C64(0xC33F413F3FC3DAE5), SPH_C64(0x1C55FF55551CC792),
	SPH_C64(0x10A2FBA2A210DB79), SPH_C64(0x65EA23EAEA65E903),
	SPH_C64(0xEC65AF6565EC6A0F), SPH_C64(0x68BAD3BABA6803B9),
	SPH_C64(0x932F712F2F934A65), SPH_C64(0xE7C05DC0C0E78E4E),
	SPH_C64(0x81DE7FDEDE8160BE), SPH_C64(0x6C1C241C1C6CFCE0),
	SPH_C64(0x2EFD1AFDFD2E46BB), SPH_C64(0x644DD74D4D641F52),
	SPH_C64(0xE092AB9292E076E4), SPH_C64(0xBC759F7575BCFA8F),
	SPH_C64(0x1E060A06061E3630), SPH_C64(0x988A838A8A98AE24),
	SPH_C64(0x40B2CBB2B2404BF9), SPH_C64(0x59E637E6E6598563),
	SPH_C64(0x360E120E0E367E70), SPH_C64(0x631F211F1F63E7F8),
	SPH_C64(0xF762A66262F75537), SPH_C64(0xA3D461D4D4A33AEE),
	SPH_C64(0x32A8E5A8A8328129), SPH_C64(0xF496A79696F452C4),
	SPH_C64(0x3AF916F9F93A629B), SPH_C64(0xF6C552C5C5F6A366),
	SPH_C64(0xB1256F2525B11035), SPH_C64(0x2059EB595920ABF2),
	SPH_C64(0xAE84918484AED054), SPH_C64(0xA772967272A7C5B7),
	SPH_C64(0xDD394B3939DDECD5), SPH_C64(0x614CD44C4C61165A),
	SPH_C64(0x3B5EE25E5E3B94CA), SPH_C64(0x8578887878859FE7),
	SPH_C64(0xD838483838D8E5DD), SPH_C64(0x868C898C8C869814),
	SPH_C64(0xB2D16ED1D1B217C6), SPH_C64(0x0BA5F2A5A50BE441),
	SPH_C64(0x4DE23BE2E24DA143), SPH_C64(0xF861A36161F84E2F),
	SPH_C64(0x45B3C8B3B34542F1), SPH_C64(0xA521632121A53415),
	SPH_C64(0xD69CB99C9CD60894), SPH_C64(0x661E221E1E66EEF0),
	SPH_C64(0x5243C54343526122), SPH_C64(0xFCC754C7C7FCB176),
	SPH_C64(0x2BFC19FCFC2B4FB3), SPH_C64(0x14040C0404142420),
	SPH_C64(0x0851F3515108E3B2), SPH_C64(0xC799B69999C725BC),
	SPH_C64(0xC46DB76D6DC4224F), SPH_C64(0x390D170D0D396568),
	SPH_C64(0x35FA13FAFA357983), SPH_C64(0x84DF7CDFDF8469B6),
	SPH_C64(0x9B7E827E7E9BA9D7), SPH_C64(0xB4246C2424B4193D),
	SPH_C64(0xD73B4D3B3BD7FEC5), SPH_C64(0x3DABE0ABAB3D9A31),
	SPH_C64(0xD1CE4FCECED1F03E), SPH_C64(0x5511331111559988),
	SPH_C64(0x898F8C8F8F89830C), SPH_C64(0x6B4ED24E4E6B044A),
	SPH_C64(0x51B7C4B7B75166D1), SPH_C64(0x60EB20EBEB60E00B),
	SPH_C64(0xCC3C443C3CCCC1FD), SPH_C64(0xBF819E8181BFFD7C),
	SPH_C64(0xFE94A19494FE40D4), SPH_C64(0x0CF704F7F70C1CEB),
	SPH_C64(0x67B9D6B9B96718A1), SPH_C64(0x5F133513135F8B98),
	SPH_C64(0x9C2C742C2C9C517D), SPH_C64(0xB8D368D3D3B805D6),
	SPH_C64(0x5CE734E7E75C8C6B), SPH_C64(0xCB6EB26E6ECB3957),
	SPH_C64(0xF3C451C4C4F3AA6E), SPH_C64(0x0F030503030F1B18),
	SPH_C64(0x1356FA565613DC8A), SPH_C64(0x4944CC4444495E1A),
	SPH_C64(0x9E7F817F7F9EA0DF), SPH_C64(0x37A9E6A9A9378821),
	SPH_C64(0x822A7E2A2A82674D), SPH_C64(0x6DBBD0BBBB6D0AB1),
	SPH_C64(0xE2C15EC1C1E28746), SPH_C64(0x0253F5535302F1A2),
	SPH_C64(0x8BDC79DCDC8B72AE), SPH_C64(0x270B1D0B0B275358),
	SPH_C64(0xD39DBA9D9DD3019C), SPH_C64(0xC16CB46C6CC12B47),
	SPH_C64(0xF531533131F5A495), SPH_C64(0xB9749C7474B9F387),
	SPH_C64(0x09F607F6F60915E3), SPH_C64(0x4346CA4646434C0A),
	SPH_C64(0x26ACE9ACAC26A509), SPH_C64(0x978986898997B53C),
	SPH_C64(0x44143C141444B4A0), SPH_C64(0x42E13EE1E142BA5B),
	SPH_C64(0x4E163A16164EA6B0), SPH_C64(0xD23A4E3A3AD2F7CD),
	SPH_C64(0xD069BB6969D0066F), SPH_C64(0x2D091B09092D4148),
	SPH_C64(0xAD70907070ADD7A7), SPH_C64(0x54B6C7B6B6546FD9),
	SPH_C64(0xB7D06DD0D0B71ECE), SPH_C64(0x7EED2AEDED7ED63B),
	SPH_C64(0xDBCC49CCCCDBE22E), SPH_C64(0x5742C6424257682A),
	SPH_C64(0xC298B59898C22CB4), SPH_C64(0x0EA4F1A4A40EED49),
	SPH_C64(0x882878282888755D), SPH_C64(0x315CE45C5C3186DA),
	SPH_C64(0x3FF815F8F83F6B93), SPH_C64(0xA486978686A4C244)
};

static const sph_u64 old1_T4[256] = {
	SPH_C64(0x1828181878D8C078), SPH_C64(0x23652323AF2605AF),
	SPH_C64(0xC657C6C6F9B87EF9), SPH_C64(0xE825E8E86FFB136F),
	SPH_C64(0x87948787A1CB4CA1), SPH_C64(0xB8D5B8B86211A962),
	SPH_C64(0x0103010105090805), SPH_C64(0x4FD14F4F6E0D426E),
	SPH_C64(0x365A3636EE9BADEE), SPH_C64(0xA6F7A6A604FF5904),
	SPH_C64(0xD26BD2D2BD0CDEBD), SPH_C64(0xF502F5F5060EFB06),
	SPH_C64(0x798B79798096EF80), SPH_C64(0x6FB16F6FCE305FCE),
	SPH_C64(0x91AE9191EF6DFCEF), SPH_C64(0x52F6525207F8AA07),
	SPH_C64(0x60A06060FD4727FD), SPH_C64(0xBCD9BCBC76358976),
	SPH_C64(0x9BB09B9BCD37ACCD), SPH_C64(0x8E8F8E8E8C8A048C),
	SPH_C64(0xA3F8A3A315D27115), SPH_C64(0x0C140C0C3C6C603C),
	SPH_C64(0x7B8D7B7B8A84FF8A), SPH_C64(0x355F3535E180B5E1),
	SPH_C64(0x1D271D1D69F5E869), SPH_C64(0xE03DE0E047B35347),
	SPH_C64(0xD764D7D7AC21F6AC), SPH_C64(0xC25BC2C2ED9C5EED),
	SPH_C64(0x2E722E2E96436D96), SPH_C64(0x4BDD4B4B7A29627A),
	SPH_C64(0xFE1FFEFE215DA321), SPH_C64(0x57F9575716D58216),
	SPH_C64(0x153F151541BDA841), SPH_C64(0x77997777B6E89FB6),
	SPH_C64(0x37593737EB92A5EB), SPH_C64(0xE532E5E5569E7B56),
	SPH_C64(0x9FBC9F9FD9138CD9), SPH_C64(0xF00DF0F01723D317),
	SPH_C64(0x4ADE4A4A7F206A7F), SPH_C64(0xDA73DADA95449E95),
	SPH_C64(0x58E8585825A2FA25), SPH_C64(0xC946C9C9CACF06CA),
	SPH_C64(0x297B29298D7C558D), SPH_C64(0x0A1E0A0A225A5022),
	SPH_C64(0xB1CEB1B14F50E14F), SPH_C64(0xA0FDA0A01AC9691A),
	SPH_C64(0x6BBD6B6BDA147FDA), SPH_C64(0x85928585ABD95CAB),
	SPH_C64(0xBDDABDBD733C8173), SPH_C64(0x5DE75D5D348FD234),
	SPH_C64(0x1030101050908050), SPH_C64(0xF401F4F40307F303),
	SPH_C64(0xCB40CBCBC0DD16C0), SPH_C64(0x3E423E3EC6D3EDC6),
	SPH_C64(0x050F0505112D2811), SPH_C64(0x67A96767E6781FE6),
	SPH_C64(0xE431E4E453977353), SPH_C64(0x27692727BB0225BB),
	SPH_C64(0x41C3414158733258), SPH_C64(0x8B808B8B9DA72C9D),
	SPH_C64(0xA7F4A7A701F65101), SPH_C64(0x7D877D7D94B2CF94),
	SPH_C64(0x95A29595FB49DCFB), SPH_C64(0xD875D8D89F568E9F),
	SPH_C64(0xFB10FBFB30708B30), SPH_C64(0xEE2FEEEE71CD2371),
	SPH_C64(0x7C847C7C91BBC791), SPH_C64(0x66AA6666E37117E3),
	SPH_C64(0xDD7ADDDD8E7BA68E), SPH_C64(0x173917174BAFB84B),
	SPH_C64(0x47C9474746450246), SPH_C64(0x9EBF9E9EDC1A84DC),
	SPH_C64(0xCA43CACAC5D41EC5), SPH_C64(0x2D772D2D99587599),
	SPH_C64(0xBFDCBFBF792E9179), SPH_C64(0x070907071B3F381B),
	SPH_C64(0xADEAADAD23AC0123), SPH_C64(0x5AEE5A5A2FB0EA2F),
	SPH_C64(0x83988383B5EF6CB5), SPH_C64(0x33553333FFB685FF),
	SPH_C64(0x63A56363F25C3FF2), SPH_C64(0x020602020A12100A),
	SPH_C64(0xAAE3AAAA38933938), SPH_C64(0x71937171A8DEAFA8),
	SPH_C64(0xC845C8C8CFC60ECF), SPH_C64(0x192B19197DD1C87D),
	SPH_C64(0x49DB4949703B7270), SPH_C64(0xD976D9D99A5F869A),
	SPH_C64(0xF20BF2F21D31C31D), SPH_C64(0xE338E3E348A84B48),
	SPH_C64(0x5BED5B5B2AB9E22A), SPH_C64(0x8885888892BC3492),
	SPH_C64(0x9AB39A9AC83EA4C8), SPH_C64(0x266A2626BE0B2DBE),
	SPH_C64(0x32563232FABF8DFA), SPH_C64(0xB0CDB0B04A59E94A),
	SPH_C64(0xE926E9E96AF21B6A), SPH_C64(0x0F110F0F33777833),
	SPH_C64(0xD562D5D5A633E6A6), SPH_C64(0x809D8080BAF474BA),
	SPH_C64(0xBEDFBEBE7C27997C), SPH_C64(0xCD4ACDCDDEEB26DE),
	SPH_C64(0x345C3434E489BDE4), SPH_C64(0x48D8484875327A75),
	SPH_C64(0xFF1CFFFF2454AB24), SPH_C64(0x7A8E7A7A8F8DF78F),
	SPH_C64(0x90AD9090EA64F4EA), SPH_C64(0x5FE15F5F3E9DC23E),
	SPH_C64(0x20602020A03D1DA0), SPH_C64(0x68B86868D50F67D5),
	SPH_C64(0x1A2E1A1A72CAD072), SPH_C64(0xAEEFAEAE2CB7192C),
	SPH_C64(0xB4C1B4B45E7DC95E), SPH_C64(0x54FC545419CE9A19),
	SPH_C64(0x93A89393E57FECE5), SPH_C64(0x22662222AA2F0DAA),
	SPH_C64(0x64AC6464E96307E9), SPH_C64(0xF10EF1F1122ADB12),
	SPH_C64(0x73957373A2CCBFA2), SPH_C64(0x123612125A82905A),
	SPH_C64(0x40C040405D7A3A5D), SPH_C64(0x0818080828484028),
	SPH_C64(0xC358C3C3E89556E8), SPH_C64(0xEC29ECEC7BDF337B),
	SPH_C64(0xDB70DBDB904D9690), SPH_C64(0xA1FEA1A11FC0611F),
	SPH_C64(0x8D8A8D8D83911C83), SPH_C64(0x3D473D3DC9C8F5C9),
	SPH_C64(0x97A49797F15BCCF1), SPH_C64(0x0000000000000000),
	SPH_C64(0xCF4CCFCFD4F936D4), SPH_C64(0x2B7D2B2B876E4587),
	SPH_C64(0x769A7676B3E197B3), SPH_C64(0x829B8282B0E664B0),
	SPH_C64(0xD667D6D6A928FEA9), SPH_C64(0x1B2D1B1B77C3D877),
	SPH_C64(0xB5C2B5B55B74C15B), SPH_C64(0xAFECAFAF29BE1129),
	SPH_C64(0x6ABE6A6ADF1D77DF), SPH_C64(0x50F050500DEABA0D),
	SPH_C64(0x45CF45454C57124C), SPH_C64(0xF308F3F31838CB18),
	SPH_C64(0x30503030F0AD9DF0), SPH_C64(0xEF2CEFEF74C42B74),
	SPH_C64(0x3F413F3FC3DAE5C3), SPH_C64(0x55FF55551CC7921C),
	SPH_C64(0xA2FBA2A210DB7910), SPH_C64(0xEA23EAEA65E90365),
	SPH_C64(0x65AF6565EC6A0FEC), SPH_C64(0xBAD3BABA6803B968),
	SPH_C64(0x2F712F2F934A6593), SPH_C64(0xC05DC0C0E78E4EE7),
	SPH_C64(0xDE7FDEDE8160BE81), SPH_C64(0x1C241C1C6CFCE06C),
	SPH_C64(0xFD1AFDFD2E46BB2E), SPH_C64(0x4DD74D4D641F5264),
	SPH_C64(0x92AB9292E076E4E0), SPH_C64(0x759F7575BCFA8FBC),
	SPH_C64(0x060A06061E36301E), SPH_C64(0x8A838A8A98AE2498),
	SPH_C64(0xB2CBB2B2404BF940), SPH_C64(0xE637E6E659856359),
	SPH_C64(0x0E120E0E367E7036), SPH_C64(0x1F211F1F63E7F863),
	SPH_C64(0x62A66262F75537F7), SPH_C64(0xD461D4D4A33AEEA3),
	SPH_C64(0xA8E5A8A832812932), SPH_C64(0x96A79696F452C4F4),
	SPH_C64(0xF916F9F93A629B3A), SPH_C64(0xC552C5C5F6A366F6),
	SPH_C64(0x256F2525B11035B1), SPH_C64(0x59EB595920ABF220),
	SPH_C64(0x84918484AED054AE), SPH_C64(0x72967272A7C5B7A7),
	SPH_C64(0x394B3939DDECD5DD), SPH_C64(0x4CD44C4C61165A61),
	SPH_C64(0x5EE25E5E3B94CA3B), SPH_C64(0x78887878859FE785),
	SPH_C64(0x38483838D8E5DDD8), SPH_C64(0x8C898C8C86981486),
	SPH_C64(0xD16ED1D1B217C6B2), SPH_C64(0xA5F2A5A50BE4410B),
	SPH_C64(0xE23BE2E24DA1434D), SPH_C64(0x61A36161F84E2FF8),
	SPH_C64(0xB3C8B3B34542F145), SPH_C64(0x21632121A53415A5),
	SPH_C64(0x9CB99C9CD60894D6), SPH_C64(0x1E221E1E66EEF066),
	SPH_C64(0x43C5434352612252), SPH_C64(0xC754C7C7FCB176FC),
	SPH_C64(0xFC19FCFC2B4FB32B), SPH_C64(0x040C040414242014),
	SPH_C64(0x51F3515108E3B208), SPH_C64(0x99B69999C725BCC7),
	SPH_C64(0x6DB76D6DC4224FC4), SPH_C64(0x0D170D0D39656839),
	SPH_C64(0xFA13FAFA35798335), SPH_C64(0xDF7CDFDF8469B684),
	SPH_C64(0x7E827E7E9BA9D79B), SPH_C64(0x246C2424B4193DB4),
	SPH_C64(0x3B4D3B3BD7FEC5D7), SPH_C64(0xABE0ABAB3D9A313D),
	SPH_C64(0xCE4FCECED1F03ED1), SPH_C64(0x1133111155998855),
	SPH_C64(0x8F8C8F8F89830C89), SPH_C64(0x4ED24E4E6B044A6B),
	SPH_C64(0xB7C4B7B75166D151), SPH_C64(0xEB20EBEB60E00B60),
	SPH_C64(0x3C443C3CCCC1FDCC), SPH_C64(0x819E8181BFFD7CBF),
	SPH_C64(0x94A19494FE40D4FE), SPH_C64(0xF704F7F70C1CEB0C),
	SPH_C64(0xB9D6B9B96718A167), SPH_C64(0x133513135F8B985F),
	SPH_C64(0x2C742C2C9C517D9C), SPH_C64(0xD368D3D3B805D6B8),
	SPH_C64(0xE734E7E75C8C6B5C), SPH_C64(0x6EB26E6ECB3957CB),
	SPH_C64(0xC451C4C4F3AA6EF3), SPH_C64(0x030503030F1B180F),
	SPH_C64(0x56FA565613DC8A13), SPH_C64(0x44CC4444495E1A49),
	SPH_C64(0x7F817F7F9EA0DF9E), SPH_C64(0xA9E6A9A937882137),
	SPH_C64(0x2A7E2A2A82674D82), SPH_C64(0xBBD0BBBB6D0AB16D),
	SPH_C64(0xC15EC1C1E28746E2), SPH_C64(0x53F5535302F1A202),
	SPH_C64(0xDC79DCDC8B72AE8B), SPH_C64(0x0B1D0B0B27535827),
	SPH_C64(0x9DBA9D9DD3019CD3), SPH_C64(0x6CB46C6CC12B47C1),
	SPH_C64(0x31533131F5A495F5), SPH_C64(0x749C7474B9F387B9),
	SPH_C64(0xF607F6F60915E309), SPH_C64(0x46CA4646434C0A43),
	SPH_C64(0xACE9ACAC26A50926), SPH_C64(0x8986898997B53C97),
	SPH_C64(0x143C141444B4A044), SPH_C64(0xE13EE1E142BA5B42),
	SPH_C64(0x163A16164EA6B04E), SPH_C64(0x3A4E3A3AD2F7CDD2),
	SPH_C64(0x69BB6969D0066FD0), SPH_C64(0x091B09092D41482D),
	SPH_C64(0x70907070ADD7A7AD), SPH_C64(0xB6C7B6B6546FD954),
	SPH_C64(0xD06DD0D0B71ECEB7), SPH_C64(0xED2AEDED7ED63B7E),
	SPH_C64(0xCC49CCCCDBE22EDB), SPH_C64(0x42C6424257682A57),
	SPH_C64(0x98B59898C22CB4C2), SPH_C64(0xA4F1A4A40EED490E),
	SPH_C64(0x2878282888755D88), SPH_C64(0x5CE45C5C3186DA31),
	SPH_C64(0xF815F8F83F6B933F), SPH_C64(0x86978686A4C244A4)
};

static const sph_u64 old1_T5[256] = {
	SPH_C64(0x28181878D8C07818), SPH_C64(0x652323AF2605AF23),
	SPH_C64(0x57C6C6F9B87EF9C6), SPH_C64(0x25E8E86FFB136FE8),
	SPH_C64(0x948787A1CB4CA187), SPH_C64(0xD5B8B86211A962B8),
	SPH_C64(0x0301010509080501), SPH_C64(0xD14F4F6E0D426E4F),
	SPH_C64(0x5A3636EE9BADEE36), SPH_C64(0xF7A6A604FF5904A6),
	SPH_C64(0x6BD2D2BD0CDEBDD2), SPH_C64(0x02F5F5060EFB06F5),
	SPH_C64(0x8B79798096EF8079), SPH_C64(0xB16F6FCE305FCE6F),
	SPH_C64(0xAE9191EF6DFCEF91), SPH_C64(0xF6525207F8AA0752),
	SPH_C64(0xA06060FD4727FD60), SPH_C64(0xD9BCBC76358976BC),
	SPH_C64(0xB09B9BCD37ACCD9B), SPH_C64(0x8F8E8E8C8A048C8E),
	SPH_C64(0xF8A3A315D27115A3), SPH_C64(0x140C0C3C6C603C0C),
	SPH_C64(0x8D7B7B8A84FF8A7B), SPH_C64(0x5F3535E180B5E135),
	SPH_C64(0x271D1D69F5E8691D), SPH_C64(0x3DE0E047B35347E0),
	SPH_C64(0x64D7D7AC21F6ACD7), SPH_C64(0x5BC2C2ED9C5EEDC2),
	SPH_C64(0x722E2E96436D962E), SPH_C64(0xDD4B4B7A29627A4B),
	SPH_C64(0x1FFEFE215DA321FE), SPH_C64(0xF9575716D5821657),
	SPH_C64(0x3F151541BDA84115), SPH_C64(0x997777B6E89FB677),
	SPH_C64(0x593737EB92A5EB37), SPH_C64(0x32E5E5569E7B56E5),
	SPH_C64(0xBC9F9FD9138CD99F), SPH_C64(0x0DF0F01723D317F0),
	SPH_C64(0xDE4A4A7F206A7F4A), SPH_C64(0x73DADA95449E95DA),
	SPH_C64(0xE8585825A2FA2558), SPH_C64(0x46C9C9CACF06CAC9),
	SPH_C64(0x7B29298D7C558D29), SPH_C64(0x1E0A0A225A50220A),
	SPH_C64(0xCEB1B14F50E14FB1), SPH_C64(0xFDA0A01AC9691AA0),
	SPH_C64(0xBD6B6BDA147FDA6B), SPH_C64(0x928585ABD95CAB85),
	SPH_C64(0xDABDBD733C8173BD), SPH_C64(0xE75D5D348FD2345D),
	SPH_C64(0x3010105090805010), SPH_C64(0x01F4F40307F303F4),
	SPH_C64(0x40CBCBC0DD16C0CB), SPH_C64(0x423E3EC6D3EDC63E),
	SPH_C64(0x0F0505112D281105), SPH_C64(0xA96767E6781FE667),
	SPH_C64(0x31E4E453977353E4), SPH_C64(0x692727BB0225BB27),
	SPH_C64(0xC341415873325841), SPH_C64(0x808B8B9DA72C9D8B),
	SPH_C64(0xF4A7A701F65101A7), SPH_C64(0x877D7D94B2CF947D),
	SPH_C64(0xA29595FB49DCFB95), SPH_C64(0x75D8D89F568E9FD8),
	SPH_C64(0x10FBFB30708B30FB), SPH_C64(0x2FEEEE71CD2371EE),
	SPH_C64(0x847C7C91BBC7917C), SPH_C64(0xAA6666E37117E366),
	SPH_C64(0x7ADDDD8E7BA68EDD), SPH_C64(0x3917174BAFB84B17),
	SPH_C64(0xC947474645024647), SPH_C64(0xBF9E9EDC1A84DC9E),
	SPH_C64(0x43CACAC5D41EC5CA), SPH_C64(0x772D2D995875992D),
	SPH_C64(0xDCBFBF792E9179BF), SPH_C64(0x0907071B3F381B07),
	SPH_C64(0xEAADAD23AC0123AD), SPH_C64(0xEE5A5A2FB0EA2F5A),
	SPH_C64(0x988383B5EF6CB583), SPH_C64(0x553333FFB685FF33),
	SPH_C64(0xA56363F25C3FF263), SPH_C64(0x0602020A12100A02),
	SPH_C64(0xE3AAAA38933938AA), SPH_C64(0x937171A8DEAFA871),
	SPH_C64(0x45C8C8CFC60ECFC8), SPH_C64(0x2B19197DD1C87D19),
	SPH_C64(0xDB4949703B727049), SPH_C64(0x76D9D99A5F869AD9),
	SPH_C64(0x0BF2F21D31C31DF2), SPH_C64(0x38E3E348A84B48E3),
	SPH_C64(0xED5B5B2AB9E22A5B), SPH_C64(0x85888892BC349288),
	SPH_C64(0xB39A9AC83EA4C89A), SPH_C64(0x6A2626BE0B2DBE26),
	SPH_C64(0x563232FABF8DFA32), SPH_C64(0xCDB0B04A59E94AB0),
	SPH_C64(0x26E9E96AF21B6AE9), SPH_C64(0x110F0F337778330F),
	SPH_C64(0x62D5D5A633E6A6D5), SPH_C64(0x9D8080BAF474BA80),
	SPH_C64(0xDFBEBE7C27997CBE), SPH_C64(0x4ACDCDDEEB26DECD),
	SPH_C64(0x5C3434E489BDE434), SPH_C64(0xD8484875327A7548),
	SPH_C64(0x1CFFFF2454AB24FF), SPH_C64(0x8E7A7A8F8DF78F7A),
	SPH_C64(0xAD9090EA64F4EA90), SPH_C64(0xE15F5F3E9DC23E5F),
	SPH_C64(0x602020A03D1DA020), SPH_C64(0xB86868D50F67D568),
	SPH_C64(0x2E1A1A72CAD0721A), SPH_C64(0xEFAEAE2CB7192CAE),
	SPH_C64(0xC1B4B45E7DC95EB4), SPH_C64(0xFC545419CE9A1954),
	SPH_C64(0xA89393E57FECE593), SPH_C64(0x662222AA2F0DAA22),
	SPH_C64(0xAC6464E96307E964), SPH_C64(0x0EF1F1122ADB12F1),
	SPH_C64(0x957373A2CCBFA273), SPH_C64(0x3612125A82905A12),
	SPH_C64(0xC040405D7A3A5D40), SPH_C64(0x1808082848402808),
	SPH_C64(0x58C3C3E89556E8C3), SPH_C64(0x29ECEC7BDF337BEC),
	SPH_C64(0x70DBDB904D9690DB), SPH_C64(0xFEA1A11FC0611FA1),
	SPH_C64(0x8A8D8D83911C838D), SPH_C64(0x473D3DC9C8F5C93D),
	SPH_C64(0xA49797F15BCCF197), SPH_C64(0x0000000000000000),
	SPH_C64(0x4CCFCFD4F936D4CF), SPH_C64(0x7D2B2B876E45872B),
	SPH_C64(0x9A7676B3E197B376), SPH_C64(0x9B8282B0E664B082),
	SPH_C64(0x67D6D6A928FEA9D6), SPH_C64(0x2D1B1B77C3D8771B),
	SPH_C64(0xC2B5B55B74C15BB5), SPH_C64(0xECAFAF29BE1129AF),
	SPH_C64(0xBE6A6ADF1D77DF6A), SPH_C64(0xF050500DEABA0D50),
	SPH_C64(0xCF45454C57124C45), SPH_C64(0x08F3F31838CB18F3),
	SPH_C64(0x503030F0AD9DF030), SPH_C64(0x2CEFEF74C42B74EF),
	SPH_C64(0x413F3FC3DAE5C33F), SPH_C64(0xFF55551CC7921C55),
	SPH_C64(0xFBA2A210DB7910A2), SPH_C64(0x23EAEA65E90365EA),
	SPH_C64(0xAF6565EC6A0FEC65), SPH_C64(0xD3BABA6803B968BA),
	SPH_C64(0x712F2F934A65932F), SPH_C64(0x5DC0C0E78E4EE7C0),
	SPH_C64(0x7FDEDE8160BE81DE), SPH_C64(0x241C1C6CFCE06C1C),
	SPH_C64(0x1AFDFD2E46BB2EFD), SPH_C64(0xD74D4D641F52644D),
	SPH_C64(0xAB9292E076E4E092), SPH_C64(0x9F7575BCFA8FBC75),
	SPH_C64(0x0A06061E36301E06), SPH_C64(0x838A8A98AE24988A),
	SPH_C64(0xCBB2B2404BF940B2), SPH_C64(0x37E6E659856359E6),
	SPH_C64(0x120E0E367E70360E), SPH_C64(0x211F1F63E7F8631F),
	SPH_C64(0xA66262F75537F762), SPH_C64(0x61D4D4A33AEEA3D4),
	SPH_C64(0xE5A8A832812932A8), SPH_C64(0xA79696F452C4F496),
	SPH_C64(0x16F9F93A629B3AF9), SPH_C64(0x52C5C5F6A366F6C5),
	SPH_C64(0x6F2525B11035B125), SPH_C64(0xEB595920ABF22059),
	SPH_C64(0x918484AED054AE84), SPH_C64(0x967272A7C5B7A772),
	SPH_C64(0x4B3939DDECD5DD39), SPH_C64(0xD44C4C61165A614C),
	SPH_C64(0xE25E5E3B94CA3B5E), SPH_C64(0x887878859FE78578),
	SPH_C64(0x483838D8E5DDD838), SPH_C64(0x898C8C869814868C),
	SPH_C64(0x6ED1D1B217C6B2D1), SPH_C64(0xF2A5A50BE4410BA5),
	SPH_C64(0x3BE2E24DA1434DE2), SPH_C64(0xA36161F84E2FF861),
	SPH_C64(0xC8B3B34542F145B3), SPH_C64(0x632121A53415A521),
	SPH_C64(0xB99C9CD60894D69C), SPH_C64(0x221E1E66EEF0661E),
	SPH_C64(0xC543435261225243), SPH_C64(0x54C7C7FCB176FCC7),
	SPH_C64(0x19FCFC2B4FB32BFC), SPH_C64(0x0C04041424201404),
	SPH_C64(0xF3515108E3B20851), SPH_C64(0xB69999C725BCC799),
	SPH_C64(0xB76D6DC4224FC46D), SPH_C64(0x170D0D396568390D),
	SPH_C64(0x13FAFA35798335FA), SPH_C64(0x7CDFDF8469B684DF),
	SPH_C64(0x827E7E9BA9D79B7E), SPH_C64(0x6C2424B4193DB424),
	SPH_C64(0x4D3B3BD7FEC5D73B), SPH_C64(0xE0ABAB3D9A313DAB),
	SPH_C64(0x4FCECED1F03ED1CE), SPH_C64(0x3311115599885511),
	SPH_C64(0x8C8F8F89830C898F), SPH_C64(0xD24E4E6B044A6B4E),
	SPH_C64(0xC4B7B75166D151B7), SPH_C64(0x20EBEB60E00B60EB),
	SPH_C64(0x443C3CCCC1FDCC3C), SPH_C64(0x9E8181BFFD7CBF81),
	SPH_C64(0xA19494FE40D4FE94), SPH_C64(0x04F7F70C1CEB0CF7),
	SPH_C64(0xD6B9B96718A167B9), SPH_C64(0x3513135F8B985F13),
	SPH_C64(0x742C2C9C517D9C2C), SPH_C64(0x68D3D3B805D6B8D3),
	SPH_C64(0x34E7E75C8C6B5CE7), SPH_C64(0xB26E6ECB3957CB6E),
	SPH_C64(0x51C4C4F3AA6EF3C4), SPH_C64(0x0503030F1B180F03),
	SPH_C64(0xFA565613DC8A1356), SPH_C64(0xCC4444495E1A4944),
	SPH_C64(0x817F7F9EA0DF9E7F), SPH_C64(0xE6A9A937882137A9),
	SPH_C64(0x7E2A2A82674D822A), SPH_C64(0xD0BBBB6D0AB16DBB),
	SPH_C64(0x5EC1C1E28746E2C1), SPH_C64(0xF5535302F1A20253),
	SPH_C64(0x79DCDC8B72AE8BDC), SPH_C64(0x1D0B0B275358270B),
	SPH_C64(0xBA9D9DD3019CD39D), SPH_C64(0xB46C6CC12B47C16C),
	SPH_C64(0x533131F5A495F531), SPH_C64(0x9C7474B9F387B974),
	SPH_C64(0x07F6F60915E309F6), SPH_C64(0xCA4646434C0A4346),
	SPH_C64(0xE9ACAC26A50926AC), SPH_C64(0x86898997B53C9789),
	SPH_C64(0x3C141444B4A04414), SPH_C64(0x3EE1E142BA5B42E1),
	SPH_C64(0x3A16164EA6B04E16), SPH_C64(0x4E3A3AD2F7CDD23A),
	SPH_C64(0xBB6969D0066FD069), SPH_C64(0x1B09092D41482D09),
	SPH_C64(0x907070ADD7A7AD70), SPH_C64(0xC7B6B6546FD954B6),
	SPH_C64(0x6DD0D0B71ECEB7D0), SPH_C64(0x2AEDED7ED63B7EED),
	SPH_C64(0x49CCCCDBE22EDBCC), SPH_C64(0xC6424257682A5742),
	SPH_C64(0xB59898C22CB4C298), SPH_C64(0xF1A4A40EED490EA4),
	SPH_C64(0x78282888755D8828), SPH_C64(0xE45C5C3186DA315C),
	SPH_C64(0x15F8F83F6B933FF8), SPH_C64(0x978686A4C244A486)
};

static const sph_u64 old1_T6[256] = {
	SPH_C64(0x181878D8C0781828), SPH_C64(0x2323AF2605AF2365),
	SPH_C64(0xC6C6F9B87EF9C657), SPH_C64(0xE8E86FFB136FE825),
	SPH_C64(0x8787A1CB4CA18794), SPH_C64(0xB8B86211A962B8D5),
	SPH_C64(0x0101050908050103), SPH_C64(0x4F4F6E0D426E4FD1),
	SPH_C64(0x3636EE9BADEE365A), SPH_C64(0xA6A604FF5904A6F7),
	SPH_C64(0xD2D2BD0CDEBDD26B), SPH_C64(0xF5F5060EFB06F502),
	SPH_C64(0x79798096EF80798B), SPH_C64(0x6F6FCE305FCE6FB1),
	SPH_C64(0x9191EF6DFCEF91AE), SPH_C64(0x525207F8AA0752F6),
	SPH_C64(0x6060FD4727FD60A0), SPH_C64(0xBCBC76358976BCD9),
	SPH_C64(0x9B9BCD37ACCD9BB0), SPH_C64(0x8E8E8C8A048C8E8F),
	SPH_C64(0xA3A315D27115A3F8), SPH_C64(0x0C0C3C6C603C0C14),
	SPH_C64(0x7B7B8A84FF8A7B8D), SPH_C64(0x3535E180B5E1355F),
	SPH_C64(0x1D1D69F5E8691D27), SPH_C64(0xE0E047B35347E03D),
	SPH_C64(0xD7D7AC21F6ACD764), SPH_C64(0xC2C2ED9C5EEDC25B),
	SPH_C64(0x2E2E96436D962E72), SPH_C64(0x4B4B7A29627A4BDD),
	SPH_C64(0xFEFE215DA321FE1F), SPH_C64(0x575716D5821657F9),
	SPH_C64(0x151541BDA841153F), SPH_C64(0x7777B6E89FB67799),
	SPH_C64(0x3737EB92A5EB3759), SPH_C64(0xE5E5569E7B56E532),
	SPH_C64(0x9F9FD9138CD99FBC), SPH_C64(0xF0F01723D317F00D),
	SPH_C64(0x4A4A7F206A7F4ADE), SPH_C64(0xDADA95449E95DA73),
	SPH_C64(0x585825A2FA2558E8), SPH_C64(0xC9C9CACF06CAC946),
	SPH_C64(0x29298D7C558D297B), SPH_C64(0x0A0A225A50220A1E),
	SPH_C64(0xB1B14F50E14FB1CE), SPH_C64(0xA0A01AC9691AA0FD),
	SPH_C64(0x6B6BDA147FDA6BBD), SPH_C64(0x8585ABD95CAB8592),
	SPH_C64(0xBDBD733C8173BDDA), SPH_C64(0x5D5D348FD2345DE7),
	SPH_C64(0x1010509080501030), SPH_C64(0xF4F40307F303F401),
	SPH_C64(0xCBCBC0DD16C0CB40), SPH_C64(0x3E3EC6D3EDC63E42),
	SPH_C64(0x0505112D2811050F), SPH_C64(0x6767E6781FE667A9),
	SPH_C64(0xE4E453977353E431), SPH_C64(0x2727BB0225BB2769),
	SPH_C64(0x41415873325841C3), SPH_C64(0x8B8B9DA72C9D8B80),
	SPH_C64(0xA7A701F65101A7F4), SPH_C64(0x7D7D94B2CF947D87),
	SPH_C64(0x9595FB49DCFB95A2), SPH_C64(0xD8D89F568E9FD875),
	SPH_C64(0xFBFB30708B30FB10), SPH_C64(0xEEEE71CD2371EE2F),
	SPH_C64(0x7C7C91BBC7917C84), SPH_C64(0x6666E37117E366AA),
	SPH_C64(0xDDDD8E7BA68EDD7A), SPH_C64(0x17174BAFB84B1739),
	SPH_C64(0x47474645024647C9), SPH_C64(0x9E9EDC1A84DC9EBF),
	SPH_C64(0xCACAC5D41EC5CA43), SPH_C64(0x2D2D995875992D77),
	SPH_C64(0xBFBF792E9179BFDC), SPH_C64(0x07071B3F381B0709),
	SPH_C64(0xADAD23AC0123ADEA), SPH_C64(0x5A5A2FB0EA2F5AEE),
	SPH_C64(0x8383B5EF6CB58398), SPH_C64(0x3333FFB685FF3355),
	SPH_C64(0x6363F25C3FF263A5), SPH_C64(0x02020A12100A0206),
	SPH_C64(0xAAAA38933938AAE3), SPH_C64(0x7171A8DEAFA87193),
	SPH_C64(0xC8C8CFC60ECFC845), SPH_C64(0x19197DD1C87D192B),
	SPH_C64(0x4949703B727049DB), SPH_C64(0xD9D99A5F869AD976),
	SPH_C64(0xF2F21D31C31DF20B), SPH_C64(0xE3E348A84B48E338),
	SPH_C64(0x5B5B2AB9E22A5BED), SPH_C64(0x888892BC34928885),
	SPH_C64(0x9A9AC83EA4C89AB3), SPH_C64(0x2626BE0B2DBE266A),
	SPH_C64(0x3232FABF8DFA3256), SPH_C64(0xB0B04A59E94AB0CD),
	SPH_C64(0xE9E96AF21B6AE926), SPH_C64(0x0F0F337778330F11),
	SPH_C64(0xD5D5A633E6A6D562), SPH_C64(0x8080BAF474BA809D),
	SPH_C64(0xBEBE7C27997CBEDF), SPH_C64(0xCDCDDEEB26DECD4A),
	SPH_C64(0x3434E489BDE4345C), SPH_C64(0x484875327A7548D8),
	SPH_C64(0xFFFF2454AB24FF1C), SPH_C64(0x7A7A8F8DF78F7A8E),
	SPH_C64(0x9090EA64F4EA90AD), SPH_C64(0x5F5F3E9DC23E5FE1),
	SPH_C64(0x2020A03D1DA02060), SPH_C64(0x6868D50F67D568B8),
	SPH_C64(0x1A1A72CAD0721A2E), SPH_C64(0xAEAE2CB7192CAEEF),
	SPH_C64(0xB4B45E7DC95EB4C1), SPH_C64(0x545419CE9A1954FC),
	SPH_C64(0x9393E57FECE593A8), SPH_C64(0x2222AA2F0DAA2266),
	SPH_C64(0x6464E96307E964AC), SPH_C64(0xF1F1122ADB12F10E),
	SPH_C64(0x7373A2CCBFA27395), SPH_C64(0x12125A82905A1236),
	SPH_C64(0x40405D7A3A5D40C0), SPH_C64(0x0808284840280818),
	SPH_C64(0xC3C3E89556E8C358), SPH_C64(0xECEC7BDF337BEC29),
	SPH_C64(0xDBDB904D9690DB70), SPH_C64(0xA1A11FC0611FA1FE),
	SPH_C64(0x8D8D83911C838D8A), SPH_C64(0x3D3DC9C8F5C93D47),
	SPH_C64(0x9797F15BCCF197A4), SPH_C64(0x0000000000000000),
	SPH_C64(0xCFCFD4F936D4CF4C), SPH_C64(0x2B2B876E45872B7D),
	SPH_C64(0x7676B3E197B3769A), SPH_C64(0x8282B0E664B0829B),
	SPH_C64(0xD6D6A928FEA9D667), SPH_C64(0x1B1B77C3D8771B2D),
	SPH_C64(0xB5B55B74C15BB5C2), SPH_C64(0xAFAF29BE1129AFEC),
	SPH_C64(0x6A6ADF1D77DF6ABE), SPH_C64(0x50500DEABA0D50F0),
	SPH_C64(0x45454C57124C45CF), SPH_C64(0xF3F31838CB18F308),
	SPH_C64(0x3030F0AD9DF03050), SPH_C64(0xEFEF74C42B74EF2C),
	SPH_C64(0x3F3FC3DAE5C33F41), SPH_C64(0x55551CC7921C55FF),
	SPH_C64(0xA2A210DB7910A2FB), SPH_C64(0xEAEA65E90365EA23),
	SPH_C64(0x6565EC6A0FEC65AF), SPH_C64(0xBABA6803B968BAD3),
	SPH_C64(0x2F2F934A65932F71), SPH_C64(0xC0C0E78E4EE7C05D),
	SPH_C64(0xDEDE8160BE81DE7F), SPH_C64(0x1C1C6CFCE06C1C24),
	SPH_C64(0xFDFD2E46BB2EFD1A), SPH_C64(0x4D4D641F52644DD7),
	SPH_C64(0x9292E076E4E092AB), SPH_C64(0x7575BCFA8FBC759F),
	SPH_C64(0x06061E36301E060A), SPH_C64(0x8A8A98AE24988A83),
	SPH_C64(0xB2B2404BF940B2CB), SPH_C64(0xE6E659856359E637),
	SPH_C64(0x0E0E367E70360E12), SPH_C64(0x1F1F63E7F8631F21),
	SPH_C64(0x6262F75537F762A6), SPH_C64(0xD4D4A33AEEA3D461),
	SPH_C64(0xA8A832812932A8E5), SPH_C64(0x9696F452C4F496A7),
	SPH_C64(0xF9F93A629B3AF916), SPH_C64(0xC5C5F6A366F6C552),
	SPH_C64(0x2525B11035B1256F), SPH_C64(0x595920ABF22059EB),
	SPH_C64(0x8484AED054AE8491), SPH_C64(0x7272A7C5B7A77296),
	SPH_C64(0x3939DDECD5DD394B), SPH_C64(0x4C4C61165A614CD4),
	SPH_C64(0x5E5E3B94CA3B5EE2), SPH_C64(0x7878859FE7857888),
	SPH_C64(0x3838D8E5DDD83848), SPH_C64(0x8C8C869814868C89),
	SPH_C64(0xD1D1B217C6B2D16E), SPH_C64(0xA5A50BE4410BA5F2),
	SPH_C64(0xE2E24DA1434DE23B), SPH_C64(0x6161F84E2FF861A3),
	SPH_C64(0xB3B34542F145B3C8), SPH_C64(0x2121A53415A52163),
	SPH_C64(0x9C9CD60894D69CB9), SPH_C64(0x1E1E66EEF0661E22),
	SPH_C64(0x43435261225243C5), SPH_C64(0xC7C7FCB176FCC754),
	SPH_C64(0xFCFC2B4FB32BFC19), SPH_C64(0x040414242014040C),
	SPH_C64(0x515108E3B20851F3), SPH_C64(0x9999C725BCC799B6),
	SPH_C64(0x6D6DC4224FC46DB7), SPH_C64(0x0D0D396568390D17),
	SPH_C64(0xFAFA35798335FA13), SPH_C64(0xDFDF8469B684DF7C),
	SPH_C64(0x7E7E9BA9D79B7E82), SPH_C64(0x2424B4193DB4246C),
	SPH_C64(0x3B3BD7FEC5D73B4D), SPH_C64(0xABAB3D9A313DABE0),
	SPH_C64(0xCECED1F03ED1CE4F), SPH_C64(0x1111559988551133),
	SPH_C64(0x8F8F89830C898F8C), SPH_C64(0x4E4E6B044A6B4ED2),
	SPH_C64(0xB7B75166D151B7C4), SPH_C64(0xEBEB60E00B60EB20),
	SPH_C64(0x3C3CCCC1FDCC3C44), SPH_C64(0x8181BFFD7CBF819E),
	SPH_C64(0x9494FE40D4FE94A1), SPH_C64(0xF7F70C1CEB0CF704),
	SPH_C64(0xB9B96718A167B9D6), SPH_C64(0x13135F8B985F1335),
	SPH_C64(0x2C2C9C517D9C2C74), SPH_C64(0xD3D3B805D6B8D368),
	SPH_C64(0xE7E75C8C6B5CE734), SPH_C64(0x6E6ECB3957CB6EB2),
	SPH_C64(0xC4C4F3AA6EF3C451), SPH_C64(0x03030F1B180F0305),
	SPH_C64(0x565613DC8A1356FA), SPH_C64(0x4444495E1A4944CC),
	SPH_C64(0x7F7F9EA0DF9E7F81), SPH_C64(0xA9A937882137A9E6),
	SPH_C64(0x2A2A82674D822A7E), SPH_C64(0xBBBB6D0AB16DBBD0),
	SPH_C64(0xC1C1E28746E2C15E), SPH_C64(0x535302F1A20253F5),
	SPH_C64(0xDCDC8B72AE8BDC79), SPH_C64(0x0B0B275358270B1D),
	SPH_C64(0x9D9DD3019CD39DBA), SPH_C64(0x6C6CC12B47C16CB4),
	SPH_C64(0x3131F5A495F53153), SPH_C64(0x7474B9F387B9749C),
	SPH_C64(0xF6F60915E309F607), SPH_C64(0x4646434C0A4346CA),
	SPH_C64(0xACAC26A50926ACE9), SPH_C64(0x898997B53C978986),
	SPH_C64(0x141444B4A044143C), SPH_C64(0xE1E142BA5B42E13E),
	SPH_C64(0x16164EA6B04E163A), SPH_C64(0x3A3AD2F7CDD23A4E),
	SPH_C64(0x6969D0066FD069BB), SPH_C64(0x09092D41482D091B),
	SPH_C64(0x7070ADD7A7AD7090), SPH_C64(0xB6B6546FD954B6C7),
	SPH_C64(0xD0D0B71ECEB7D06D), SPH_C64(0xEDED7ED63B7EED2A),
	SPH_C64(0xCCCCDBE22EDBCC49), SPH_C64(0x424257682A5742C6),
	SPH_C64(0x9898C22CB4C298B5), SPH_C64(0xA4A40EED490EA4F1),
	SPH_C64(0x282888755D882878), SPH_C64(0x5C5C3186DA315CE4),
	SPH_C64(0xF8F83F6B933FF815), SPH_C64(0x8686A4C244A48697)
};

static const sph_u64 old1_T7[256] = {
	SPH_C64(0x1878D8C078182818), SPH_C64(0x23AF2605AF236523),
	SPH_C64(0xC6F9B87EF9C657C6), SPH_C64(0xE86FFB136FE825E8),
	SPH_C64(0x87A1CB4CA1879487), SPH_C64(0xB86211A962B8D5B8),
	SPH_C64(0x0105090805010301), SPH_C64(0x4F6E0D426E4FD14F),
	SPH_C64(0x36EE9BADEE365A36), SPH_C64(0xA604FF5904A6F7A6),
	SPH_C64(0xD2BD0CDEBDD26BD2), SPH_C64(0xF5060EFB06F502F5),
	SPH_C64(0x798096EF80798B79), SPH_C64(0x6FCE305FCE6FB16F),
	SPH_C64(0x91EF6DFCEF91AE91), SPH_C64(0x5207F8AA0752F652),
	SPH_C64(0x60FD4727FD60A060), SPH_C64(0xBC76358976BCD9BC),
	SPH_C64(0x9BCD37ACCD9BB09B), SPH_C64(0x8E8C8A048C8E8F8E),
	SPH_C64(0xA315D27115A3F8A3), SPH_C64(0x0C3C6C603C0C140C),
	SPH_C64(0x7B8A84FF8A7B8D7B), SPH_C64(0x35E180B5E1355F35),
	SPH_C64(0x1D69F5E8691D271D), SPH_C64(0xE047B35347E03DE0),
	SPH_C64(0xD7AC21F6ACD764D7), SPH_C64(0xC2ED9C5EEDC25BC2),
	SPH_C64(0x2E96436D962E722E), SPH_C64(0x4B7A29627A4BDD4B),
	SPH_C64(0xFE215DA321FE1FFE), SPH_C64(0x5716D5821657F957),
	SPH_C64(0x1541BDA841153F15), SPH_C64(0x77B6E89FB6779977),
	SPH_C64(0x37EB92A5EB375937), SPH_C64(0xE5569E7B56E532E5),
	SPH_C64(0x9FD9138CD99FBC9F), SPH_C64(0xF01723D317F00DF0),
	SPH_C64(0x4A7F206A7F4ADE4A), SPH_C64(0xDA95449E95DA73DA),
	SPH_C64(0x5825A2FA2558E858), SPH_C64(0xC9CACF06CAC946C9),
	SPH_C64(0x298D7C558D297B29), SPH_C64(0x0A225A50220A1E0A),
	SPH_C64(0xB14F50E14FB1CEB1), SPH_C64(0xA01AC9691AA0FDA0),
	SPH_C64(0x6BDA147FDA6BBD6B), SPH_C64(0x85ABD95CAB859285),
	SPH_C64(0xBD733C8173BDDABD), SPH_C64(0x5D348FD2345DE75D),
	SPH_C64(0x1050908050103010), SPH_C64(0xF40307F303F401F4),
	SPH_C64(0xCBC0DD16C0CB40CB), SPH_C64(0x3EC6D3EDC63E423E),
	SPH_C64(0x05112D2811050F05), SPH_C64(0x67E6781FE667A967),
	SPH_C64(0xE453977353E431E4), SPH_C64(0x27BB0225BB276927),
	SPH_C64(0x415873325841C341), SPH_C64(0x8B9DA72C9D8B808B),
	SPH_C64(0xA701F65101A7F4A7), SPH_C64(0x7D94B2CF947D877D),
	SPH_C64(0x95FB49DCFB95A295), SPH_C64(0xD89F568E9FD875D8),
	SPH_C64(0xFB30708B30FB10FB), SPH_C64(0xEE71CD2371EE2FEE),
	SPH_C64(0x7C91BBC7917C847C), SPH_C64(0x66E37117E366AA66),
	SPH_C64(0xDD8E7BA68EDD7ADD), SPH_C64(0x174BAFB84B173917),
	SPH_C64(0x474645024647C947), SPH_C64(0x9EDC1A84DC9EBF9E),
	SPH_C64(0xCAC5D41EC5CA43CA), SPH_C64(0x2D995875992D772D),
	SPH_C64(0xBF792E9179BFDCBF), SPH_C64(0x071B3F381B070907),
	SPH_C64(0xAD23AC0123ADEAAD), SPH_C64(0x5A2FB0EA2F5AEE5A),
	SPH_C64(0x83B5EF6CB5839883), SPH_C64(0x33FFB685FF335533),
	SPH_C64(0x63F25C3FF263A563), SPH_C64(0x020A12100A020602),
	SPH_C64(0xAA38933938AAE3AA), SPH_C64(0x71A8DEAFA8719371),
	SPH_C64(0xC8CFC60ECFC845C8), SPH_C64(0x197DD1C87D192B19),
	SPH_C64(0x49703B727049DB49), SPH_C64(0xD99A5F869AD976D9),
	SPH_C64(0xF21D31C31DF20BF2), SPH_C64(0xE348A84B48E338E3),
	SPH_C64(0x5B2AB9E22A5BED5B), SPH_C64(0x8892BC3492888588),
	SPH_C64(0x9AC83EA4C89AB39A), SPH_C64(0x26BE0B2DBE266A26),
	SPH_C64(0x32FABF8DFA325632), SPH_C64(0xB04A59E94AB0CDB0),
	SPH_C64(0xE96AF21B6AE926E9), SPH_C64(0x0F337778330F110F),
	SPH_C64(0xD5A633E6A6D562D5), SPH_C64(0x80BAF474BA809D80),
	SPH_C64(0xBE7C27997CBEDFBE), SPH_C64(0xCDDEEB26DECD4ACD),
	SPH_C64(0x34E489BDE4345C34), SPH_C64(0x4875327A7548D848),
	SPH_C64(0xFF2454AB24FF1CFF), SPH_C64(0x7A8F8DF78F7A8E7A),
	SPH_C64(0x90EA64F4EA90AD90), SPH_C64(0x5F3E9DC23E5FE15F),
	SPH_C64(0x20A03D1DA0206020), SPH_C64(0x68D50F67D568B868),
	SPH_C64(0x1A72CAD0721A2E1A), SPH_C64(0xAE2CB7192CAEEFAE),
	SPH_C64(0xB45E7DC95EB4C1B4), SPH_C64(0x5419CE9A1954FC54),
	SPH_C64(0x93E57FECE593A893), SPH_C64(0x22AA2F0DAA226622),
	SPH_C64(0x64E96307E964AC64), SPH_C64(0xF1122ADB12F10EF1),
	SPH_C64(0x73A2CCBFA2739573), SPH_C64(0x125A82905A123612),
	SPH_C64(0x405D7A3A5D40C040), SPH_C64(0x0828484028081808),
	SPH_C64(0xC3E89556E8C358C3), SPH_C64(0xEC7BDF337BEC29EC),
	SPH_C64(0xDB904D9690DB70DB), SPH_C64(0xA11FC0611FA1FEA1),
	SPH_C64(0x8D83911C838D8A8D), SPH_C64(0x3DC9C8F5C93D473D),
	SPH_C64(0x97F15BCCF197A497), SPH_C64(0x0000000000000000),
	SPH_C64(0xCFD4F936D4CF4CCF), SPH_C64(0x2B876E45872B7D2B),
	SPH_C64(0x76B3E197B3769A76), SPH_C64(0x82B0E664B0829B82),
	SPH_C64(0xD6A928FEA9D667D6), SPH_C64(0x1B77C3D8771B2D1B),
	SPH_C64(0xB55B74C15BB5C2B5), SPH_C64(0xAF29BE1129AFECAF),
	SPH_C64(0x6ADF1D77DF6ABE6A), SPH_C64(0x500DEABA0D50F050),
	SPH_C64(0x454C57124C45CF45), SPH_C64(0xF31838CB18F308F3),
	SPH_C64(0x30F0AD9DF0305030), SPH_C64(0xEF74C42B74EF2CEF),
	SPH_C64(0x3FC3DAE5C33F413F), SPH_C64(0x551CC7921C55FF55),
	SPH_C64(0xA210DB7910A2FBA2), SPH_C64(0xEA65E90365EA23EA),
	SPH_C64(0x65EC6A0FEC65AF65), SPH_C64(0xBA6803B968BAD3BA),
	SPH_C64(0x2F934A65932F712F), SPH_C64(0xC0E78E4EE7C05DC0),
	SPH_C64(0xDE8160BE81DE7FDE), SPH_C64(0x1C6CFCE06C1C241C),
	SPH_C64(0xFD2E46BB2EFD1AFD), SPH_C64(0x4D641F52644DD74D),
	SPH_C64(0x92E076E4E092AB92), SPH_C64(0x75BCFA8FBC759F75),
	SPH_C64(0x061E36301E060A06), SPH_C64(0x8A98AE24988A838A),
	SPH_C64(0xB2404BF940B2CBB2), SPH_C64(0xE659856359E637E6),
	SPH_C64(0x0E367E70360E120E), SPH_C64(0x1F63E7F8631F211F),
	SPH_C64(0x62F75537F762A662), SPH_C64(0xD4A33AEEA3D461D4),
	SPH_C64(0xA832812932A8E5A8), SPH_C64(0x96F452C4F496A796),
	SPH_C64(0xF93A629B3AF916F9), SPH_C64(0xC5F6A366F6C552C5),
	SPH_C64(0x25B11035B1256F25), SPH_C64(0x5920ABF22059EB59),
	SPH_C64(0x84AED054AE849184), SPH_C64(0x72A7C5B7A7729672),
	SPH_C64(0x39DDECD5DD394B39), SPH_C64(0x4C61165A614CD44C),
	SPH_C64(0x5E3B94CA3B5EE25E), SPH_C64(0x78859FE785788878),
	SPH_C64(0x38D8E5DDD8384838), SPH_C64(0x8C869814868C898C),
	SPH_C64(0xD1B217C6B2D16ED1), SPH_C64(0xA50BE4410BA5F2A5),
	SPH_C64(0xE24DA1434DE23BE2), SPH_C64(0x61F84E2FF861A361),
	SPH_C64(0xB34542F145B3C8B3), SPH_C64(0x21A53415A5216321),
	SPH_C64(0x9CD60894D69CB99C), SPH_C64(0x1E66EEF0661E221E),
	SPH_C64(0x435261225243C543), SPH_C64(0xC7FCB176FCC754C7),
	SPH_C64(0xFC2B4FB32BFC19FC), SPH_C64(0x0414242014040C04),
	SPH_C64(0x5108E3B20851F351), SPH_C64(0x99C725BCC799B699),
	SPH_C64(0x6DC4224FC46DB76D), SPH_C64(0x0D396568390D170D),
	SPH_C64(0xFA35798335FA13FA), SPH_C64(0xDF8469B684DF7CDF),
	SPH_C64(0x7E9BA9D79B7E827E), SPH_C64(0x24B4193DB4246C24),
	SPH_C64(0x3BD7FEC5D73B4D3B), SPH_C64(0xAB3D9A313DABE0AB),
	SPH_C64(0xCED1F03ED1CE4FCE), SPH_C64(0x1155998855113311),
	SPH_C64(0x8F89830C898F8C8F), SPH_C64(0x4E6B044A6B4ED24E),
	SPH_C64(0xB75166D151B7C4B7), SPH_C64(0xEB60E00B60EB20EB),
	SPH_C64(0x3CCCC1FDCC3C443C), SPH_C64(0x81BFFD7CBF819E81),
	SPH_C64(0x94FE40D4FE94A194), SPH_C64(0xF70C1CEB0CF704F7),
	SPH_C64(0xB96718A167B9D6B9), SPH_C64(0x135F8B985F133513),
	SPH_C64(0x2C9C517D9C2C742C), SPH_C64(0xD3B805D6B8D368D3),
	SPH_C64(0xE75C8C6B5CE734E7), SPH_C64(0x6ECB3957CB6EB26E),
	SPH_C64(0xC4F3AA6EF3C451C4), SPH_C64(0x030F1B180F030503),
	SPH_C64(0x5613DC8A1356FA56), SPH_C64(0x44495E1A4944CC44),
	SPH_C64(0x7F9EA0DF9E7F817F), SPH_C64(0xA937882137A9E6A9),
	SPH_C64(0x2A82674D822A7E2A), SPH_C64(0xBB6D0AB16DBBD0BB),
	SPH_C64(0xC1E28746E2C15EC1), SPH_C64(0x5302F1A20253F553),
	SPH_C64(0xDC8B72AE8BDC79DC), SPH_C64(0x0B275358270B1D0B),
	SPH_C64(0x9DD3019CD39DBA9D), SPH_C64(0x6CC12B47C16CB46C),
	SPH_C64(0x31F5A495F5315331), SPH_C64(0x74B9F387B9749C74),
	SPH_C64(0xF60915E309F607F6), SPH_C64(0x46434C0A4346CA46),
	SPH_C64(0xAC26A50926ACE9AC), SPH_C64(0x8997B53C97898689),
	SPH_C64(0x1444B4A044143C14), SPH_C64(0xE142BA5B42E13EE1),
	SPH_C64(0x164EA6B04E163A16), SPH_C64(0x3AD2F7CDD23A4E3A),
	SPH_C64(0x69D0066FD069BB69), SPH_C64(0x092D41482D091B09),
	SPH_C64(0x70ADD7A7AD709070), SPH_C64(0xB6546FD954B6C7B6),
	SPH_C64(0xD0B71ECEB7D06DD0), SPH_C64(0xED7ED63B7EED2AED),
	SPH_C64(0xCCDBE22EDBCC49CC), SPH_C64(0x4257682A5742C642),
	SPH_C64(0x98C22CB4C298B598), SPH_C64(0xA40EED490EA4F1A4),
	SPH_C64(0x2888755D88287828), SPH_C64(0x5C3186DA315CE45C),
	SPH_C64(0xF83F6B933FF815F8), SPH_C64(0x86A4C244A4869786)
};

#endif

static const sph_u64 old1_RC[10] = {
	SPH_C64(0x4F01B887E8C62318),
	SPH_C64(0x52916F79F5D2A636),
	SPH_C64(0x357B0CA38E9BBC60),
	SPH_C64(0x57FE4B2EC2D7E01D),
	SPH_C64(0xDA4AF09FE5377715),
	SPH_C64(0x856BA0B10A29C958),
	SPH_C64(0x67053ECBF4105DBD),
	SPH_C64(0xD8957DA78B4127E4),
	SPH_C64(0x9E4717DD667CEEFB),
	SPH_C64(0x33835AAD07BF2DCA)
};

/* ====================================================================== */

#define DECL8(z)   sph_u64 z ## 0, z ## 1, z ## 2, z ## 3, \
                   z ## 4, z ## 5, z ## 6, z ## 7

#if SPH_LITTLE_FAST
#define READ_DATA_W(x)   do { \
		n ## x = sph_dec64le_aligned( \
			(const unsigned char *)src + 8 * (x)); \
	} while (0)
#define UPDATE_STATE_W(x)   do { \
		state[x] ^= n ## x ^ sph_dec64le_aligned( \
			(const unsigned char *)src + 8 * (x)); \
	} while (0)
#define LVARS   DECL8(n); DECL8(h);
#else
#define READ_DATA_W(x)   do { \
		sn ## x = n ## x = sph_dec64le_aligned( \
			(const unsigned char *)src + 8 * (x)); \
	} while (0)
#define UPDATE_STATE_W(x)   do { \
		state[x] ^= n ## x ^ sn ## x; \
	} while (0)
#define LVARS   DECL8(n); DECL8(sn); DECL8(h);
#endif

#define READ_STATE_W(x)   do { h ## x = state[x]; } while (0)

#define MUL8(FUN)   do { \
		FUN(0); \
		FUN(1); \
		FUN(2); \
		FUN(3); \
		FUN(4); \
		FUN(5); \
		FUN(6); \
		FUN(7); \
	} while (0)

/*
 * First operation: XOR the input data with the first round key.
 */
#define ROUND0_W(x)   do { \
		n ## x ^= h ## x; \
	} while (0)

#define READ_DATA      MUL8(READ_DATA_W)
#define READ_STATE     MUL8(READ_STATE_W)
#define ROUND0         MUL8(ROUND0_W)
#define UPDATE_STATE   MUL8(UPDATE_STATE_W)

#define BYTE(x, n)     ((unsigned)((x) >> (8 * (n))) & 0xFF)

#if SPH_SMALL_FOOTPRINT_WHIRLPOOL

static SPH_INLINE sph_u64
table_skew(sph_u64 val, int num)
{
	return SPH_ROTL64(val, 8 * num);
}

#define ROUND_ELT(table, in, i0, i1, i2, i3, i4, i5, i6, i7) \
	(table ## 0[BYTE(in ## i0, 0)] \
	^ table_skew(table ## 0[BYTE(in ## i1, 1)], 1) \
	^ table_skew(table ## 0[BYTE(in ## i2, 2)], 2) \
	^ table_skew(table ## 0[BYTE(in ## i3, 3)], 3) \
	^ table_skew(table ## 0[BYTE(in ## i4, 4)], 4) \
	^ table_skew(table ## 0[BYTE(in ## i5, 5)], 5) \
	^ table_skew(table ## 0[BYTE(in ## i6, 6)], 6) \
	^ table_skew(table ## 0[BYTE(in ## i7, 7)], 7))
#else
#define ROUND_ELT(table, in, i0, i1, i2, i3, i4, i5, i6, i7) \
	(table ## 0[BYTE(in ## i0, 0)] \
	^ table ## 1[BYTE(in ## i1, 1)] \
	^ table ## 2[BYTE(in ## i2, 2)] \
	^ table ## 3[BYTE(in ## i3, 3)] \
	^ table ## 4[BYTE(in ## i4, 4)] \
	^ table ## 5[BYTE(in ## i5, 5)] \
	^ table ## 6[BYTE(in ## i6, 6)] \
	^ table ## 7[BYTE(in ## i7, 7)])
#endif

#define ROUND(table, in, out, c0, c1, c2, c3, c4, c5, c6, c7)   do { \
		out ## 0 = ROUND_ELT(table, in, 0, 7, 6, 5, 4, 3, 2, 1) ^ c0; \
		out ## 1 = ROUND_ELT(table, in, 1, 0, 7, 6, 5, 4, 3, 2) ^ c1; \
		out ## 2 = ROUND_ELT(table, in, 2, 1, 0, 7, 6, 5, 4, 3) ^ c2; \
		out ## 3 = ROUND_ELT(table, in, 3, 2, 1, 0, 7, 6, 5, 4) ^ c3; \
		out ## 4 = ROUND_ELT(table, in, 4, 3, 2, 1, 0, 7, 6, 5) ^ c4; \
		out ## 5 = ROUND_ELT(table, in, 5, 4, 3, 2, 1, 0, 7, 6) ^ c5; \
		out ## 6 = ROUND_ELT(table, in, 6, 5, 4, 3, 2, 1, 0, 7) ^ c6; \
		out ## 7 = ROUND_ELT(table, in, 7, 6, 5, 4, 3, 2, 1, 0) ^ c7; \
	} while (0)

#define ROUND_KSCHED(table, in, out, c) \
	ROUND(table, in, out, c, 0, 0, 0, 0, 0, 0, 0)

#define ROUND_WENC(table, in, key, out) \
	ROUND(table, in, out, key ## 0, key ## 1, key ## 2, \
		key ## 3, key ## 4, key ## 5, key ## 6, key ## 7)

#define TRANSFER(dst, src)   do { \
		dst ## 0 = src ## 0; \
		dst ## 1 = src ## 1; \
		dst ## 2 = src ## 2; \
		dst ## 3 = src ## 3; \
		dst ## 4 = src ## 4; \
		dst ## 5 = src ## 5; \
		dst ## 6 = src ## 6; \
		dst ## 7 = src ## 7; \
	} while (0)

/* see sph_whirlpool.h */
void
sph_whirlpool_init(void *cc)
{
	sph_whirlpool_context *sc;

	sc = cc;
	/*
	 * We want to set all eight 64-bit words to 0. A "memset()"
	 * is not, theoretically, fully standard, but in practice it
	 * will work everywhere.
	 */
	memset(sc->state, 0, sizeof sc->state);
#if SPH_64
	sc->count = 0;
#else
	sc->count_high = sc->count_low = 0;
#endif
}

#define ROUND_FUN(name, type) \
static void \
name ## _round(const void *src, sph_u64 *state) \
{ \
	LVARS \
	int r; \
 \
	READ_DATA; \
	READ_STATE; \
	ROUND0; \
	for (r = 0; r < 10; r ++) { \
		DECL8(tmp); \
 \
		ROUND_KSCHED(type ## _T, h, tmp, type ## _RC[r]); \
		TRANSFER(h, tmp); \
		ROUND_WENC(type ## _T, n, h, tmp); \
		TRANSFER(n, tmp); \
	} \
	UPDATE_STATE; \
}

ROUND_FUN(whirlpool, plain)
ROUND_FUN(whirlpool0, old0)
ROUND_FUN(whirlpool1, old1)

/*
 * We want big-endian encoding of the message length, over 256 bits. BE64
 * triggers that. However, our block length is 512 bits, not 1024 bits.
 * Internally, our encoding/decoding is little-endian, which is not a
 * problem here since we also deactivate output in md_helper.c.
 */
#define BE64   1
#define SVAL   sc->state
#define BLEN   64U
#define PLW4   1

#define RFUN   whirlpool_round
#define HASH   whirlpool
#include "md_helper.c"
#undef RFUN
#undef HASH

#define RFUN   whirlpool0_round
#define HASH   whirlpool0
#include "md_helper.c"
#undef RFUN
#undef HASH

#define RFUN   whirlpool1_round
#define HASH   whirlpool1
#include "md_helper.c"
#undef RFUN
#undef HASH

#define MAKE_CLOSE(name) \
void \
sph_ ## name ## _close(void *cc, void *dst) \
{ \
	sph_ ## name ## _context *sc; \
	int i; \
 \
	name ## _close(cc, dst, 0); \
	sc = cc; \
	for (i = 0; i < 8; i ++) \
		sph_enc64le((unsigned char *)dst + 8 * i, sc->state[i]); \
	sph_ ## name ## _init(cc); \
}

MAKE_CLOSE(whirlpool)
MAKE_CLOSE(whirlpool0)
MAKE_CLOSE(whirlpool1)

#endif
