// Package playmoviespartner provides access to the Google Play Movies Partner API.
//
// See https://developers.google.com/playmoviespartner/
//
// Usage example:
//
//   import "google.golang.org/api/playmoviespartner/v1"
//   ...
//   playmoviespartnerService, err := playmoviespartner.New(oauthHttpClient)
package playmoviespartner // import "google.golang.org/api/playmoviespartner/v1"

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	context "golang.org/x/net/context"
	ctxhttp "golang.org/x/net/context/ctxhttp"
	gensupport "google.golang.org/api/gensupport"
	googleapi "google.golang.org/api/googleapi"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = ctxhttp.Do

const apiId = "playmoviespartner:v1"
const apiName = "playmoviespartner"
const apiVersion = "v1"
const basePath = "https://playmoviespartner.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// View the digital assets you publish on Google Play Movies and TV
	PlaymoviesPartnerReadonlyScope = "https://www.googleapis.com/auth/playmovies_partner.readonly"
)

func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Accounts = NewAccountsService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.Avails = NewAccountsAvailsService(s)
	rs.Components = NewAccountsComponentsService(s)
	rs.ExperienceLocales = NewAccountsExperienceLocalesService(s)
	rs.Orders = NewAccountsOrdersService(s)
	rs.StoreInfos = NewAccountsStoreInfosService(s)
	return rs
}

type AccountsService struct {
	s *Service

	Avails *AccountsAvailsService

	Components *AccountsComponentsService

	ExperienceLocales *AccountsExperienceLocalesService

	Orders *AccountsOrdersService

	StoreInfos *AccountsStoreInfosService
}

func NewAccountsAvailsService(s *Service) *AccountsAvailsService {
	rs := &AccountsAvailsService{s: s}
	return rs
}

type AccountsAvailsService struct {
	s *Service
}

func NewAccountsComponentsService(s *Service) *AccountsComponentsService {
	rs := &AccountsComponentsService{s: s}
	rs.Type = NewAccountsComponentsTypeService(s)
	return rs
}

type AccountsComponentsService struct {
	s *Service

	Type *AccountsComponentsTypeService
}

func NewAccountsComponentsTypeService(s *Service) *AccountsComponentsTypeService {
	rs := &AccountsComponentsTypeService{s: s}
	return rs
}

type AccountsComponentsTypeService struct {
	s *Service
}

func NewAccountsExperienceLocalesService(s *Service) *AccountsExperienceLocalesService {
	rs := &AccountsExperienceLocalesService{s: s}
	return rs
}

type AccountsExperienceLocalesService struct {
	s *Service
}

func NewAccountsOrdersService(s *Service) *AccountsOrdersService {
	rs := &AccountsOrdersService{s: s}
	return rs
}

type AccountsOrdersService struct {
	s *Service
}

func NewAccountsStoreInfosService(s *Service) *AccountsStoreInfosService {
	rs := &AccountsStoreInfosService{s: s}
	rs.Country = NewAccountsStoreInfosCountryService(s)
	return rs
}

type AccountsStoreInfosService struct {
	s *Service

	Country *AccountsStoreInfosCountryService
}

func NewAccountsStoreInfosCountryService(s *Service) *AccountsStoreInfosCountryService {
	rs := &AccountsStoreInfosCountryService{s: s}
	return rs
}

type AccountsStoreInfosCountryService struct {
	s *Service
}

// Avail: An Avail describes the Availability Window of a specific Edit
// in a given country, which means the period Google is allowed to sell
// or rent the Edit. Avails are exposed in EMA format Version 1.6b
// (available at http://www.movielabs.com/md/avails/) Studios can see
// the Avails for the Titles they own. Post-production houses cannot see
// any Avails.
type Avail struct {
	// AltId: Other identifier referring to the Edit, as defined by partner.
	// Example: "GOOGLER_2006"
	AltId string `json:"altId,omitempty"`

	// AvailId: ID internally generated by Google to uniquely identify an
	// Avail. Not part of EMA Specs.
	AvailId string `json:"availId,omitempty"`

	// CaptionExemption: Communicating an exempt category as defined by FCC
	// regulations. It is not required for non-US Avails. Example: "1"
	CaptionExemption string `json:"captionExemption,omitempty"`

	// CaptionIncluded: Communicating if caption file will be delivered.
	CaptionIncluded bool `json:"captionIncluded,omitempty"`

	// ContentId: Title Identifier. This should be the Title Level EIDR.
	// Example: "10.5240/1489-49A2-3956-4B2D-FE16-5".
	ContentId string `json:"contentId,omitempty"`

	// DisplayName: The name of the studio that owns the Edit referred in
	// the Avail. This is the equivalent of `studio_name` in other
	// resources, but it follows the EMA nomenclature. Example: "Google
	// Films".
	DisplayName string `json:"displayName,omitempty"`

	// EncodeId: Manifestation Identifier. This should be the Manifestation
	// Level EIDR. Example: "10.2340/1489-49A2-3956-4B2D-FE16-7"
	EncodeId string `json:"encodeId,omitempty"`

	// End: End of term in YYYY-MM-DD format in the timezone of the country
	// of the Avail. "Open" if no end date is available. Example:
	// "2019-02-17"
	End string `json:"end,omitempty"`

	// EpisodeAltId: Other identifier referring to the episode, as defined
	// by partner. Only available on TV avails. Example: "rs_googlers_s1_3".
	EpisodeAltId string `json:"episodeAltId,omitempty"`

	// EpisodeNumber: The number assigned to the episode within a season.
	// Only available on TV Avails. Example: "3".
	EpisodeNumber string `json:"episodeNumber,omitempty"`

	// EpisodeTitleInternalAlias: OPTIONAL.TV Only. Title used by involved
	// parties to refer to this episode. Only available on TV Avails.
	// Example: "Coding at Google".
	EpisodeTitleInternalAlias string `json:"episodeTitleInternalAlias,omitempty"`

	// FormatProfile: Indicates the format profile covered by the
	// transaction.
	//
	// Possible values:
	//   "FORMAT_PROFILE_UNSPECIFIED"
	//   "SD"
	//   "HD"
	FormatProfile string `json:"formatProfile,omitempty"`

	// LicenseType: Type of transaction.
	//
	// Possible values:
	//   "LICENSE_TYPE_UNSPECIFIED"
	//   "EST"
	//   "VOD"
	//   "SVOD"
	//   "POEST"
	LicenseType string `json:"licenseType,omitempty"`

	// PphNames: Name of the post-production houses that manage the Avail.
	// Not part of EMA Specs.
	PphNames []string `json:"pphNames,omitempty"`

	// PriceType: Type of pricing that should be applied to this Avail based
	// on how the partner classify them. Example: "Tier", "WSP", "SRP", or
	// "Category".
	PriceType string `json:"priceType,omitempty"`

	// PriceValue: Value to be applied to the pricing type. Example: "4" or
	// "2.99"
	PriceValue string `json:"priceValue,omitempty"`

	// ProductId: Edit Identifier. This should be the Edit Level EIDR.
	// Example: "10.2340/1489-49A2-3956-4B2D-FE16-6"
	ProductId string `json:"productId,omitempty"`

	// RatingReason: Value representing the rating reason. Rating reasons
	// should be formatted as per [EMA ratings
	// spec](http://www.movielabs.com/md/ratings/) and comma-separated for
	// inclusion of multiple reasons. Example: "L, S, V"
	RatingReason string `json:"ratingReason,omitempty"`

	// RatingSystem: Rating system applied to the version of title within
	// territory of Avail. Rating systems should be formatted as per [EMA
	// ratings spec](http://www.movielabs.com/md/ratings/) Example: "MPAA"
	RatingSystem string `json:"ratingSystem,omitempty"`

	// RatingValue: Value representing the rating. Ratings should be
	// formatted as per http://www.movielabs.com/md/ratings/ Example: "PG"
	RatingValue string `json:"ratingValue,omitempty"`

	// ReleaseDate: Release date of the Title in earliest released
	// territory. Typically it is just the year, but it is free-form as per
	// EMA spec. Examples: "1979", "Oct 2014"
	ReleaseDate string `json:"releaseDate,omitempty"`

	// SeasonAltId: Other identifier referring to the season, as defined by
	// partner. Only available on TV avails. Example: "rs_googlers_s1".
	SeasonAltId string `json:"seasonAltId,omitempty"`

	// SeasonNumber: The number assigned to the season within a series. Only
	// available on TV Avails. Example: "1".
	SeasonNumber string `json:"seasonNumber,omitempty"`

	// SeasonTitleInternalAlias: Title used by involved parties to refer to
	// this season. Only available on TV Avails. Example: "Googlers, The".
	SeasonTitleInternalAlias string `json:"seasonTitleInternalAlias,omitempty"`

	// SeriesAltId: Other identifier referring to the series, as defined by
	// partner. Only available on TV avails. Example: "rs_googlers".
	SeriesAltId string `json:"seriesAltId,omitempty"`

	// SeriesTitleInternalAlias: Title used by involved parties to refer to
	// this series. Only available on TV Avails. Example: "Googlers, The".
	SeriesTitleInternalAlias string `json:"seriesTitleInternalAlias,omitempty"`

	// Start: Start of term in YYYY-MM-DD format in the timezone of the
	// country of the Avail. Example: "2013-05-14".
	Start string `json:"start,omitempty"`

	// StoreLanguage: Spoken language of the intended audience. Language
	// shall be encoded in accordance with RFC 5646. Example: "fr".
	StoreLanguage string `json:"storeLanguage,omitempty"`

	// SuppressionLiftDate: First date an Edit could be publically announced
	// as becoming available at a specific future date in territory of
	// Avail. *Not* the Avail start date or pre-order start date. Format is
	// YYYY-MM-DD. Only available for pre-orders. Example: "2012-12-10"
	SuppressionLiftDate string `json:"suppressionLiftDate,omitempty"`

	// Territory: ISO 3166-1 alpha-2 country code for the country or
	// territory of this Avail. For Avails, we use Territory in lieu of
	// Country to comply with EMA specifications. But please note that
	// Territory and Country identify the same thing. Example: "US".
	Territory string `json:"territory,omitempty"`

	// TitleInternalAlias: Title used by involved parties to refer to this
	// content. Example: "Googlers, The". Only available on Movie Avails.
	TitleInternalAlias string `json:"titleInternalAlias,omitempty"`

	// VideoId: Google-generated ID identifying the video linked to this
	// Avail, once delivered. Not part of EMA Specs. Example: 'gtry456_xc'
	VideoId string `json:"videoId,omitempty"`

	// WorkType: Work type as enumerated in EMA.
	//
	// Possible values:
	//   "TITLE_TYPE_UNSPECIFIED"
	//   "MOVIE"
	//   "SEASON"
	//   "EPISODE"
	//   "BUNDLE"
	WorkType string `json:"workType,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AltId") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *Avail) MarshalJSON() ([]byte, error) {
	type noMethod Avail
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// Component: A Component is an element (audio, video, subtitle,
// artwork, trailer, etc...) that is used in a set of ExperienceLocales.
// A Component is owned by a Studio and managed either by the Studio
// itself or by one Post-Production House. Each Component is identified
// by a `component_id` and its `type`.
type Component struct {
	// AltCutIds: List of Alternative Cut IDs, sometimes available in lieu
	// of the main Edit-level EIDR IDs. This is not an EIDR ID, but a
	// Partner-provided ID. Example: "206346_79838".
	AltCutIds []string `json:"altCutIds,omitempty"`

	// ApprovedTime: Timestamp when the Component was approved.
	ApprovedTime string `json:"approvedTime,omitempty"`

	// ComponentDetailType: Detail about the type of the Component.
	//
	// Possible values:
	//   "COMPONENT_DETAIL_TYPE_UNSPECIFIED"
	//   "COMPONENT_DETAIL_TYPE_NORMAL"
	//   "COMPONENT_DETAIL_TYPE_POSTER"
	//   "COMPONENT_DETAIL_TYPE_PRIMARY"
	//   "COMPONENT_DETAIL_TYPE_FORCED"
	//   "COMPONENT_DETAIL_TYPE_DUBBED"
	//   "COMPONENT_DETAIL_TYPE_SDH"
	ComponentDetailType string `json:"componentDetailType,omitempty"`

	// ComponentId: ID internally generated by Google to uniquely identify
	// the Component. Example: 'wteyrc_647xc'
	ComponentId string `json:"componentId,omitempty"`

	// CustomIds: List of custom IDs (defined by the partner) linked to the
	// ExperienceLocale using this Component. Example: "R86241"
	CustomIds []string `json:"customIds,omitempty"`

	// EditLevelEidrs: List of Edit-level EIDR IDs. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-6".
	EditLevelEidrs []string `json:"editLevelEidrs,omitempty"`

	// ElIds: IDs internally generated by Google to uniquely identify the
	// ExperienceLocales for which the Component is used. Example:
	// 'KRZiVjY9h7t'
	ElIds []string `json:"elIds,omitempty"`

	// Filename: File name of the Component when delivered.
	Filename string `json:"filename,omitempty"`

	// Language: Language of the component, using the "BCP 47" format.
	// Examples: "en", "en-US", "es", "es-419".
	Language string `json:"language,omitempty"`

	// Name: Default Edit name, usually in the language of the country of
	// origin. Example: "Googlers, The".
	Name string `json:"name,omitempty"`

	// NormalizedPriority: A simpler representation of the priority.
	//
	// Possible values:
	//   "NORMALIZED_PRIORITY_UNSPECIFIED"
	//   "LOW_PRIORITY"
	//   "HIGH_PRIORITY"
	NormalizedPriority string `json:"normalizedPriority,omitempty"`

	// PlayableUnitType: Type of the playable unit for which the Component
	// is intended.
	//
	// Possible values:
	//   "PLAYABLE_UNIT_TYPE_UNSPECIFIED"
	//   "PLAYABLE_UNIT_TYPE_FEATURE"
	//   "PLAYABLE_UNIT_TYPE_TRAILER"
	//   "PLAYBLE_UNIT_TYPE_BONUS_CONTENT"
	PlayableUnitType string `json:"playableUnitType,omitempty"`

	// PphName: Name of the post-production house that manages the
	// Component.
	PphName string `json:"pphName,omitempty"`

	// Priority: Component priority, as defined by Google. The higher the
	// value, the higher the priority.
	Priority float64 `json:"priority,omitempty"`

	// ProcessingErrors: Processing errors during XML file parsing. Example:
	// 'Invalid input file'
	ProcessingErrors []string `json:"processingErrors,omitempty"`

	// ReceivedTime: Timestamp when the Component was received.
	ReceivedTime string `json:"receivedTime,omitempty"`

	// RejectionNote: Notes explaining why a Component has been rejected.
	RejectionNote string `json:"rejectionNote,omitempty"`

	// Status: High-level status of the Component.
	//
	// Possible values:
	//   "STATUS_UNSPECIFIED"
	//   "STATUS_APPROVED"
	//   "STATUS_FAILED"
	//   "STATUS_PROCESSING"
	//   "STATUS_UNFULFILLED"
	//   "STATUS_NOT_AVAILABLE"
	Status string `json:"status,omitempty"`

	// StatusDetail: Detailed status of the Component
	//
	// Possible values:
	//   "COMPONENT_STATUS_UNSPECIFIED"
	//   "COMPONENT_STATUS_QC_APPROVED"
	//   "COMPONENT_STATUS_QC_REJECTION"
	//   "COMPONENT_STATUS_FILE_REJECTION"
	//   "COMPONENT_STATUS_FILE_PROCESSING"
	//   "COMPONENT_STATUS_READY_FOR_QC"
	StatusDetail string `json:"statusDetail,omitempty"`

	// StudioName: Name of the studio that owns the Component.
	StudioName string `json:"studioName,omitempty"`

	// TitleLevelEidrs: List of Title-level EIDR IDs. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-5".
	TitleLevelEidrs []string `json:"titleLevelEidrs,omitempty"`

	// Type: Type of the Component. Example: AUDIO_51
	//
	// Possible values:
	//   "COMPONENT_TYPE_UNSPECIFIED"
	//   "VIDEO"
	//   "AUDIO_20"
	//   "AUDIO_51"
	//   "SUBTITLE"
	//   "ARTWORK"
	//   "METADATA"
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AltCutIds") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *Component) MarshalJSON() ([]byte, error) {
	type noMethod Component
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ExperienceLocale: An ExperienceLocale tracks the fulfillment of a
// Title in a country using a specific language, when delivered using
// component-based delivery. For example, a Title in Switzerland might
// have 3 ExperienceLocales: they both share the same country ("CH"),
// but each has different languages ("de", "fr", and "it"). Each
// ExperienceLocale is uniquely identified by an `el_id`, which is
// generated by Google. Externally, an ExperienceLocale can also be
// identified by partners using its EIDR IDs, AltCutID or `custom_id`
// (when provided).
type ExperienceLocale struct {
	// AltCutId: Alternative Cut ID, sometimes available in lieu of the main
	// Edit-level EIDR ID. This is not an EIDR ID, but a Partner-provided
	// ID. Example: "206346_79838".
	AltCutId string `json:"altCutId,omitempty"`

	// ApprovedTime: Timestamp when the ExperienceLocale was approved.
	ApprovedTime string `json:"approvedTime,omitempty"`

	// ChannelId: YouTube Channel ID linked to the ExperienceLocale.
	// Example: "UCRG64darCZhb".
	ChannelId string `json:"channelId,omitempty"`

	// Country: Country where the ExperienceLocale is available, using the
	// "ISO 3166-1 alpha-2" format. Example: "US".
	Country string `json:"country,omitempty"`

	// CreatedTime: Timestamp when the ExperienceLocale was created.
	CreatedTime string `json:"createdTime,omitempty"`

	// CustomIds: List of custom IDs (defined by the partner) linked to this
	// ExperienceLocale. Example: "R86241"
	CustomIds []string `json:"customIds,omitempty"`

	// EarliestAvailStartTime: Timestamp of the earliest start date of the
	// Avails linked to this ExperienceLocale.
	EarliestAvailStartTime string `json:"earliestAvailStartTime,omitempty"`

	// EditLevelEidr: Edit-level EIDR ID. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-6".
	EditLevelEidr string `json:"editLevelEidr,omitempty"`

	// ElId: ID internally generated by Google to uniquely identify a
	// ExperienceLocale. Example: 'KRZiVjY9h7t'
	ElId string `json:"elId,omitempty"`

	// InventoryId: InventoryID as defined in the EMA specs.
	InventoryId string `json:"inventoryId,omitempty"`

	// Language: Language of the ExperienceLocale, using the "BCP 47"
	// format. Examples: "en", "en-US", "es", "es-419".
	Language string `json:"language,omitempty"`

	// Name: Default Edit name, usually in the language of the country of
	// origin. Example: "Googlers, The".
	Name string `json:"name,omitempty"`

	// NormalizedPriority: A simpler representation of the priority.
	//
	// Possible values:
	//   "NORMALIZED_PRIORITY_UNSPECIFIED"
	//   "LOW_PRIORITY"
	//   "HIGH_PRIORITY"
	NormalizedPriority string `json:"normalizedPriority,omitempty"`

	// PlayableSequenceId: PlayableSequenceID as defined in the EMA specs.
	PlayableSequenceId string `json:"playableSequenceId,omitempty"`

	// PphNames: Name of the post-production houses that manage the
	// ExperienceLocale.
	PphNames []string `json:"pphNames,omitempty"`

	// PresentationId: PresentationID as defined in the EMA specs.
	PresentationId string `json:"presentationId,omitempty"`

	// Priority: ExperienceLocale priority, as defined by Google. The higher
	// the value, the higher the priority. Example: 90
	Priority float64 `json:"priority,omitempty"`

	// Status: High-level status of the ExperienceLocale.
	//
	// Possible values:
	//   "STATUS_UNSPECIFIED"
	//   "STATUS_APPROVED"
	//   "STATUS_FAILED"
	//   "STATUS_PROCESSING"
	//   "STATUS_UNFULFILLED"
	//   "STATUS_NOT_AVAILABLE"
	Status string `json:"status,omitempty"`

	// StudioName: Name of the studio that owns the ExperienceLocale.
	StudioName string `json:"studioName,omitempty"`

	// TitleLevelEidr: Title-level EIDR ID. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-5".
	TitleLevelEidr string `json:"titleLevelEidr,omitempty"`

	// TrailerId: Trailer ID, as defined by Google, linked to the trailer
	// video in the ExperienceLocale. Example: 'gtry457_tr'.
	TrailerId string `json:"trailerId,omitempty"`

	// Type: Type of the Edit linked to the ExperienceLocale.
	//
	// Possible values:
	//   "TITLE_TYPE_UNSPECIFIED"
	//   "MOVIE"
	//   "SEASON"
	//   "EPISODE"
	//   "BUNDLE"
	Type string `json:"type,omitempty"`

	// VideoId: Video ID, as defined by Google, linked to the feature video
	// in the ExperienceLocale. Example: 'gtry456_xc'.
	VideoId string `json:"videoId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AltCutId") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ExperienceLocale) MarshalJSON() ([]byte, error) {
	type noMethod ExperienceLocale
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ListAvailsResponse: Response to the 'ListAvails' method.
type ListAvailsResponse struct {
	// Avails: List of Avails that match the request criteria.
	Avails []*Avail `json:"avails,omitempty"`

	// NextPageToken: See _List methods rules_ for info about this field.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// TotalSize: See _List methods rules_ for more information about this
	// field.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Avails") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ListAvailsResponse) MarshalJSON() ([]byte, error) {
	type noMethod ListAvailsResponse
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ListComponentsResponse: Response to the 'ListComponents' method.
type ListComponentsResponse struct {
	// Components: List of Components that match the request criteria.
	Components []*Component `json:"components,omitempty"`

	// NextPageToken: See _List methods rules_ for info about this field.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// TotalSize: See _List methods rules_ for more information about this
	// field.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Components") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ListComponentsResponse) MarshalJSON() ([]byte, error) {
	type noMethod ListComponentsResponse
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ListExperienceLocalesResponse: Response to the
// 'ListExperienceLocales' method.
type ListExperienceLocalesResponse struct {
	// ExperienceLocales: List of ExperienceLocales that match the request
	// criteria.
	ExperienceLocales []*ExperienceLocale `json:"experienceLocales,omitempty"`

	// NextPageToken: See _List methods rules_ for info about this field.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// TotalSize: See _List methods rules_ for more information about this
	// field.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ExperienceLocales")
	// to unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ListExperienceLocalesResponse) MarshalJSON() ([]byte, error) {
	type noMethod ListExperienceLocalesResponse
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ListOrdersResponse: Response to the 'ListOrders' method.
type ListOrdersResponse struct {
	// NextPageToken: See _List methods rules_ for info about this field.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// Orders: List of Orders that match the request criteria.
	Orders []*Order `json:"orders,omitempty"`

	// TotalSize: See _List methods rules_ for more information about this
	// field.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ListOrdersResponse) MarshalJSON() ([]byte, error) {
	type noMethod ListOrdersResponse
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// ListStoreInfosResponse: Response to the 'ListStoreInfos' method.
type ListStoreInfosResponse struct {
	// NextPageToken: See 'List methods rules' for info about this field.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// StoreInfos: List of StoreInfos that match the request criteria.
	StoreInfos []*StoreInfo `json:"storeInfos,omitempty"`

	// TotalSize: See _List methods rules_ for more information about this
	// field.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *ListStoreInfosResponse) MarshalJSON() ([]byte, error) {
	type noMethod ListStoreInfosResponse
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// Order: An Order tracks the fulfillment of an Edit when delivered
// using the legacy, non-component-based delivery. Each Order is
// uniquely identified by an `order_id`, which is generated by Google.
// Externally, Orders can also be identified by partners using its
// `custom_id` (when provided).
type Order struct {
	// ApprovedTime: Timestamp when the Order was approved.
	ApprovedTime string `json:"approvedTime,omitempty"`

	// ChannelId: YouTube Channel ID that should be used to fulfill the
	// Order. Example: "UCRG64darCZhb".
	ChannelId string `json:"channelId,omitempty"`

	// ChannelName: YouTube Channel Name that should be used to fulfill the
	// Order. Example: "Google_channel".
	ChannelName string `json:"channelName,omitempty"`

	// Countries: Countries where the Order is available, using the "ISO
	// 3166-1 alpha-2" format (example: "US").
	Countries []string `json:"countries,omitempty"`

	// CustomId: ID that can be used to externally identify an Order. This
	// ID is provided by partners when submitting the Avails. Example:
	// 'GOOGLER_2006'
	CustomId string `json:"customId,omitempty"`

	// EarliestAvailStartTime: Timestamp of the earliest start date of the
	// Avails linked to this Order.
	EarliestAvailStartTime string `json:"earliestAvailStartTime,omitempty"`

	// EpisodeName: Default Episode name, usually in the language of the
	// country of origin. Only available for TV Edits Example: "Googlers,
	// The - Pilot".
	EpisodeName string `json:"episodeName,omitempty"`

	// LegacyPriority: Legacy Order priority, as defined by Google. Example:
	// 'P0'
	LegacyPriority string `json:"legacyPriority,omitempty"`

	// Name: Default Edit name, usually in the language of the country of
	// origin. Example: "Googlers, The".
	Name string `json:"name,omitempty"`

	// NormalizedPriority: A simpler representation of the priority.
	//
	// Possible values:
	//   "NORMALIZED_PRIORITY_UNSPECIFIED"
	//   "LOW_PRIORITY"
	//   "HIGH_PRIORITY"
	NormalizedPriority string `json:"normalizedPriority,omitempty"`

	// OrderId: ID internally generated by Google to uniquely identify an
	// Order. Example: 'abcde12_x'
	OrderId string `json:"orderId,omitempty"`

	// OrderedTime: Timestamp when the Order was created.
	OrderedTime string `json:"orderedTime,omitempty"`

	// PphName: Name of the post-production house that manages the Edit
	// ordered.
	PphName string `json:"pphName,omitempty"`

	// Priority: Order priority, as defined by Google. The higher the value,
	// the higher the priority. Example: 90
	Priority float64 `json:"priority,omitempty"`

	// ReceivedTime: Timestamp when the Order was fulfilled.
	ReceivedTime string `json:"receivedTime,omitempty"`

	// RejectionNote: Field explaining why an Order has been rejected.
	// Example: "Trailer audio is 2ch mono, please re-deliver in stereo".
	RejectionNote string `json:"rejectionNote,omitempty"`

	// SeasonName: Default Season name, usually in the language of the
	// country of origin. Only available for TV Edits Example: "Googlers,
	// The - A Brave New World".
	SeasonName string `json:"seasonName,omitempty"`

	// ShowName: Default Show name, usually in the language of the country
	// of origin. Only available for TV Edits Example: "Googlers, The".
	ShowName string `json:"showName,omitempty"`

	// Status: High-level status of the order.
	//
	// Possible values:
	//   "STATUS_UNSPECIFIED"
	//   "STATUS_APPROVED"
	//   "STATUS_FAILED"
	//   "STATUS_PROCESSING"
	//   "STATUS_UNFULFILLED"
	//   "STATUS_NOT_AVAILABLE"
	Status string `json:"status,omitempty"`

	// StatusDetail: Detailed status of the order
	//
	// Possible values:
	//   "ORDER_STATUS_UNSPECIFIED"
	//   "ORDER_STATUS_QC_APPROVED"
	//   "ORDER_STATUS_QC_REJECTION"
	//   "ORDER_STATUS_INTERNAL_FIX"
	//   "ORDER_STATUS_OPEN_ORDER"
	//   "ORDER_STATUS_NOT_AVAILABLE"
	//   "ORDER_STATUS_AWAITING_REDELIVERY"
	//   "ORDER_STATUS_READY_FOR_QC"
	//   "ORDER_STATUS_FILE_PROCESSING"
	StatusDetail string `json:"statusDetail,omitempty"`

	// StudioName: Name of the studio that owns the Edit ordered.
	StudioName string `json:"studioName,omitempty"`

	// Type: Type of the Edit linked to the Order.
	//
	// Possible values:
	//   "TITLE_TYPE_UNSPECIFIED"
	//   "MOVIE"
	//   "SEASON"
	//   "EPISODE"
	//   "BUNDLE"
	Type string `json:"type,omitempty"`

	// VideoId: Google-generated ID identifying the video linked to this
	// Order, once delivered. Example: 'gtry456_xc'.
	VideoId string `json:"videoId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ApprovedTime") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *Order) MarshalJSON() ([]byte, error) {
	type noMethod Order
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// StoreInfo: Information about a playable sequence (video) associated
// with an Edit and available at the Google Play Store. Internally, each
// StoreInfo is uniquely identified by a `video_id` and `country`.
// Externally, Title-level EIDR or Edit-level EIDR, if provided, can
// also be used to identify a specific title or edit in a country.
type StoreInfo struct {
	// AudioTracks: Audio tracks available for this Edit.
	AudioTracks []string `json:"audioTracks,omitempty"`

	// Country: Country where Edit is available in ISO 3166-1 alpha-2
	// country code. Example: "US".
	Country string `json:"country,omitempty"`

	// EditLevelEidr: Edit-level EIDR ID. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-6".
	EditLevelEidr string `json:"editLevelEidr,omitempty"`

	// EpisodeNumber: The number assigned to the episode within a season.
	// Only available on TV Edits. Example: "1".
	EpisodeNumber string `json:"episodeNumber,omitempty"`

	// HasAudio51: Whether the Edit has a 5.1 channel audio track.
	HasAudio51 bool `json:"hasAudio51,omitempty"`

	// HasEstOffer: Whether the Edit has a EST offer.
	HasEstOffer bool `json:"hasEstOffer,omitempty"`

	// HasHdOffer: Whether the Edit has a HD offer.
	HasHdOffer bool `json:"hasHdOffer,omitempty"`

	// HasInfoCards: Whether the Edit has info cards.
	HasInfoCards bool `json:"hasInfoCards,omitempty"`

	// HasSdOffer: Whether the Edit has a SD offer.
	HasSdOffer bool `json:"hasSdOffer,omitempty"`

	// HasVodOffer: Whether the Edit has a VOD offer.
	HasVodOffer bool `json:"hasVodOffer,omitempty"`

	// LiveTime: Timestamp when the Edit went live on the Store.
	LiveTime string `json:"liveTime,omitempty"`

	// Mid: Knowledge Graph ID associated to this Edit, if available. This
	// ID links the Edit to its knowledge entity, externally accessible at
	// http://freebase.com. In the absense of Title EIDR or Edit EIDR, this
	// ID helps link together multiple Edits across countries. Example:
	// '/m/0ffx29'
	Mid string `json:"mid,omitempty"`

	// Name: Default Edit name, usually in the language of the country of
	// origin. Example: "Googlers, The".
	Name string `json:"name,omitempty"`

	// PphNames: Name of the post-production houses that manage the Edit.
	PphNames []string `json:"pphNames,omitempty"`

	// SeasonId: Google-generated ID identifying the season linked to the
	// Edit. Only available for TV Edits. Example: 'ster23ex'
	SeasonId string `json:"seasonId,omitempty"`

	// SeasonName: Default Season name, usually in the language of the
	// country of origin. Only available for TV Edits Example: "Googlers,
	// The - A Brave New World".
	SeasonName string `json:"seasonName,omitempty"`

	// SeasonNumber: The number assigned to the season within a show. Only
	// available on TV Edits. Example: "1".
	SeasonNumber string `json:"seasonNumber,omitempty"`

	// ShowId: Google-generated ID identifying the show linked to the Edit.
	// Only available for TV Edits. Example: 'et2hsue_x'
	ShowId string `json:"showId,omitempty"`

	// ShowName: Default Show name, usually in the language of the country
	// of origin. Only available for TV Edits Example: "Googlers, The".
	ShowName string `json:"showName,omitempty"`

	// StudioName: Name of the studio that owns the Edit ordered.
	StudioName string `json:"studioName,omitempty"`

	// Subtitles: Subtitles available for this Edit.
	Subtitles []string `json:"subtitles,omitempty"`

	// TitleLevelEidr: Title-level EIDR ID. Example:
	// "10.5240/1489-49A2-3956-4B2D-FE16-5".
	TitleLevelEidr string `json:"titleLevelEidr,omitempty"`

	// TrailerId: Google-generated ID identifying the trailer linked to the
	// Edit. Example: 'bhd_4e_cx'
	TrailerId string `json:"trailerId,omitempty"`

	// Type: Edit type, like Movie, Episode or Season.
	//
	// Possible values:
	//   "TITLE_TYPE_UNSPECIFIED"
	//   "MOVIE"
	//   "SEASON"
	//   "EPISODE"
	//   "BUNDLE"
	Type string `json:"type,omitempty"`

	// VideoId: Google-generated ID identifying the video linked to the
	// Edit. Example: 'gtry456_xc'
	VideoId string `json:"videoId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AudioTracks") to
	// unconditionally include in API requests. By default, fields with
	// empty values are omitted from API requests. However, any non-pointer,
	// non-interface field appearing in ForceSendFields will be sent to the
	// server regardless of whether the field is empty or not. This may be
	// used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`
}

func (s *StoreInfo) MarshalJSON() ([]byte, error) {
	type noMethod StoreInfo
	raw := noMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields)
}

// method id "playmoviespartner.accounts.avails.get":

type AccountsAvailsGetCall struct {
	s            *Service
	accountId    string
	availId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// Get: Get an Avail given its avail group id and avail id.
func (r *AccountsAvailsService) Get(accountId string, availId string) *AccountsAvailsGetCall {
	c := &AccountsAvailsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	c.availId = availId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsAvailsGetCall) Fields(s ...googleapi.Field) *AccountsAvailsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsAvailsGetCall) IfNoneMatch(entityTag string) *AccountsAvailsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsAvailsGetCall) Context(ctx context.Context) *AccountsAvailsGetCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsAvailsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/avails/{availId}")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
		"availId":   c.availId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.avails.get" call.
// Exactly one of *Avail or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Avail.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *AccountsAvailsGetCall) Do(opts ...googleapi.CallOption) (*Avail, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &Avail{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Get an Avail given its avail group id and avail id.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.avails.get",
	//   "parameterOrder": [
	//     "accountId",
	//     "availId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "availId": {
	//       "description": "REQUIRED. Avail ID.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/avails/{availId}",
	//   "response": {
	//     "$ref": "Avail"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// method id "playmoviespartner.accounts.avails.list":

type AccountsAvailsListCall struct {
	s            *Service
	accountId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// List: List Avails owned or managed by the partner. See
// _Authentication and Authorization rules_ and _List methods rules_ for
// more information about this method.
func (r *AccountsAvailsService) List(accountId string) *AccountsAvailsListCall {
	c := &AccountsAvailsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	return c
}

// AltId sets the optional parameter "altId": Filter Avails that match a
// case-insensitive, partner-specific custom id. NOTE: this field is
// deprecated and will be removed on V2; `alt_ids` should be used
// instead.
func (c *AccountsAvailsListCall) AltId(altId string) *AccountsAvailsListCall {
	c.urlParams_.Set("altId", altId)
	return c
}

// AltIds sets the optional parameter "altIds": Filter Avails that match
// (case-insensitive) any of the given partner-specific custom ids.
func (c *AccountsAvailsListCall) AltIds(altIds ...string) *AccountsAvailsListCall {
	c.urlParams_.SetMulti("altIds", append([]string{}, altIds...))
	return c
}

// PageSize sets the optional parameter "pageSize": See _List methods
// rules_ for info about this field.
func (c *AccountsAvailsListCall) PageSize(pageSize int64) *AccountsAvailsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": See _List methods
// rules_ for info about this field.
func (c *AccountsAvailsListCall) PageToken(pageToken string) *AccountsAvailsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// PphNames sets the optional parameter "pphNames": See _List methods
// rules_ for info about this field.
func (c *AccountsAvailsListCall) PphNames(pphNames ...string) *AccountsAvailsListCall {
	c.urlParams_.SetMulti("pphNames", append([]string{}, pphNames...))
	return c
}

// StudioNames sets the optional parameter "studioNames": See _List
// methods rules_ for info about this field.
func (c *AccountsAvailsListCall) StudioNames(studioNames ...string) *AccountsAvailsListCall {
	c.urlParams_.SetMulti("studioNames", append([]string{}, studioNames...))
	return c
}

// Territories sets the optional parameter "territories": Filter Avails
// that match (case-insensitive) any of the given country codes, using
// the "ISO 3166-1 alpha-2" format (examples: "US", "us", "Us").
func (c *AccountsAvailsListCall) Territories(territories ...string) *AccountsAvailsListCall {
	c.urlParams_.SetMulti("territories", append([]string{}, territories...))
	return c
}

// Title sets the optional parameter "title": Filter that matches Avails
// with a `title_internal_alias`, `series_title_internal_alias`,
// `season_title_internal_alias`, or `episode_title_internal_alias` that
// contains the given case-insensitive title.
func (c *AccountsAvailsListCall) Title(title string) *AccountsAvailsListCall {
	c.urlParams_.Set("title", title)
	return c
}

// VideoIds sets the optional parameter "videoIds": Filter Avails that
// match any of the given `video_id`s.
func (c *AccountsAvailsListCall) VideoIds(videoIds ...string) *AccountsAvailsListCall {
	c.urlParams_.SetMulti("videoIds", append([]string{}, videoIds...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsAvailsListCall) Fields(s ...googleapi.Field) *AccountsAvailsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsAvailsListCall) IfNoneMatch(entityTag string) *AccountsAvailsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsAvailsListCall) Context(ctx context.Context) *AccountsAvailsListCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsAvailsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/avails")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.avails.list" call.
// Exactly one of *ListAvailsResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListAvailsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsAvailsListCall) Do(opts ...googleapi.CallOption) (*ListAvailsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ListAvailsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List Avails owned or managed by the partner. See _Authentication and Authorization rules_ and _List methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.avails.list",
	//   "parameterOrder": [
	//     "accountId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "altId": {
	//       "description": "Filter Avails that match a case-insensitive, partner-specific custom id. NOTE: this field is deprecated and will be removed on V2; `alt_ids` should be used instead.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "altIds": {
	//       "description": "Filter Avails that match (case-insensitive) any of the given partner-specific custom ids.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pphNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "studioNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "territories": {
	//       "description": "Filter Avails that match (case-insensitive) any of the given country codes, using the \"ISO 3166-1 alpha-2\" format (examples: \"US\", \"us\", \"Us\").",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "title": {
	//       "description": "Filter that matches Avails with a `title_internal_alias`, `series_title_internal_alias`, `season_title_internal_alias`, or `episode_title_internal_alias` that contains the given case-insensitive title.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "videoIds": {
	//       "description": "Filter Avails that match any of the given `video_id`s.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/avails",
	//   "response": {
	//     "$ref": "ListAvailsResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAvailsListCall) Pages(ctx context.Context, f func(*ListAvailsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "playmoviespartner.accounts.components.list":

type AccountsComponentsListCall struct {
	s            *Service
	accountId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// List: List Components owned or managed by the partner. See
// _Authentication and Authorization rules_ and _List methods rules_ for
// more information about this method.
func (r *AccountsComponentsService) List(accountId string) *AccountsComponentsListCall {
	c := &AccountsComponentsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	return c
}

// AltCutId sets the optional parameter "altCutId": Filter Components
// that match a case-insensitive, partner-specific Alternative Cut ID.
func (c *AccountsComponentsListCall) AltCutId(altCutId string) *AccountsComponentsListCall {
	c.urlParams_.Set("altCutId", altCutId)
	return c
}

// CustomId sets the optional parameter "customId": Filter Components
// that match a case-insensitive partner-specific custom id.
func (c *AccountsComponentsListCall) CustomId(customId string) *AccountsComponentsListCall {
	c.urlParams_.Set("customId", customId)
	return c
}

// EditLevelEidr sets the optional parameter "editLevelEidr": Filter
// Components that match a given edit-level EIDR.
func (c *AccountsComponentsListCall) EditLevelEidr(editLevelEidr string) *AccountsComponentsListCall {
	c.urlParams_.Set("editLevelEidr", editLevelEidr)
	return c
}

// ElId sets the optional parameter "elId": Experience ID, as defined by
// Google.
func (c *AccountsComponentsListCall) ElId(elId string) *AccountsComponentsListCall {
	c.urlParams_.Set("elId", elId)
	return c
}

// Filename sets the optional parameter "filename": Filter Components
// that match a case-insensitive substring of the physical name of the
// delivered file.
func (c *AccountsComponentsListCall) Filename(filename string) *AccountsComponentsListCall {
	c.urlParams_.Set("filename", filename)
	return c
}

// InventoryId sets the optional parameter "inventoryId": InventoryID
// available in Common Manifest.
func (c *AccountsComponentsListCall) InventoryId(inventoryId string) *AccountsComponentsListCall {
	c.urlParams_.Set("inventoryId", inventoryId)
	return c
}

// PageSize sets the optional parameter "pageSize": See _List methods
// rules_ for info about this field.
func (c *AccountsComponentsListCall) PageSize(pageSize int64) *AccountsComponentsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": See _List methods
// rules_ for info about this field.
func (c *AccountsComponentsListCall) PageToken(pageToken string) *AccountsComponentsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// PlayableSequenceId sets the optional parameter "playableSequenceId":
// PlayableSequenceID available in Common Manifest.
func (c *AccountsComponentsListCall) PlayableSequenceId(playableSequenceId string) *AccountsComponentsListCall {
	c.urlParams_.Set("playableSequenceId", playableSequenceId)
	return c
}

// PphNames sets the optional parameter "pphNames": See _List methods
// rules_ for info about this field.
func (c *AccountsComponentsListCall) PphNames(pphNames ...string) *AccountsComponentsListCall {
	c.urlParams_.SetMulti("pphNames", append([]string{}, pphNames...))
	return c
}

// PresentationId sets the optional parameter "presentationId":
// PresentationID available in Common Manifest.
func (c *AccountsComponentsListCall) PresentationId(presentationId string) *AccountsComponentsListCall {
	c.urlParams_.Set("presentationId", presentationId)
	return c
}

// Status sets the optional parameter "status": Filter Components that
// match one of the given status.
//
// Possible values:
//   "STATUS_UNSPECIFIED"
//   "STATUS_APPROVED"
//   "STATUS_FAILED"
//   "STATUS_PROCESSING"
//   "STATUS_UNFULFILLED"
//   "STATUS_NOT_AVAILABLE"
func (c *AccountsComponentsListCall) Status(status ...string) *AccountsComponentsListCall {
	c.urlParams_.SetMulti("status", append([]string{}, status...))
	return c
}

// StudioNames sets the optional parameter "studioNames": See _List
// methods rules_ for info about this field.
func (c *AccountsComponentsListCall) StudioNames(studioNames ...string) *AccountsComponentsListCall {
	c.urlParams_.SetMulti("studioNames", append([]string{}, studioNames...))
	return c
}

// TitleLevelEidr sets the optional parameter "titleLevelEidr": Filter
// Components that match a given title-level EIDR.
func (c *AccountsComponentsListCall) TitleLevelEidr(titleLevelEidr string) *AccountsComponentsListCall {
	c.urlParams_.Set("titleLevelEidr", titleLevelEidr)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsComponentsListCall) Fields(s ...googleapi.Field) *AccountsComponentsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsComponentsListCall) IfNoneMatch(entityTag string) *AccountsComponentsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsComponentsListCall) Context(ctx context.Context) *AccountsComponentsListCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsComponentsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/components")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.components.list" call.
// Exactly one of *ListComponentsResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListComponentsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsComponentsListCall) Do(opts ...googleapi.CallOption) (*ListComponentsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ListComponentsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List Components owned or managed by the partner. See _Authentication and Authorization rules_ and _List methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.components.list",
	//   "parameterOrder": [
	//     "accountId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "altCutId": {
	//       "description": "Filter Components that match a case-insensitive, partner-specific Alternative Cut ID.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "customId": {
	//       "description": "Filter Components that match a case-insensitive partner-specific custom id.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "editLevelEidr": {
	//       "description": "Filter Components that match a given edit-level EIDR.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "elId": {
	//       "description": "Experience ID, as defined by Google.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "filename": {
	//       "description": "Filter Components that match a case-insensitive substring of the physical name of the delivered file.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "inventoryId": {
	//       "description": "InventoryID available in Common Manifest.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "playableSequenceId": {
	//       "description": "PlayableSequenceID available in Common Manifest.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pphNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "presentationId": {
	//       "description": "PresentationID available in Common Manifest.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "status": {
	//       "description": "Filter Components that match one of the given status.",
	//       "enum": [
	//         "STATUS_UNSPECIFIED",
	//         "STATUS_APPROVED",
	//         "STATUS_FAILED",
	//         "STATUS_PROCESSING",
	//         "STATUS_UNFULFILLED",
	//         "STATUS_NOT_AVAILABLE"
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "studioNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "titleLevelEidr": {
	//       "description": "Filter Components that match a given title-level EIDR.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/components",
	//   "response": {
	//     "$ref": "ListComponentsResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsComponentsListCall) Pages(ctx context.Context, f func(*ListComponentsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "playmoviespartner.accounts.components.type.get":

type AccountsComponentsTypeGetCall struct {
	s            *Service
	accountId    string
	componentId  string
	type_        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// Get: Get a Component given its id.
func (r *AccountsComponentsTypeService) Get(accountId string, componentId string, type_ string) *AccountsComponentsTypeGetCall {
	c := &AccountsComponentsTypeGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	c.componentId = componentId
	c.type_ = type_
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsComponentsTypeGetCall) Fields(s ...googleapi.Field) *AccountsComponentsTypeGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsComponentsTypeGetCall) IfNoneMatch(entityTag string) *AccountsComponentsTypeGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsComponentsTypeGetCall) Context(ctx context.Context) *AccountsComponentsTypeGetCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsComponentsTypeGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/components/{componentId}/type/{type}")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId":   c.accountId,
		"componentId": c.componentId,
		"type":        c.type_,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.components.type.get" call.
// Exactly one of *Component or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Component.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsComponentsTypeGetCall) Do(opts ...googleapi.CallOption) (*Component, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &Component{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Get a Component given its id.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.components.type.get",
	//   "parameterOrder": [
	//     "accountId",
	//     "componentId",
	//     "type"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "componentId": {
	//       "description": "REQUIRED. Component ID.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "type": {
	//       "description": "REQUIRED. Component Type.",
	//       "enum": [
	//         "COMPONENT_TYPE_UNSPECIFIED",
	//         "VIDEO",
	//         "AUDIO_20",
	//         "AUDIO_51",
	//         "SUBTITLE",
	//         "ARTWORK",
	//         "METADATA"
	//       ],
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/components/{componentId}/type/{type}",
	//   "response": {
	//     "$ref": "Component"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// method id "playmoviespartner.accounts.experienceLocales.get":

type AccountsExperienceLocalesGetCall struct {
	s            *Service
	accountId    string
	elId         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// Get: Get an ExperienceLocale given its id. See _Authentication and
// Authorization rules_ and _Get methods rules_ for more information
// about this method.
func (r *AccountsExperienceLocalesService) Get(accountId string, elId string) *AccountsExperienceLocalesGetCall {
	c := &AccountsExperienceLocalesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	c.elId = elId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsExperienceLocalesGetCall) Fields(s ...googleapi.Field) *AccountsExperienceLocalesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsExperienceLocalesGetCall) IfNoneMatch(entityTag string) *AccountsExperienceLocalesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsExperienceLocalesGetCall) Context(ctx context.Context) *AccountsExperienceLocalesGetCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsExperienceLocalesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/experienceLocales/{elId}")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
		"elId":      c.elId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.experienceLocales.get" call.
// Exactly one of *ExperienceLocale or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ExperienceLocale.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsExperienceLocalesGetCall) Do(opts ...googleapi.CallOption) (*ExperienceLocale, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ExperienceLocale{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Get an ExperienceLocale given its id. See _Authentication and Authorization rules_ and _Get methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.experienceLocales.get",
	//   "parameterOrder": [
	//     "accountId",
	//     "elId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "elId": {
	//       "description": "REQUIRED. ExperienceLocale ID, as defined by Google.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/experienceLocales/{elId}",
	//   "response": {
	//     "$ref": "ExperienceLocale"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// method id "playmoviespartner.accounts.experienceLocales.list":

type AccountsExperienceLocalesListCall struct {
	s            *Service
	accountId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// List: List ExperienceLocales owned or managed by the partner. See
// _Authentication and Authorization rules_ and _List methods rules_ for
// more information about this method.
func (r *AccountsExperienceLocalesService) List(accountId string) *AccountsExperienceLocalesListCall {
	c := &AccountsExperienceLocalesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	return c
}

// AltCutId sets the optional parameter "altCutId": Filter
// ExperienceLocales that match a case-insensitive, partner-specific
// Alternative Cut ID.
func (c *AccountsExperienceLocalesListCall) AltCutId(altCutId string) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("altCutId", altCutId)
	return c
}

// CustomId sets the optional parameter "customId": Filter
// ExperienceLocales that match a case-insensitive, partner-specific
// custom id.
func (c *AccountsExperienceLocalesListCall) CustomId(customId string) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("customId", customId)
	return c
}

// EditLevelEidr sets the optional parameter "editLevelEidr": Filter
// ExperienceLocales that match a given edit-level EIDR.
func (c *AccountsExperienceLocalesListCall) EditLevelEidr(editLevelEidr string) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("editLevelEidr", editLevelEidr)
	return c
}

// PageSize sets the optional parameter "pageSize": See _List methods
// rules_ for info about this field.
func (c *AccountsExperienceLocalesListCall) PageSize(pageSize int64) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": See _List methods
// rules_ for info about this field.
func (c *AccountsExperienceLocalesListCall) PageToken(pageToken string) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// PphNames sets the optional parameter "pphNames": See _List methods
// rules_ for info about this field.
func (c *AccountsExperienceLocalesListCall) PphNames(pphNames ...string) *AccountsExperienceLocalesListCall {
	c.urlParams_.SetMulti("pphNames", append([]string{}, pphNames...))
	return c
}

// Status sets the optional parameter "status": Filter ExperienceLocales
// that match one of the given status.
//
// Possible values:
//   "STATUS_UNSPECIFIED"
//   "STATUS_APPROVED"
//   "STATUS_FAILED"
//   "STATUS_PROCESSING"
//   "STATUS_UNFULFILLED"
//   "STATUS_NOT_AVAILABLE"
func (c *AccountsExperienceLocalesListCall) Status(status ...string) *AccountsExperienceLocalesListCall {
	c.urlParams_.SetMulti("status", append([]string{}, status...))
	return c
}

// StudioNames sets the optional parameter "studioNames": See _List
// methods rules_ for info about this field.
func (c *AccountsExperienceLocalesListCall) StudioNames(studioNames ...string) *AccountsExperienceLocalesListCall {
	c.urlParams_.SetMulti("studioNames", append([]string{}, studioNames...))
	return c
}

// TitleLevelEidr sets the optional parameter "titleLevelEidr": Filter
// ExperienceLocales that match a given title-level EIDR.
func (c *AccountsExperienceLocalesListCall) TitleLevelEidr(titleLevelEidr string) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("titleLevelEidr", titleLevelEidr)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsExperienceLocalesListCall) Fields(s ...googleapi.Field) *AccountsExperienceLocalesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsExperienceLocalesListCall) IfNoneMatch(entityTag string) *AccountsExperienceLocalesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsExperienceLocalesListCall) Context(ctx context.Context) *AccountsExperienceLocalesListCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsExperienceLocalesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/experienceLocales")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.experienceLocales.list" call.
// Exactly one of *ListExperienceLocalesResponse or error will be
// non-nil. Any non-2xx status code is an error. Response headers are in
// either *ListExperienceLocalesResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsExperienceLocalesListCall) Do(opts ...googleapi.CallOption) (*ListExperienceLocalesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ListExperienceLocalesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List ExperienceLocales owned or managed by the partner. See _Authentication and Authorization rules_ and _List methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.experienceLocales.list",
	//   "parameterOrder": [
	//     "accountId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "altCutId": {
	//       "description": "Filter ExperienceLocales that match a case-insensitive, partner-specific Alternative Cut ID.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "customId": {
	//       "description": "Filter ExperienceLocales that match a case-insensitive, partner-specific custom id.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "editLevelEidr": {
	//       "description": "Filter ExperienceLocales that match a given edit-level EIDR.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pphNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "status": {
	//       "description": "Filter ExperienceLocales that match one of the given status.",
	//       "enum": [
	//         "STATUS_UNSPECIFIED",
	//         "STATUS_APPROVED",
	//         "STATUS_FAILED",
	//         "STATUS_PROCESSING",
	//         "STATUS_UNFULFILLED",
	//         "STATUS_NOT_AVAILABLE"
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "studioNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "titleLevelEidr": {
	//       "description": "Filter ExperienceLocales that match a given title-level EIDR.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/experienceLocales",
	//   "response": {
	//     "$ref": "ListExperienceLocalesResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsExperienceLocalesListCall) Pages(ctx context.Context, f func(*ListExperienceLocalesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "playmoviespartner.accounts.orders.get":

type AccountsOrdersGetCall struct {
	s            *Service
	accountId    string
	orderId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// Get: Get an Order given its id. See _Authentication and Authorization
// rules_ and _Get methods rules_ for more information about this
// method.
func (r *AccountsOrdersService) Get(accountId string, orderId string) *AccountsOrdersGetCall {
	c := &AccountsOrdersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	c.orderId = orderId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsOrdersGetCall) Fields(s ...googleapi.Field) *AccountsOrdersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsOrdersGetCall) IfNoneMatch(entityTag string) *AccountsOrdersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsOrdersGetCall) Context(ctx context.Context) *AccountsOrdersGetCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsOrdersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/orders/{orderId}")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
		"orderId":   c.orderId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.orders.get" call.
// Exactly one of *Order or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Order.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *AccountsOrdersGetCall) Do(opts ...googleapi.CallOption) (*Order, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &Order{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Get an Order given its id. See _Authentication and Authorization rules_ and _Get methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.orders.get",
	//   "parameterOrder": [
	//     "accountId",
	//     "orderId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "orderId": {
	//       "description": "REQUIRED. Order ID.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/orders/{orderId}",
	//   "response": {
	//     "$ref": "Order"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// method id "playmoviespartner.accounts.orders.list":

type AccountsOrdersListCall struct {
	s            *Service
	accountId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// List: List Orders owned or managed by the partner. See
// _Authentication and Authorization rules_ and _List methods rules_ for
// more information about this method.
func (r *AccountsOrdersService) List(accountId string) *AccountsOrdersListCall {
	c := &AccountsOrdersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	return c
}

// CustomId sets the optional parameter "customId": Filter Orders that
// match a case-insensitive, partner-specific custom id.
func (c *AccountsOrdersListCall) CustomId(customId string) *AccountsOrdersListCall {
	c.urlParams_.Set("customId", customId)
	return c
}

// Name sets the optional parameter "name": Filter that matches Orders
// with a `name`, `show`, `season` or `episode` that contains the given
// case-insensitive name.
func (c *AccountsOrdersListCall) Name(name string) *AccountsOrdersListCall {
	c.urlParams_.Set("name", name)
	return c
}

// PageSize sets the optional parameter "pageSize": See _List methods
// rules_ for info about this field.
func (c *AccountsOrdersListCall) PageSize(pageSize int64) *AccountsOrdersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": See _List methods
// rules_ for info about this field.
func (c *AccountsOrdersListCall) PageToken(pageToken string) *AccountsOrdersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// PphNames sets the optional parameter "pphNames": See _List methods
// rules_ for info about this field.
func (c *AccountsOrdersListCall) PphNames(pphNames ...string) *AccountsOrdersListCall {
	c.urlParams_.SetMulti("pphNames", append([]string{}, pphNames...))
	return c
}

// Status sets the optional parameter "status": Filter Orders that match
// one of the given status.
//
// Possible values:
//   "STATUS_UNSPECIFIED"
//   "STATUS_APPROVED"
//   "STATUS_FAILED"
//   "STATUS_PROCESSING"
//   "STATUS_UNFULFILLED"
//   "STATUS_NOT_AVAILABLE"
func (c *AccountsOrdersListCall) Status(status ...string) *AccountsOrdersListCall {
	c.urlParams_.SetMulti("status", append([]string{}, status...))
	return c
}

// StudioNames sets the optional parameter "studioNames": See _List
// methods rules_ for info about this field.
func (c *AccountsOrdersListCall) StudioNames(studioNames ...string) *AccountsOrdersListCall {
	c.urlParams_.SetMulti("studioNames", append([]string{}, studioNames...))
	return c
}

// VideoIds sets the optional parameter "videoIds": Filter Orders that
// match any of the given `video_id`s.
func (c *AccountsOrdersListCall) VideoIds(videoIds ...string) *AccountsOrdersListCall {
	c.urlParams_.SetMulti("videoIds", append([]string{}, videoIds...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsOrdersListCall) Fields(s ...googleapi.Field) *AccountsOrdersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsOrdersListCall) IfNoneMatch(entityTag string) *AccountsOrdersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsOrdersListCall) Context(ctx context.Context) *AccountsOrdersListCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsOrdersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/orders")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.orders.list" call.
// Exactly one of *ListOrdersResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListOrdersResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsOrdersListCall) Do(opts ...googleapi.CallOption) (*ListOrdersResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ListOrdersResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List Orders owned or managed by the partner. See _Authentication and Authorization rules_ and _List methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.orders.list",
	//   "parameterOrder": [
	//     "accountId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "customId": {
	//       "description": "Filter Orders that match a case-insensitive, partner-specific custom id.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "name": {
	//       "description": "Filter that matches Orders with a `name`, `show`, `season` or `episode` that contains the given case-insensitive name.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pphNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "status": {
	//       "description": "Filter Orders that match one of the given status.",
	//       "enum": [
	//         "STATUS_UNSPECIFIED",
	//         "STATUS_APPROVED",
	//         "STATUS_FAILED",
	//         "STATUS_PROCESSING",
	//         "STATUS_UNFULFILLED",
	//         "STATUS_NOT_AVAILABLE"
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "studioNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "videoIds": {
	//       "description": "Filter Orders that match any of the given `video_id`s.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/orders",
	//   "response": {
	//     "$ref": "ListOrdersResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsOrdersListCall) Pages(ctx context.Context, f func(*ListOrdersResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "playmoviespartner.accounts.storeInfos.list":

type AccountsStoreInfosListCall struct {
	s            *Service
	accountId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// List: List StoreInfos owned or managed by the partner. See
// _Authentication and Authorization rules_ and _List methods rules_ for
// more information about this method.
func (r *AccountsStoreInfosService) List(accountId string) *AccountsStoreInfosListCall {
	c := &AccountsStoreInfosListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	return c
}

// Countries sets the optional parameter "countries": Filter StoreInfos
// that match (case-insensitive) any of the given country codes, using
// the "ISO 3166-1 alpha-2" format (examples: "US", "us", "Us").
func (c *AccountsStoreInfosListCall) Countries(countries ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("countries", append([]string{}, countries...))
	return c
}

// Mids sets the optional parameter "mids": Filter StoreInfos that match
// any of the given `mid`s.
func (c *AccountsStoreInfosListCall) Mids(mids ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("mids", append([]string{}, mids...))
	return c
}

// Name sets the optional parameter "name": Filter that matches
// StoreInfos with a `name` or `show_name` that contains the given
// case-insensitive name.
func (c *AccountsStoreInfosListCall) Name(name string) *AccountsStoreInfosListCall {
	c.urlParams_.Set("name", name)
	return c
}

// PageSize sets the optional parameter "pageSize": See _List methods
// rules_ for info about this field.
func (c *AccountsStoreInfosListCall) PageSize(pageSize int64) *AccountsStoreInfosListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": See _List methods
// rules_ for info about this field.
func (c *AccountsStoreInfosListCall) PageToken(pageToken string) *AccountsStoreInfosListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// PphNames sets the optional parameter "pphNames": See _List methods
// rules_ for info about this field.
func (c *AccountsStoreInfosListCall) PphNames(pphNames ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("pphNames", append([]string{}, pphNames...))
	return c
}

// SeasonIds sets the optional parameter "seasonIds": Filter StoreInfos
// that match any of the given `season_id`s.
func (c *AccountsStoreInfosListCall) SeasonIds(seasonIds ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("seasonIds", append([]string{}, seasonIds...))
	return c
}

// StudioNames sets the optional parameter "studioNames": See _List
// methods rules_ for info about this field.
func (c *AccountsStoreInfosListCall) StudioNames(studioNames ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("studioNames", append([]string{}, studioNames...))
	return c
}

// VideoId sets the optional parameter "videoId": Filter StoreInfos that
// match a given `video_id`. NOTE: this field is deprecated and will be
// removed on V2; `video_ids` should be used instead.
func (c *AccountsStoreInfosListCall) VideoId(videoId string) *AccountsStoreInfosListCall {
	c.urlParams_.Set("videoId", videoId)
	return c
}

// VideoIds sets the optional parameter "videoIds": Filter StoreInfos
// that match any of the given `video_id`s.
func (c *AccountsStoreInfosListCall) VideoIds(videoIds ...string) *AccountsStoreInfosListCall {
	c.urlParams_.SetMulti("videoIds", append([]string{}, videoIds...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsStoreInfosListCall) Fields(s ...googleapi.Field) *AccountsStoreInfosListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsStoreInfosListCall) IfNoneMatch(entityTag string) *AccountsStoreInfosListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsStoreInfosListCall) Context(ctx context.Context) *AccountsStoreInfosListCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsStoreInfosListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/storeInfos")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.storeInfos.list" call.
// Exactly one of *ListStoreInfosResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListStoreInfosResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *AccountsStoreInfosListCall) Do(opts ...googleapi.CallOption) (*ListStoreInfosResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &ListStoreInfosResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List StoreInfos owned or managed by the partner. See _Authentication and Authorization rules_ and _List methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.storeInfos.list",
	//   "parameterOrder": [
	//     "accountId"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "countries": {
	//       "description": "Filter StoreInfos that match (case-insensitive) any of the given country codes, using the \"ISO 3166-1 alpha-2\" format (examples: \"US\", \"us\", \"Us\").",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "mids": {
	//       "description": "Filter StoreInfos that match any of the given `mid`s.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "name": {
	//       "description": "Filter that matches StoreInfos with a `name` or `show_name` that contains the given case-insensitive name.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pphNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "seasonIds": {
	//       "description": "Filter StoreInfos that match any of the given `season_id`s.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "studioNames": {
	//       "description": "See _List methods rules_ for info about this field.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "videoId": {
	//       "description": "Filter StoreInfos that match a given `video_id`. NOTE: this field is deprecated and will be removed on V2; `video_ids` should be used instead.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "videoIds": {
	//       "description": "Filter StoreInfos that match any of the given `video_id`s.",
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/storeInfos",
	//   "response": {
	//     "$ref": "ListStoreInfosResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsStoreInfosListCall) Pages(ctx context.Context, f func(*ListStoreInfosResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "playmoviespartner.accounts.storeInfos.country.get":

type AccountsStoreInfosCountryGetCall struct {
	s            *Service
	accountId    string
	videoId      string
	country      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
}

// Get: Get a StoreInfo given its video id and country. See
// _Authentication and Authorization rules_ and _Get methods rules_ for
// more information about this method.
func (r *AccountsStoreInfosCountryService) Get(accountId string, videoId string, country string) *AccountsStoreInfosCountryGetCall {
	c := &AccountsStoreInfosCountryGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accountId = accountId
	c.videoId = videoId
	c.country = country
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *AccountsStoreInfosCountryGetCall) Fields(s ...googleapi.Field) *AccountsStoreInfosCountryGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *AccountsStoreInfosCountryGetCall) IfNoneMatch(entityTag string) *AccountsStoreInfosCountryGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *AccountsStoreInfosCountryGetCall) Context(ctx context.Context) *AccountsStoreInfosCountryGetCall {
	c.ctx_ = ctx
	return c
}

func (c *AccountsStoreInfosCountryGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accounts/{accountId}/storeInfos/{videoId}/country/{country}")
	urls += "?" + c.urlParams_.Encode()
	req, _ := http.NewRequest("GET", urls, body)
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"accountId": c.accountId,
		"videoId":   c.videoId,
		"country":   c.country,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "playmoviespartner.accounts.storeInfos.country.get" call.
// Exactly one of *StoreInfo or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *StoreInfo.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsStoreInfosCountryGetCall) Do(opts ...googleapi.CallOption) (*StoreInfo, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, &googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		}
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, err
	}
	ret := &StoreInfo{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := json.NewDecoder(res.Body).Decode(target); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Get a StoreInfo given its video id and country. See _Authentication and Authorization rules_ and _Get methods rules_ for more information about this method.",
	//   "httpMethod": "GET",
	//   "id": "playmoviespartner.accounts.storeInfos.country.get",
	//   "parameterOrder": [
	//     "accountId",
	//     "videoId",
	//     "country"
	//   ],
	//   "parameters": {
	//     "accountId": {
	//       "description": "REQUIRED. See _General rules_ for more information about this field.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "country": {
	//       "description": "REQUIRED. Edit country.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "videoId": {
	//       "description": "REQUIRED. Video ID.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/accounts/{accountId}/storeInfos/{videoId}/country/{country}",
	//   "response": {
	//     "$ref": "StoreInfo"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/playmovies_partner.readonly"
	//   ]
	// }

}
