package main

import (
	"fmt"
	"path"
	"strings"
)

// This file defines the specific type of some variables.
//
// There are two types of lists:
// * lkShell is a list whose elements are split by shell rules
// * lkSpace is a list whose elements are split by whitespace
//
// See vartypecheck.go for how these types are checked.
//
// Last synced with mk/defaults/mk.conf revision 1.118

func (gd *GlobalData) InitVartypes() {
	usr("ALLOW_VULNERABLE_PACKAGES", lkNone, CheckvarYes)
	usr("MANINSTALL", lkShell, enum("maninstall catinstall"))
	usr("MANZ", lkNone, CheckvarYes)
	usr("GZIP", lkShell, CheckvarShellWord)
	usr("MKCRYPTO", lkNone, CheckvarYesNo)
	usr("OBJHOSTNAME", lkNone, CheckvarYes)
	usr("OBJMACHINE", lkNone, CheckvarYes)
	usr("PKG_SUFX", lkNone, CheckvarFilename)
	usr("PKGSRC_LOCKTYPE", lkNone, enum("none sleep once"))
	usr("PKGSRC_SLEEPSECS", lkNone, CheckvarInteger)
	usr("USETBL", lkNone, CheckvarYes)
	usr("ABI", lkNone, enum("32 64"))
	usr("PKG_DEVELOPER", lkNone, CheckvarYesNo)
	usr("USE_ABI_DEPENDS", lkNone, CheckvarYesNo)
	usr("PKG_REGISTER_SHELLS", lkNone, enum("YES NO"))
	usr("PKGSRC_COMPILER", lkShell, enum("ccache ccc clang distcc f2c gcc hp icc ido gcc mipspro mipspro-ucode pcc sunpro xlc"))
	usr("PKGSRC_MESSAGE_RECIPIENTS", lkShell, CheckvarMailAddress)
	usr("PKGSRC_SHOW_BUILD_DEFS", lkNone, CheckvarYesNo)
	usr("PKGSRC_SHOW_PATCH_ERRORMSG", lkNone, CheckvarYesNo)
	usr("PKGSRC_RUN_TEST", lkNone, CheckvarYesNo)
	usr("PREFER_PKGSRC", lkShell, CheckvarIdentifier)
	usr("PREFER_NATIVE", lkShell, CheckvarIdentifier)
	usr("PREFER_NATIVE_PTHREADS", lkNone, CheckvarYesNo)
	usr("LOCALBASE", lkNone, CheckvarPathname)
	usr("CROSSBASE", lkNone, CheckvarPathname)
	usr("VARBASE", lkNone, CheckvarPathname)
	usr("X11_TYPE", lkNone, enum("modular native"))
	usr("X11BASE", lkNone, CheckvarPathname)
	usr("MOTIFBASE", lkNone, CheckvarPathname)
	usr("PKGINFODIR", lkNone, CheckvarPathname)
	usr("PKGMANDIR", lkNone, CheckvarPathname)
	usr("USE_XPKGWEDGE", lkNone, CheckvarYesNo)
	usr("BSDSRCDIR", lkNone, CheckvarPathname)
	usr("BSDXSRCDIR", lkNone, CheckvarPathname)
	usr("DISTDIR", lkNone, CheckvarPathname)
	usr("DIST_PATH", lkNone, CheckvarPathlist)
	usr("DEFAULT_VIEW", lkNone, CheckvarUnchecked)
	usr("FETCH_CMD", lkNone, CheckvarShellCommand)
	usr("FETCH_USING", lkNone, enum("curl custom fetch ftp manual wget"))
	usr("FETCH_RESUME_ARGS", lkShell, CheckvarShellWord)
	usr("FETCH_OUTPUT_ARGS", lkShell, CheckvarShellWord)
	usr("LIBTOOLIZE_PLIST", lkNone, CheckvarYesNo)
	usr("PKG_RESUME_TRANSFERS", lkNone, CheckvarYesNo)
	usr("PKG_SYSCONFBASE", lkNone, CheckvarPathname)
	usr("RCD_SCRIPTS_DIR", lkNone, CheckvarPathname)
	usr("PACKAGES", lkNone, CheckvarPathname)
	usr("PKGVULNDIR", lkNone, CheckvarPathname)
	usr("PASSIVE_FETCH", lkNone, CheckvarYes)
	usr("PATCH_FUZZ_FACTOR", lkNone, enum("-F0 -F1 -F2 -F3"))
	usr("ACCEPTABLE_LICENSES", lkShell, CheckvarIdentifier)
	usr("SPECIFIC_PKGS", lkNone, CheckvarYes)
	usr("SITE_SPECIFIC_PKGS", lkShell, CheckvarPkgPath)
	usr("HOST_SPECIFIC_PKGS", lkShell, CheckvarPkgPath)
	usr("GROUP_SPECIFIC_PKGS", lkShell, CheckvarPkgPath)
	usr("USER_SPECIFIC_PKGS", lkShell, CheckvarPkgPath)
	usr("FAILOVER_FETCH", lkNone, CheckvarYes)
	usr("MASTER_SORT", lkShell, CheckvarUnchecked)
	usr("MASTER_SORT_REGEX", lkShell, CheckvarUnchecked)
	usr("PATCH_DEBUG", lkNone, CheckvarYes)
	usr("PKG_FC", lkNone, CheckvarShellCommand)
	usr("IMAKE", lkNone, CheckvarShellCommand)
	usr("IMAKEOPTS", lkShell, CheckvarShellWord)
	usr("PRE_ROOT_CMD", lkNone, CheckvarShellCommand)
	pkg("USE_GAMESGROUP", lkNone, CheckvarYesNo)
	usr("SU_CMD", lkNone, CheckvarShellCommand)
	usr("SU_CMD_PATH_APPEND", lkNone, CheckvarPathlist)
	usr("FATAL_OBJECT_FMT_SKEW", lkNone, CheckvarYesNo)
	usr("WARN_NO_OBJECT_FMT", lkNone, CheckvarYesNo)
	usr("SMART_MESSAGES", lkNone, CheckvarYes)
	usr("BINPKG_SITES", lkShell, CheckvarURL)
	usr("BIN_INSTALL_FLAG", lkShell, CheckvarShellWord)
	usr("LOCALPATCHES", lkNone, CheckvarPathname)

	// some other variables, sorted alphabetically

	acl(".CURDIR", lkNone, CheckvarPathname, "buildlink3.mk:; *: use, use-loadtime")
	acl(".TARGET", lkNone, CheckvarPathname, "buildlink3.mk:; *: use, use-loadtime")
	acl("ALL_ENV", lkShell, CheckvarShellWord, "")
	acl("ALTERNATIVES_FILE", lkNone, CheckvarFilename, "")
	acl("ALTERNATIVES_SRC", lkShell, CheckvarPathname, "")
	pkg("APACHE_MODULE", lkNone, CheckvarYes)
	sys("AR", lkNone, CheckvarShellCommand)
	sys("AS", lkNone, CheckvarShellCommand)
	pkglist("AUTOCONF_REQD", lkShell, CheckvarVersion)
	acl("AUTOMAKE_OVERRIDE", lkShell, CheckvarPathmask, "")
	pkglist("AUTOMAKE_REQD", lkShell, CheckvarVersion)
	pkg("AUTO_MKDIRS", lkNone, CheckvarYesNo)
	usr("BATCH", lkNone, CheckvarYes)
	acl("BDB185_DEFAULT", lkNone, CheckvarUnchecked, "")
	sys("BDBBASE", lkNone, CheckvarPathname)
	pkg("BDB_ACCEPTED", lkShell, enum("db1 db2 db3 db4 db5"))
	acl("BDB_DEFAULT", lkNone, enum("db1 db2 db3 db4 db5"), "")
	sys("BDB_LIBS", lkShell, CheckvarLdFlag)
	sys("BDB_TYPE", lkNone, enum("db1 db2 db3 db4 db5"))
	sys("BINGRP", lkNone, CheckvarUserGroupName)
	sys("BINMODE", lkNone, CheckvarFileMode)
	sys("BINOWN", lkNone, CheckvarUserGroupName)
	acl("BOOTSTRAP_DEPENDS", lkSpace, CheckvarDependencyWithPath, "Makefile, Makefile.common, *.mk: append")
	pkg("BOOTSTRAP_PKG", lkNone, CheckvarYesNo)
	acl("BROKEN", lkNone, CheckvarMessage, "")
	pkg("BROKEN_GETTEXT_DETECTION", lkNone, CheckvarYesNo)
	pkglist("BROKEN_EXCEPT_ON_PLATFORM", lkShell, CheckvarMachinePlatformPattern)
	pkglist("BROKEN_ON_PLATFORM", lkSpace, CheckvarMachinePlatformPattern)
	sys("BSD_MAKE_ENV", lkShell, CheckvarShellWord)
	acl("BUILDLINK_ABI_DEPENDS.*", lkSpace, CheckvarDependency, "builtin.mk: append, use-loadtime; *: append")
	acl("BUILDLINK_API_DEPENDS.*", lkSpace, CheckvarDependency, "builtin.mk: append, use-loadtime; *: append")
	acl("BUILDLINK_CONTENTS_FILTER", lkNone, CheckvarShellCommand, "")
	sys("BUILDLINK_CFLAGS", lkShell, CheckvarCFlag)
	bl3list("BUILDLINK_CFLAGS.*", lkShell, CheckvarCFlag)
	sys("BUILDLINK_CPPFLAGS", lkShell, CheckvarCFlag)
	bl3list("BUILDLINK_CPPFLAGS.*", lkShell, CheckvarCFlag)
	acl("BUILDLINK_CONTENTS_FILTER.*", lkNone, CheckvarShellCommand, "buildlink3.mk: set")
	acl("BUILDLINK_DEPENDS", lkSpace, CheckvarIdentifier, "buildlink3.mk: append")
	acl("BUILDLINK_DEPMETHOD.*", lkShell, CheckvarBuildlinkDepmethod, "buildlink3.mk: default, append, use; Makefile: set, append; Makefile.common, *.mk: append")
	acl("BUILDLINK_DIR", lkNone, CheckvarPathname, "*: use")
	bl3list("BUILDLINK_FILES.*", lkShell, CheckvarPathmask)
	acl("BUILDLINK_FILES_CMD.*", lkNone, CheckvarShellCommand, "")
	acl("BUILDLINK_INCDIRS.*", lkShell, CheckvarPathname, "buildlink3.mk: default, append; Makefile, Makefile.common, *.mk: use")
	acl("BUILDLINK_JAVA_PREFIX.*", lkNone, CheckvarPathname, "buildlink3.mk: set, use")
	acl("BUILDLINK_LDADD.*", lkShell, CheckvarLdFlag, "builtin.mk: set, default, append, use; buildlink3.mk: append, use; Makefile, Makefile.common, *.mk: use")
	acl("BUILDLINK_LDFLAGS", lkShell, CheckvarLdFlag, "*: use")
	bl3list("BUILDLINK_LDFLAGS.*", lkShell, CheckvarLdFlag)
	acl("BUILDLINK_LIBDIRS.*", lkShell, CheckvarPathname, "buildlink3.mk, builtin.mk: append; Makefile, Makefile.common, *.mk: use")
	acl("BUILDLINK_LIBS.*", lkShell, CheckvarLdFlag, "buildlink3.mk: append")
	acl("BUILDLINK_PASSTHRU_DIRS", lkShell, CheckvarPathname, "Makefile, Makefile.common, buildlink3.mk, hacks.mk: append")
	acl("BUILDLINK_PASSTHRU_RPATHDIRS", lkShell, CheckvarPathname, "Makefile, Makefile.common, buildlink3.mk, hacks.mk: append")
	acl("BUILDLINK_PKGSRCDIR.*", lkNone, CheckvarRelativePkgDir, "buildlink3.mk: default, use-loadtime")
	acl("BUILDLINK_PREFIX.*", lkNone, CheckvarPathname, "builtin.mk: set, use; buildlink3.mk: use; Makefile, Makefile.common, *.mk: use")
	acl("BUILDLINK_RPATHDIRS.*", lkShell, CheckvarPathname, "buildlink3.mk: append")
	acl("BUILDLINK_TARGETS", lkShell, CheckvarIdentifier, "")
	acl("BUILDLINK_FNAME_TRANSFORM.*", lkNone, CheckvarSedCommands, "Makefile, buildlink3.mk, builtin.mk, hacks.mk: append")
	acl("BUILDLINK_TRANSFORM", lkShell, CheckvarWrapperTransform, "*: append")
	acl("BUILDLINK_TREE", lkShell, CheckvarIdentifier, "buildlink3.mk: append")
	acl("BUILD_DEFS", lkShell, CheckvarVarname, "Makefile, Makefile.common, options.mk: append")
	acl("BUILD_DEPENDS", lkSpace, CheckvarDependencyWithPath, "Makefile, Makefile.common, *.mk: append")
	pkglist("BUILD_DIRS", lkShell, CheckvarWrksrcSubdirectory)
	pkglist("BUILD_ENV", lkShell, CheckvarShellWord)
	sys("BUILD_MAKE_CMD", lkNone, CheckvarShellCommand)
	pkglist("BUILD_MAKE_FLAGS", lkShell, CheckvarShellWord)
	pkglist("BUILD_TARGET", lkShell, CheckvarIdentifier)
	pkg("BUILD_USES_MSGFMT", lkNone, CheckvarYes)
	acl("BUILTIN_PKG", lkNone, CheckvarIdentifier, "builtin.mk: set, use-loadtime, use")
	acl("BUILTIN_PKG.*", lkNone, CheckvarPkgName, "builtin.mk: set, use-loadtime, use")
	acl("BUILTIN_FIND_FILES_VAR", lkShell, CheckvarVarname, "builtin.mk: set")
	acl("BUILTIN_FIND_FILES.*", lkShell, CheckvarPathname, "builtin.mk: set")
	acl("BUILTIN_FIND_GREP.*", lkNone, CheckvarString, "builtin.mk: set")
	acl("BUILTIN_FIND_LIBS", lkShell, CheckvarPathname, "builtin.mk: set")
	acl("BUILTIN_IMAKE_CHECK", lkShell, CheckvarUnchecked, "builtin.mk: set")
	acl("BUILTIN_IMAKE_CHECK.*", lkNone, CheckvarYesNo, "")
	sys("BUILTIN_X11_TYPE", lkNone, CheckvarUnchecked)
	sys("BUILTIN_X11_VERSION", lkNone, CheckvarUnchecked)
	acl("CATEGORIES", lkShell, CheckvarCategory, "Makefile: set, append; Makefile.common: set, default, append")
	sys("CC_VERSION", lkNone, CheckvarMessage)
	sys("CC", lkNone, CheckvarShellCommand)
	pkglist("CFLAGS*", lkShell, CheckvarCFlag) // may also be changed by the user
	acl("CHECK_BUILTIN", lkNone, CheckvarYesNo, "builtin.mk: default; Makefile: set")
	acl("CHECK_BUILTIN.*", lkNone, CheckvarYesNo, "buildlink3.mk: set; builtin.mk: default; *: use-loadtime")
	acl("CHECK_FILES_SKIP", lkShell, CheckvarBasicRegularExpression, "Makefile, Makefile.common: append")
	pkg("CHECK_FILES_SUPPORTED", lkNone, CheckvarYesNo)
	usr("CHECK_HEADERS", lkNone, CheckvarYesNo)
	pkglist("CHECK_HEADERS_SKIP", lkShell, CheckvarPathmask)
	usr("CHECK_INTERPRETER", lkNone, CheckvarYesNo)
	pkglist("CHECK_INTERPRETER_SKIP", lkShell, CheckvarPathmask)
	usr("CHECK_PERMS", lkNone, CheckvarYesNo)
	pkglist("CHECK_PERMS_SKIP", lkShell, CheckvarPathmask)
	usr("CHECK_PORTABILITY", lkNone, CheckvarYesNo)
	pkglist("CHECK_PORTABILITY_SKIP", lkShell, CheckvarPathmask)
	acl("CHECK_SHLIBS", lkNone, CheckvarYesNo, "Makefile: set")
	pkglist("CHECK_SHLIBS_SKIP", lkShell, CheckvarPathmask)
	acl("CHECK_SHLIBS_SUPPORTED", lkNone, CheckvarYesNo, "Makefile: set")
	pkglist("CHECK_WRKREF_SKIP", lkShell, CheckvarPathmask)
	pkg("CMAKE_ARG_PATH", lkNone, CheckvarPathname)
	pkglist("CMAKE_ARGS", lkShell, CheckvarShellWord)
	acl("COMMENT", lkNone, CheckvarComment, "Makefile, Makefile.common: set, append")
	acl("COMPILER_RPATH_FLAG", lkNone, enum("-Wl,-rpath"), "*: use")
	pkglist("CONFIGURE_ARGS", lkShell, CheckvarShellWord)
	pkglist("CONFIGURE_DIRS", lkShell, CheckvarWrksrcSubdirectory)
	acl("CONFIGURE_ENV", lkShell, CheckvarShellWord, "Makefile, Makefile.common: append, set, use; buildlink3.mk, builtin.mk: append; *.mk: append, use")
	pkg("CONFIGURE_HAS_INFODIR", lkNone, CheckvarYesNo)
	pkg("CONFIGURE_HAS_LIBDIR", lkNone, CheckvarYesNo)
	pkg("CONFIGURE_HAS_MANDIR", lkNone, CheckvarYesNo)
	pkg("CONFIGURE_SCRIPT", lkNone, CheckvarPathname)
	acl("CONFIG_GUESS_OVERRIDE", lkShell, CheckvarPathmask, "Makefile, Makefile.common: set, append")
	acl("CONFIG_STATUS_OVERRIDE", lkShell, CheckvarPathmask, "Makefile, Makefile.common: set, append")
	acl("CONFIG_SHELL", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	acl("CONFIG_SUB_OVERRIDE", lkShell, CheckvarPathmask, "Makefile, Makefile.common: set, append")
	pkglist("CONFLICTS", lkSpace, CheckvarDependency)
	pkglist("CONF_FILES", lkShell, CheckvarShellWord)
	pkg("CONF_FILES_MODE", lkNone, enum("0644 0640 0600 0400"))
	pkglist("CONF_FILES_PERMS", lkShell, CheckvarPerms)
	sys("COPY", lkNone, enum("-c")) // The flag that tells ${INSTALL} to copy a file
	sys("CPP", lkNone, CheckvarShellCommand)
	pkglist("CPPFLAGS*", lkShell, CheckvarCFlag)
	acl("CRYPTO", lkNone, CheckvarYes, "Makefile: set")
	sys("CXX", lkNone, CheckvarShellCommand)
	pkglist("CXXFLAGS*", lkShell, CheckvarCFlag)
	acl("DEINSTALL_FILE", lkNone, CheckvarPathname, "Makefile: set")
	acl("DEINSTALL_SRC", lkShell, CheckvarPathname, "Makefile: set; Makefile.common: default, set")
	acl("DEINSTALL_TEMPLATES", lkShell, CheckvarPathname, "Makefile: set, append; Makefile.common: set, default, append")
	sys("DELAYED_ERROR_MSG", lkNone, CheckvarShellCommand)
	sys("DELAYED_WARNING_MSG", lkNone, CheckvarShellCommand)
	pkglist("DEPENDS", lkSpace, CheckvarDependencyWithPath)
	usr("DEPENDS_TARGET", lkShell, CheckvarIdentifier)
	acl("DESCR_SRC", lkShell, CheckvarPathname, "Makefile: set; Makefile.common: default, set")
	sys("DESTDIR", lkNone, CheckvarPathname)
	acl("DESTDIR_VARNAME", lkNone, CheckvarVarname, "Makefile, Makefile.common: set")
	sys("DEVOSSAUDIO", lkNone, CheckvarPathname)
	sys("DEVOSSSOUND", lkNone, CheckvarPathname)
	pkglist("DISTFILES", lkShell, CheckvarFilename)
	pkg("DISTINFO_FILE", lkNone, CheckvarRelativePkgPath)
	pkg("DISTNAME", lkNone, CheckvarFilename)
	pkg("DIST_SUBDIR", lkNone, CheckvarPathname)
	acl("DJB_BUILD_ARGS", lkShell, CheckvarShellWord, "")
	acl("DJB_BUILD_TARGETS", lkShell, CheckvarIdentifier, "")
	acl("DJB_CONFIG_CMDS", lkNone, CheckvarShellCommands, "options.mk: set")
	acl("DJB_CONFIG_DIRS", lkShell, CheckvarWrksrcSubdirectory, "")
	acl("DJB_CONFIG_HOME", lkNone, CheckvarFilename, "")
	acl("DJB_CONFIG_PREFIX", lkNone, CheckvarPathname, "")
	acl("DJB_INSTALL_TARGETS", lkShell, CheckvarIdentifier, "")
	acl("DJB_MAKE_TARGETS", lkNone, CheckvarYesNo, "")
	acl("DJB_RESTRICTED", lkNone, CheckvarYesNo, "Makefile: set")
	acl("DJB_SLASHPACKAGE", lkNone, CheckvarYesNo, "")
	acl("DLOPEN_REQUIRE_PTHREADS", lkNone, CheckvarYesNo, "")
	acl("DL_AUTO_VARS", lkNone, CheckvarYes, "Makefile, Makefile.common, options.mk: set")
	acl("DL_LIBS", lkShell, CheckvarLdFlag, "")
	sys("DOCOWN", lkNone, CheckvarUserGroupName)
	sys("DOCGRP", lkNone, CheckvarUserGroupName)
	sys("DOCMODE", lkNone, CheckvarFileMode)
	sys("DOWNLOADED_DISTFILE", lkNone, CheckvarPathname)
	sys("DO_NADA", lkNone, CheckvarShellCommand)
	pkg("DYNAMIC_SITES_CMD", lkNone, CheckvarShellCommand)
	pkg("DYNAMIC_SITES_SCRIPT", lkNone, CheckvarPathname)
	acl("ECHO", lkNone, CheckvarShellCommand, "*: use")
	sys("ECHO_MSG", lkNone, CheckvarShellCommand)
	sys("ECHO_N", lkNone, CheckvarShellCommand)
	pkg("EGDIR", lkNone, CheckvarPathname) // Not defined anywhere, but used in many places like this.
	sys("EMACS_BIN", lkNone, CheckvarPathname)
	sys("EMACS_ETCPREFIX", lkNone, CheckvarPathname)
	sys("EMACS_FLAVOR", lkNone, enum("emacs xemacs"))
	sys("EMACS_INFOPREFIX", lkNone, CheckvarPathname)
	sys("EMACS_LISPPREFIX", lkNone, CheckvarPathname)
	acl("EMACS_MODULES", lkShell, CheckvarIdentifier, "Makefile, Makefile.common: set, append")
	sys("EMACS_PKGNAME_PREFIX", lkNone, CheckvarIdentifier) // Or the empty string.
	sys("EMACS_TYPE", lkNone, enum("emacs xemacs"))
	acl("EMACS_USE_LEIM", lkNone, CheckvarYes, "")
	acl("EMACS_VERSIONS_ACCEPTED", lkShell, enum("emacs25 emacs24 emacs24nox emacs23 emacs23nox emacs22 emacs22nox emacs21 emacs21nox emacs20 xemacs215 xemacs215nox xemacs214 xemacs214nox"), "Makefile: set")
	sys("EMACS_VERSION_MAJOR", lkNone, CheckvarInteger)
	sys("EMACS_VERSION_MINOR", lkNone, CheckvarInteger)
	acl("EMACS_VERSION_REQD", lkShell, enum("emacs24 emacs24nox emacs23 emacs23nox emacs22 emacs22nox emacs21 emacs21nox emacs20 xemacs215 xemacs214"), "Makefile: set, append")
	sys("EMULDIR", lkNone, CheckvarPathname)
	sys("EMULSUBDIR", lkNone, CheckvarPathname)
	sys("OPSYS_EMULDIR", lkNone, CheckvarPathname)
	sys("EMULSUBDIRSLASH", lkNone, CheckvarPathname)
	sys("EMUL_ARCH", lkNone, enum("arm i386 m68k none ns32k sparc vax x86_64"))
	sys("EMUL_DISTRO", lkNone, CheckvarIdentifier)
	sys("EMUL_IS_NATIVE", lkNone, CheckvarYes)
	pkg("EMUL_MODULES.*", lkShell, CheckvarIdentifier)
	sys("EMUL_OPSYS", lkNone, enum("darwin freebsd hpux irix linux osf1 solaris sunos none"))
	pkg("EMUL_PKG_FMT", lkNone, enum("plain rpm"))
	usr("EMUL_PLATFORM", lkNone, CheckvarEmulPlatform)
	pkg("EMUL_PLATFORMS", lkShell, CheckvarEmulPlatform)
	usr("EMUL_PREFER", lkShell, CheckvarEmulPlatform)
	pkg("EMUL_REQD", lkSpace, CheckvarDependency)
	usr("EMUL_TYPE.*", lkNone, enum("native builtin suse suse-9.1 suse-9.x suse-10.0 suse-10.x"))
	sys("ERROR_CAT", lkNone, CheckvarShellCommand)
	sys("ERROR_MSG", lkNone, CheckvarShellCommand)
	acl("EVAL_PREFIX", lkSpace, CheckvarShellWord, "Makefile, Makefile.common: append") // XXX: Combining ShellWord with lkSpace looks weird.
	sys("EXPORT_SYMBOLS_LDFLAGS", lkShell, CheckvarLdFlag)
	sys("EXTRACT_CMD", lkNone, CheckvarShellCommand)
	pkg("EXTRACT_DIR", lkNone, CheckvarPathname)
	pkglist("EXTRACT_ELEMENTS", lkShell, CheckvarPathmask)
	pkglist("EXTRACT_ENV", lkShell, CheckvarShellWord)
	pkglist("EXTRACT_ONLY", lkShell, CheckvarPathname)
	acl("EXTRACT_OPTS", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_BIN", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_LHA", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_PAX", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_RAR", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_TAR", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_ZIP", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	acl("EXTRACT_OPTS_ZOO", lkShell, CheckvarShellWord, "Makefile, Makefile.common: set, append")
	pkg("EXTRACT_SUFX", lkNone, CheckvarDistSuffix)
	pkg("EXTRACT_USING", lkNone, enum("bsdtar gtar nbtar pax"))
	sys("FAIL_MSG", lkNone, CheckvarShellCommand)
	sys("FAMBASE", lkNone, CheckvarPathname)
	pkg("FAM_ACCEPTED", lkShell, enum("fam gamin"))
	usr("FAM_DEFAULT", lkNone, enum("fam gamin"))
	sys("FAM_TYPE", lkNone, enum("fam gamin"))
	acl("FETCH_BEFORE_ARGS", lkShell, CheckvarShellWord, "Makefile: set, append")
	pkglist("FETCH_MESSAGE", lkShell, CheckvarShellWord)
	pkg("FILESDIR", lkNone, CheckvarRelativePkgPath)
	pkglist("FILES_SUBST", lkShell, CheckvarShellWord)
	acl("FILES_SUBST_SED", lkShell, CheckvarShellWord, "")
	pkglist("FIX_RPATH", lkShell, CheckvarVarname)
	pkglist("FLEX_REQD", lkShell, CheckvarVersion)
	acl("FONTS_DIRS.*", lkShell, CheckvarPathname, "Makefile: set, append, use; Makefile.common: append, use")
	sys("GAMEDATAMODE", lkNone, CheckvarFileMode)
	sys("GAMES_GROUP", lkNone, CheckvarUserGroupName)
	sys("GAMEMODE", lkNone, CheckvarFileMode)
	sys("GAMES_USER", lkNone, CheckvarUserGroupName)
	pkglist("GCC_REQD", lkShell, CheckvarVersion)
	pkglist("GENERATE_PLIST", lkNone, CheckvarShellCommands)
	pkg("GITHUB_PROJECT", lkNone, CheckvarIdentifier)
	pkg("GITHUB_TAG", lkNone, CheckvarIdentifier)
	pkg("GITHUB_RELEASE", lkNone, CheckvarFilename)
	pkg("GITHUB_TYPE", lkNone, enum("tag release"))
	acl("GNU_ARCH", lkNone, enum("mips"), "")
	acl("GNU_CONFIGURE", lkNone, CheckvarYes, "Makefile, Makefile.common: set")
	acl("GNU_CONFIGURE_INFODIR", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	acl("GNU_CONFIGURE_LIBDIR", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	pkg("GNU_CONFIGURE_LIBSUBDIR", lkNone, CheckvarPathname)
	acl("GNU_CONFIGURE_MANDIR", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	acl("GNU_CONFIGURE_PREFIX", lkNone, CheckvarPathname, "Makefile: set")
	acl("HAS_CONFIGURE", lkNone, CheckvarYes, "Makefile, Makefile.common: set")
	pkglist("HEADER_TEMPLATES", lkShell, CheckvarPathname)
	pkg("HOMEPAGE", lkNone, CheckvarHomepage)
	acl("IGNORE_PKG.*", lkNone, CheckvarYes, "*: set, use-loadtime")
	acl("INCOMPAT_CURSES", lkSpace, CheckvarMachinePlatformPattern, "Makefile: set, append")
	acl("INCOMPAT_ICONV", lkSpace, CheckvarMachinePlatformPattern, "")
	acl("INFO_DIR", lkNone, CheckvarPathname, "") // relative to PREFIX
	pkg("INFO_FILES", lkNone, CheckvarYes)
	sys("INSTALL", lkNone, CheckvarShellCommand)
	pkglist("INSTALLATION_DIRS", lkShell, CheckvarPrefixPathname)
	pkg("INSTALLATION_DIRS_FROM_PLIST", lkNone, CheckvarYes)
	sys("INSTALL_DATA", lkNone, CheckvarShellCommand)
	sys("INSTALL_DATA_DIR", lkNone, CheckvarShellCommand)
	pkglist("INSTALL_DIRS", lkShell, CheckvarWrksrcSubdirectory)
	pkglist("INSTALL_ENV", lkShell, CheckvarShellWord)
	acl("INSTALL_FILE", lkNone, CheckvarPathname, "Makefile: set")
	sys("INSTALL_GAME", lkNone, CheckvarShellCommand)
	sys("INSTALL_GAME_DATA", lkNone, CheckvarShellCommand)
	sys("INSTALL_LIB", lkNone, CheckvarShellCommand)
	sys("INSTALL_LIB_DIR", lkNone, CheckvarShellCommand)
	pkglist("INSTALL_MAKE_FLAGS", lkShell, CheckvarShellWord)
	sys("INSTALL_MAN", lkNone, CheckvarShellCommand)
	sys("INSTALL_MAN_DIR", lkNone, CheckvarShellCommand)
	sys("INSTALL_PROGRAM", lkNone, CheckvarShellCommand)
	sys("INSTALL_PROGRAM_DIR", lkNone, CheckvarShellCommand)
	sys("INSTALL_SCRIPT", lkNone, CheckvarShellCommand)
	acl("INSTALL_SCRIPTS_ENV", lkShell, CheckvarShellWord, "")
	sys("INSTALL_SCRIPT_DIR", lkNone, CheckvarShellCommand)
	acl("INSTALL_SRC", lkShell, CheckvarPathname, "Makefile: set; Makefile.common: default, set")
	pkg("INSTALL_TARGET", lkShell, CheckvarIdentifier)
	acl("INSTALL_TEMPLATES", lkShell, CheckvarPathname, "Makefile: set, append; Makefile.common: set, default, append")
	acl("INSTALL_UNSTRIPPED", lkNone, CheckvarYesNo, "Makefile, Makefile.common: set")
	pkg("INTERACTIVE_STAGE", lkShell, enum("fetch extract configure build test install"))
	acl("IS_BUILTIN.*", lkNone, CheckvarYesNoIndirectly, "builtin.mk: set, use-loadtime, use")
	sys("JAVA_BINPREFIX", lkNone, CheckvarPathname)
	pkg("JAVA_CLASSPATH", lkNone, CheckvarShellWord)
	pkg("JAVA_HOME", lkNone, CheckvarPathname)
	pkg("JAVA_NAME", lkNone, CheckvarFilename)
	pkglist("JAVA_UNLIMIT", lkShell, enum("cmdsize datasize stacksize"))
	pkglist("JAVA_WRAPPERS", lkSpace, CheckvarFilename)
	pkg("JAVA_WRAPPER_BIN.*", lkNone, CheckvarPathname)
	sys("KRB5BASE", lkNone, CheckvarPathname)
	acl("KRB5_ACCEPTED", lkShell, enum("heimdal mit-krb5"), "")
	usr("KRB5_DEFAULT", lkNone, enum("heimdal mit-krb5"))
	sys("KRB5_TYPE", lkNone, CheckvarIdentifier)
	sys("LD", lkNone, CheckvarShellCommand)
	pkglist("LDFLAGS*", lkShell, CheckvarLdFlag)
	sys("LIBGRP", lkNone, CheckvarUserGroupName)
	sys("LIBMODE", lkNone, CheckvarFileMode)
	sys("LIBOWN", lkNone, CheckvarUserGroupName)
	sys("LIBOSSAUDIO", lkNone, CheckvarPathname)
	pkglist("LIBS*", lkShell, CheckvarLdFlag)
	sys("LIBTOOL", lkNone, CheckvarShellCommand)
	acl("LIBTOOL_OVERRIDE", lkShell, CheckvarPathmask, "Makefile: set, append")
	pkglist("LIBTOOL_REQD", lkShell, CheckvarVersion)
	acl("LICENCE", lkNone, CheckvarLicense, "Makefile, Makefile.common: set; options.mk: set")
	acl("LICENSE", lkNone, CheckvarLicense, "Makefile, Makefile.common: set; options.mk: set")
	pkg("LICENSE_FILE", lkNone, CheckvarPathname)
	sys("LINKER_RPATH_FLAG", lkNone, CheckvarShellWord)
	sys("LOWER_OPSYS", lkNone, CheckvarIdentifier)
	acl("LTCONFIG_OVERRIDE", lkShell, CheckvarPathmask, "Makefile: set, append; Makefile.common: append")
	sys("MACHINE_ARCH", lkNone, enumMachineArch)
	sys("MACHINE_GNU_ARCH", lkNone, enumMachineGnuArch)
	sys("MACHINE_GNU_PLATFORM", lkNone, CheckvarMachineGnuPlatform)
	sys("MACHINE_PLATFORM", lkNone, CheckvarMachinePlatform)
	acl("MAINTAINER", lkNone, CheckvarMailAddress, "Makefile: set; Makefile.common: default")
	sys("MAKE", lkNone, CheckvarShellCommand)
	pkglist("MAKEFLAGS", lkShell, CheckvarShellWord)
	acl("MAKEVARS", lkShell, CheckvarVarname, "builtin.mk: append; buildlink3.mk: append; hacks.mk: append")
	pkglist("MAKE_DIRS", lkShell, CheckvarPathname)
	pkglist("MAKE_DIRS_PERMS", lkShell, CheckvarPerms)
	acl("MAKE_ENV", lkShell, CheckvarShellWord, "Makefile: append, set, use; Makefile.common: append, set, use; buildlink3.mk: append; builtin.mk: append; *.mk: append, use")
	pkg("MAKE_FILE", lkNone, CheckvarPathname)
	pkglist("MAKE_FLAGS", lkShell, CheckvarShellWord)
	usr("MAKE_JOBS", lkNone, CheckvarInteger)
	pkg("MAKE_JOBS_SAFE", lkNone, CheckvarYesNo)
	pkg("MAKE_PROGRAM", lkNone, CheckvarShellCommand)
	acl("MANCOMPRESSED", lkNone, CheckvarYesNo, "Makefile: set; Makefile.common: default, set")
	acl("MANCOMPRESSED_IF_MANZ", lkNone, CheckvarYes, "Makefile: set; Makefile.common: default, set")
	sys("MANGRP", lkNone, CheckvarUserGroupName)
	sys("MANMODE", lkNone, CheckvarFileMode)
	sys("MANOWN", lkNone, CheckvarUserGroupName)
	pkglist("MASTER_SITES", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_APACHE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_BACKUP", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_CYGWIN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_DEBIAN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_FREEBSD", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_FREEBSD_LOCAL", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_GENTOO", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_GITHUB", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_GNOME", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_GNU", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_GNUSTEP", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_IFARCHIVE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_HASKELL_HACKAGE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_KDE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_LOCAL", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_MOZILLA", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_MOZILLA_ALL", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_MOZILLA_ESR", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_MYSQL", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_NETLIB", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_OPENOFFICE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_OSDN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_PERL_CPAN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_R_CRAN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_RUBYGEMS", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_SOURCEFORGE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_SUNSITE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_SUSE", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_TEX_CTAN", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_XCONTRIB", lkShell, CheckvarFetchURL)
	sys("MASTER_SITE_XEMACS", lkShell, CheckvarFetchURL)
	pkglist("MESSAGE_SRC", lkShell, CheckvarPathname)
	acl("MESSAGE_SUBST", lkShell, CheckvarShellWord, "Makefile.common: append; Makefile: append; options.mk: append")
	pkg("META_PACKAGE", lkNone, CheckvarYes)
	sys("MISSING_FEATURES", lkShell, CheckvarIdentifier)
	acl("MYSQL_VERSIONS_ACCEPTED", lkShell, enum("51 55 56"), "Makefile: set")
	usr("MYSQL_VERSION_DEFAULT", lkNone, CheckvarVersion)
	sys("NM", lkNone, CheckvarShellCommand)
	sys("NONBINMODE", lkNone, CheckvarFileMode)
	pkg("NOT_FOR_COMPILER", lkShell, enum("ccache ccc clang distcc f2c gcc hp icc ido mipspro mipspro-ucode pcc sunpro xlc"))
	pkglist("NOT_FOR_PLATFORM", lkSpace, CheckvarMachinePlatformPattern)
	pkg("NOT_FOR_UNPRIVILEGED", lkNone, CheckvarYesNo)
	acl("NO_BIN_ON_CDROM", lkNone, CheckvarRestricted, "Makefile, Makefile.common: set")
	acl("NO_BIN_ON_FTP", lkNone, CheckvarRestricted, "Makefile, Makefile.common: set")
	acl("NO_BUILD", lkNone, CheckvarYes, "Makefile, Makefile.common: set; Makefile.*: default, set")
	pkg("NO_CHECKSUM", lkNone, CheckvarYes)
	pkg("NO_CONFIGURE", lkNone, CheckvarYes)
	acl("NO_EXPORT_CPP", lkNone, CheckvarYes, "Makefile: set")
	pkg("NO_EXTRACT", lkNone, CheckvarYes)
	pkg("NO_INSTALL_MANPAGES", lkNone, CheckvarYes) // only has an effect for Imake packages.
	acl("NO_PKGTOOLS_REQD_CHECK", lkNone, CheckvarYes, "Makefile: set")
	acl("NO_SRC_ON_CDROM", lkNone, CheckvarRestricted, "Makefile, Makefile.common: set")
	acl("NO_SRC_ON_FTP", lkNone, CheckvarRestricted, "Makefile, Makefile.common: set")
	pkglist("ONLY_FOR_COMPILER", lkShell, enum("ccc clang gcc hp icc ido mipspro mipspro-ucode pcc sunpro xlc"))
	pkglist("ONLY_FOR_PLATFORM", lkSpace, CheckvarMachinePlatformPattern)
	pkg("ONLY_FOR_UNPRIVILEGED", lkNone, CheckvarYesNo)
	sys("OPSYS", lkNone, CheckvarIdentifier)
	acl("OPSYSVARS", lkShell, CheckvarVarname, "Makefile, Makefile.common: append")
	acl("OSVERSION_SPECIFIC", lkNone, CheckvarYes, "Makefile, Makefile.common: set")
	sys("OS_VERSION", lkNone, CheckvarVersion)
	pkg("OVERRIDE_DIRDEPTH*", lkNone, CheckvarInteger)
	pkg("OVERRIDE_GNU_CONFIG_SCRIPTS", lkNone, CheckvarYes)
	acl("OWNER", lkNone, CheckvarMailAddress, "Makefile: set; Makefile.common: default")
	pkglist("OWN_DIRS", lkShell, CheckvarPathname)
	pkglist("OWN_DIRS_PERMS", lkShell, CheckvarPerms)
	sys("PAMBASE", lkNone, CheckvarPathname)
	usr("PAM_DEFAULT", lkNone, enum("linux-pam openpam solaris-pam"))
	acl("PATCHDIR", lkNone, CheckvarRelativePkgPath, "Makefile: set; Makefile.common: default, set")
	pkglist("PATCHFILES", lkShell, CheckvarFilename)
	acl("PATCH_ARGS", lkShell, CheckvarShellWord, "")
	acl("PATCH_DIST_ARGS", lkShell, CheckvarShellWord, "Makefile: set, append")
	acl("PATCH_DIST_CAT", lkNone, CheckvarShellCommand, "")
	acl("PATCH_DIST_STRIP*", lkNone, CheckvarShellWord, "Makefile, Makefile.common: set; buildlink3.mk:; builtin.mk:; *.mk: set")
	acl("PATCH_SITES", lkShell, CheckvarFetchURL, "Makefile: set; options.mk: set; Makefile.common: set")
	acl("PATCH_STRIP", lkNone, CheckvarShellWord, "")
	pkg("PERL5_USE_PACKLIST", lkNone, CheckvarYesNo)
	acl("PERL5_PACKLIST", lkShell, CheckvarPerl5Packlist, "Makefile: set; options.mk: set, append")
	acl("PERL5_PACKLIST_DIR", lkNone, CheckvarPathname, "")
	sys("PGSQL_PREFIX", lkNone, CheckvarPathname)
	acl("PGSQL_VERSIONS_ACCEPTED", lkShell, enum("91 92 93 94"), "")
	usr("PGSQL_VERSION_DEFAULT", lkNone, CheckvarVersion)
	sys("PG_LIB_EXT", lkNone, enum("dylib so"))
	sys("PGSQL_TYPE", lkNone, enum("postgresql81-client postgresql80-client"))
	sys("PGPKGSRCDIR", lkNone, CheckvarPathname)
	sys("PHASE_MSG", lkNone, CheckvarShellCommand)
	usr("PHP_VERSION_REQD", lkNone, CheckvarVersion)
	sys("PKGBASE", lkNone, CheckvarIdentifier)
	acl("PKGCONFIG_FILE.*", lkShell, CheckvarPathname, "builtin.mk: set, append; pkgconfig-builtin.mk: use-loadtime")
	acl("PKGCONFIG_OVERRIDE", lkShell, CheckvarPathmask, "Makefile: set, append; Makefile.common: append")
	pkg("PKGCONFIG_OVERRIDE_STAGE", lkNone, CheckvarStage)
	pkg("PKGDIR", lkNone, CheckvarRelativePkgDir)
	sys("PKGDIRMODE", lkNone, CheckvarFileMode)
	sys("PKGLOCALEDIR", lkNone, CheckvarPathname)
	pkg("PKGNAME", lkNone, CheckvarPkgName)
	sys("PKGNAME_NOREV", lkNone, CheckvarPkgName)
	sys("PKGPATH", lkNone, CheckvarPathname)
	acl("PKGREPOSITORY", lkNone, CheckvarUnchecked, "")
	acl("PKGREVISION", lkNone, CheckvarPkgRevision, "Makefile: set")
	sys("PKGSRCDIR", lkNone, CheckvarPathname)
	acl("PKGSRCTOP", lkNone, CheckvarYes, "Makefile: set")
	acl("PKGTOOLS_ENV", lkShell, CheckvarShellWord, "")
	sys("PKGVERSION", lkNone, CheckvarVersion)
	sys("PKGWILDCARD", lkNone, CheckvarFilemask)
	sys("PKG_ADMIN", lkNone, CheckvarShellCommand)
	sys("PKG_APACHE", lkNone, enum("apache22 apache24"))
	pkg("PKG_APACHE_ACCEPTED", lkShell, enum("apache22 apache24"))
	usr("PKG_APACHE_DEFAULT", lkNone, enum("apache22 apache24"))
	usr("PKG_CONFIG", lkNone, CheckvarYes)
	// ^^ No, this is not the popular command from GNOME, but the setting
	// whether the pkgsrc user wants configuration files automatically
	// installed or not.
	sys("PKG_CREATE", lkNone, CheckvarShellCommand)
	sys("PKG_DBDIR", lkNone, CheckvarPathname)
	cmdline("PKG_DEBUG_LEVEL", lkNone, CheckvarInteger)
	usr("PKG_DEFAULT_OPTIONS", lkShell, CheckvarOption)
	sys("PKG_DELETE", lkNone, CheckvarShellCommand)
	acl("PKG_DESTDIR_SUPPORT", lkShell, enum("destdir user-destdir"), "Makefile, Makefile.common: set")
	pkglist("PKG_FAIL_REASON", lkShell, CheckvarShellWord)
	acl("PKG_GECOS.*", lkNone, CheckvarMessage, "Makefile: set")
	acl("PKG_GID.*", lkNone, CheckvarInteger, "Makefile: set")
	acl("PKG_GROUPS", lkShell, CheckvarShellWord, "Makefile: set, append")
	pkglist("PKG_GROUPS_VARS", lkShell, CheckvarVarname)
	acl("PKG_HOME.*", lkNone, CheckvarPathname, "Makefile: set")
	acl("PKG_HACKS", lkShell, CheckvarIdentifier, "hacks.mk: append")
	sys("PKG_INFO", lkNone, CheckvarShellCommand)
	sys("PKG_JAVA_HOME", lkNone, CheckvarPathname)
	jvms := enum("openjdk8 oracle-jdk8 openjdk7 sun-jdk7 sun-jdk6 jdk16 jdk15 kaffe") // See mk/java-vm.mk:/_PKG_JVMS/
	sys("PKG_JVM", lkNone, jvms)
	acl("PKG_JVMS_ACCEPTED", lkShell, jvms, "Makefile: set; Makefile.common: default, set")
	usr("PKG_JVM_DEFAULT", lkNone, jvms)
	acl("PKG_LEGACY_OPTIONS", lkShell, CheckvarOption, "")
	acl("PKG_LIBTOOL", lkNone, CheckvarPathname, "Makefile: set")
	acl("PKG_OPTIONS", lkSpace, CheckvarOption, "bsd.options.mk: set; *: use-loadtime, use")
	usr("PKG_OPTIONS.*", lkSpace, CheckvarOption)
	acl("PKG_OPTIONS_DEPRECATED_WARNINGS", lkShell, CheckvarShellWord, "")
	acl("PKG_OPTIONS_GROUP.*", lkSpace, CheckvarOption, "options.mk: set; Makefile: set")
	acl("PKG_OPTIONS_LEGACY_OPTS", lkSpace, CheckvarUnchecked, "Makefile, Makefile.common: append; options.mk: append")
	acl("PKG_OPTIONS_LEGACY_VARS", lkSpace, CheckvarUnchecked, "Makefile, Makefile.common: append; options.mk: append")
	acl("PKG_OPTIONS_NONEMPTY_SETS", lkSpace, CheckvarIdentifier, "")
	acl("PKG_OPTIONS_OPTIONAL_GROUPS", lkSpace, CheckvarIdentifier, "options.mk: set, append")
	acl("PKG_OPTIONS_REQUIRED_GROUPS", lkSpace, CheckvarIdentifier, "options.mk: set; Makefile: set")
	acl("PKG_OPTIONS_SET.*", lkSpace, CheckvarOption, "")
	acl("PKG_OPTIONS_VAR", lkNone, CheckvarPkgOptionsVar, "options.mk: set; Makefile, Makefile.common: set; bsd.options.mk: use-loadtime")
	acl("PKG_PRESERVE", lkNone, CheckvarYes, "Makefile: set")
	acl("PKG_SHELL", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	acl("PKG_SHELL.*", lkNone, CheckvarPathname, "Makefile, Makefile.common: set")
	acl("PKG_SHLIBTOOL", lkNone, CheckvarPathname, "")
	pkglist("PKG_SKIP_REASON", lkShell, CheckvarShellWord)
	acl("PKG_SUGGESTED_OPTIONS", lkShell, CheckvarOption, "options.mk: set, append; Makefile: set, append; Makefile.common: set")
	acl("PKG_SUPPORTED_OPTIONS", lkShell, CheckvarOption, "options.mk: set, append, use; Makefile: set, append; Makefile.common: set")
	pkg("PKG_SYSCONFDIR*", lkNone, CheckvarPathname)
	pkglist("PKG_SYSCONFDIR_PERMS", lkShell, CheckvarPerms)
	sys("PKG_SYSCONFBASEDIR", lkNone, CheckvarPathname)
	pkg("PKG_SYSCONFSUBDIR", lkNone, CheckvarPathname)
	acl("PKG_SYSCONFVAR", lkNone, CheckvarIdentifier, "") // FIXME: name/type mismatch.
	acl("PKG_UID", lkNone, CheckvarInteger, "Makefile: set")
	acl("PKG_USERS", lkShell, CheckvarShellWord, "Makefile: set, append")
	pkg("PKG_USERS_VARS", lkShell, CheckvarVarname)
	acl("PKG_USE_KERBEROS", lkNone, CheckvarYes, "Makefile, Makefile.common: set")
	// PLIST.* has special handling code
	pkglist("PLIST_VARS", lkShell, CheckvarIdentifier)
	pkglist("PLIST_SRC", lkShell, CheckvarRelativePkgPath)
	pkglist("PLIST_SUBST", lkShell, CheckvarShellWord)
	acl("PLIST_TYPE", lkNone, enum("dynamic static"), "")
	acl("PREPEND_PATH", lkShell, CheckvarPathname, "")
	acl("PREFIX", lkNone, CheckvarPathname, "*: use")
	acl("PREV_PKGPATH", lkNone, CheckvarPathname, "*: use") // doesn't exist any longer
	acl("PRINT_PLIST_AWK", lkNone, CheckvarAwkCommand, "*: append")
	acl("PRIVILEGED_STAGES", lkShell, enum("install package clean"), "")
	acl("PTHREAD_AUTO_VARS", lkNone, CheckvarYesNo, "Makefile: set")
	sys("PTHREAD_CFLAGS", lkShell, CheckvarCFlag)
	sys("PTHREAD_LDFLAGS", lkShell, CheckvarLdFlag)
	sys("PTHREAD_LIBS", lkShell, CheckvarLdFlag)
	acl("PTHREAD_OPTS", lkShell, enum("native optional require"), "Makefile: set, append; Makefile.common: append; buildlink3.mk: append")
	sys("PTHREAD_TYPE", lkNone, CheckvarIdentifier) // Or "native" or "none".
	pkg("PY_PATCHPLIST", lkNone, CheckvarYes)
	acl("PYPKGPREFIX", lkNone, enum("py27 py33 py34 py35"), "pyversion.mk: set; *: use-loadtime, use")
	pkg("PYTHON_FOR_BUILD_ONLY", lkNone, CheckvarYes)
	pkglist("REPLACE_PYTHON", lkShell, CheckvarPathmask)
	pkg("PYTHON_VERSIONS_ACCEPTED", lkShell, CheckvarVersion)
	pkg("PYTHON_VERSIONS_INCOMPATIBLE", lkShell, CheckvarVersion)
	usr("PYTHON_VERSION_DEFAULT", lkNone, CheckvarVersion)
	usr("PYTHON_VERSION_REQD", lkNone, CheckvarVersion)
	pkglist("PYTHON_VERSIONED_DEPENDENCIES", lkShell, CheckvarPythonDependency)
	sys("RANLIB", lkNone, CheckvarShellCommand)
	pkglist("RCD_SCRIPTS", lkShell, CheckvarFilename)
	acl("RCD_SCRIPT_SRC.*", lkShell, CheckvarPathname, "Makefile: set")
	acl("REPLACE.*", lkNone, CheckvarString, "Makefile: set")
	pkglist("REPLACE_AWK", lkShell, CheckvarPathmask)
	pkglist("REPLACE_BASH", lkShell, CheckvarPathmask)
	pkglist("REPLACE_CSH", lkShell, CheckvarPathmask)
	acl("REPLACE_EMACS", lkShell, CheckvarPathmask, "")
	acl("REPLACE_FILES.*", lkShell, CheckvarPathmask, "Makefile, Makefile.common: set, append")
	acl("REPLACE_INTERPRETER", lkShell, CheckvarIdentifier, "Makefile, Makefile.common: append")
	pkglist("REPLACE_KSH", lkShell, CheckvarPathmask)
	pkglist("REPLACE_LOCALEDIR_PATTERNS", lkShell, CheckvarFilemask)
	pkglist("REPLACE_LUA", lkShell, CheckvarPathmask)
	pkglist("REPLACE_PERL", lkShell, CheckvarPathmask)
	pkglist("REPLACE_PYTHON", lkShell, CheckvarPathmask)
	pkglist("REPLACE_SH", lkShell, CheckvarPathmask)
	pkglist("REQD_DIRS", lkShell, CheckvarPathname)
	pkglist("REQD_DIRS_PERMS", lkShell, CheckvarPerms)
	pkglist("REQD_FILES", lkShell, CheckvarPathname)
	pkg("REQD_FILES_MODE", lkNone, enum("0644 0640 0600 0400"))
	pkglist("REQD_FILES_PERMS", lkShell, CheckvarPerms)
	pkg("RESTRICTED", lkNone, CheckvarMessage)
	usr("ROOT_USER", lkNone, CheckvarUserGroupName)
	usr("ROOT_GROUP", lkNone, CheckvarUserGroupName)
	usr("RUBY_VERSION_REQD", lkNone, CheckvarVersion)
	sys("RUN", lkNone, CheckvarShellCommand)
	sys("RUN_LDCONFIG", lkNone, CheckvarYesNo)
	acl("SCRIPTS_ENV", lkShell, CheckvarShellWord, "Makefile, Makefile.common: append")
	usr("SETUID_ROOT_PERMS", lkShell, CheckvarShellWord)
	sys("SHAREGRP", lkNone, CheckvarUserGroupName)
	sys("SHAREMODE", lkNone, CheckvarFileMode)
	sys("SHAREOWN", lkNone, CheckvarUserGroupName)
	sys("SHCOMMENT", lkNone, CheckvarShellCommand)
	acl("SHLIB_HANDLING", lkNone, enum("YES NO no"), "")
	acl("SHLIBTOOL", lkNone, CheckvarShellCommand, "Makefile: use")
	acl("SHLIBTOOL_OVERRIDE", lkShell, CheckvarPathmask, "Makefile: set, append; Makefile.common: append")
	acl("SITES.*", lkShell, CheckvarFetchURL, "Makefile, Makefile.common, options.mk: set, append, use")
	pkglist("SPECIAL_PERMS", lkShell, CheckvarPerms)
	sys("STEP_MSG", lkNone, CheckvarShellCommand)
	acl("SUBDIR", lkShell, CheckvarFilename, "Makefile: append; *:")
	acl("SUBST_CLASSES", lkShell, CheckvarIdentifier, "Makefile: set, append; *: append")
	acl("SUBST_FILES.*", lkShell, CheckvarPathmask, "Makefile: set, append; Makefile.*, *.mk: set, append")
	acl("SUBST_FILTER_CMD.*", lkNone, CheckvarShellCommand, "Makefile, Makefile.*, *.mk: set")
	acl("SUBST_MESSAGE.*", lkNone, CheckvarMessage, "Makefile, Makefile.*, *.mk: set")
	acl("SUBST_SED.*", lkNone, CheckvarSedCommands, "Makefile, Makefile.*, *.mk: set, append")
	pkg("SUBST_STAGE.*", lkNone, CheckvarStage)
	pkglist("SUBST_VARS.*", lkShell, CheckvarVarname)
	pkglist("SUPERSEDES", lkSpace, CheckvarDependency)
	pkglist("TEST_DIRS", lkShell, CheckvarWrksrcSubdirectory)
	pkglist("TEST_ENV", lkShell, CheckvarShellWord)
	acl("TEST_TARGET", lkShell, CheckvarIdentifier, "Makefile: set; Makefile.common: default, set; options.mk: set, append")
	acl("TEX_ACCEPTED", lkShell, enum("teTeX1 teTeX2 teTeX3"), "Makefile, Makefile.common: set")
	acl("TEX_DEPMETHOD", lkNone, enum("build run"), "Makefile, Makefile.common: set")
	pkglist("TEXINFO_REQD", lkShell, CheckvarVersion)
	acl("TOOL_DEPENDS", lkSpace, CheckvarDependencyWithPath, "Makefile, Makefile.common, *.mk: append")
	sys("TOOLS_ALIASES", lkShell, CheckvarFilename)
	sys("TOOLS_BROKEN", lkShell, CheckvarTool)
	sys("TOOLS_CMD.*", lkNone, CheckvarPathname)
	sys("TOOLS_CREATE", lkShell, CheckvarTool)
	acl("TOOLS_DEPENDS.*", lkSpace, CheckvarDependencyWithPath, "buildlink3.mk:; Makefile, Makefile.*: set, default; *: use")
	sys("TOOLS_GNU_MISSING", lkShell, CheckvarTool)
	sys("TOOLS_NOOP", lkShell, CheckvarTool)
	sys("TOOLS_PATH.*", lkNone, CheckvarPathname)
	sys("TOOLS_PLATFORM.*", lkNone, CheckvarShellCommand)
	sys("TOUCH_FLAGS", lkShell, CheckvarShellWord)
	pkglist("UAC_REQD_EXECS", lkShell, CheckvarPrefixPathname)
	acl("UNLIMIT_RESOURCES", lkShell, enum("datasize stacksize memorysize"), "Makefile: set, append; Makefile.common: append")
	usr("UNPRIVILEGED_USER", lkNone, CheckvarUserGroupName)
	usr("UNPRIVILEGED_GROUP", lkNone, CheckvarUserGroupName)
	pkglist("UNWRAP_FILES", lkShell, CheckvarPathmask)
	usr("UPDATE_TARGET", lkShell, CheckvarIdentifier)
	pkg("USE_BSD_MAKEFILE", lkNone, CheckvarYes)
	acl("USE_BUILTIN.*", lkNone, CheckvarYesNoIndirectly, "builtin.mk: set")
	pkg("USE_CMAKE", lkNone, CheckvarYes)
	usr("USE_DESTDIR", lkNone, CheckvarYes)
	pkg("USE_FEATURES", lkShell, CheckvarIdentifier)
	pkg("USE_GCC_RUNTIME", lkNone, CheckvarYesNo)
	pkg("USE_GNU_CONFIGURE_HOST", lkNone, CheckvarYesNo)
	acl("USE_GNU_ICONV", lkNone, CheckvarYes, "Makefile, Makefile.common: set; options.mk: set")
	acl("USE_IMAKE", lkNone, CheckvarYes, "Makefile: set")
	pkg("USE_JAVA", lkNone, enum("run yes build"))
	pkg("USE_JAVA2", lkNone, enum("YES yes no 1.4 1.5 6 7 8"))
	acl("USE_LANGUAGES", lkShell, enum("ada c c99 c++ fortran fortran77 java objc"), "Makefile, Makefile.common, options.mk: set, append")
	pkg("USE_LIBTOOL", lkNone, CheckvarYes)
	pkg("USE_MAKEINFO", lkNone, CheckvarYes)
	pkg("USE_MSGFMT_PLURALS", lkNone, CheckvarYes)
	pkg("USE_NCURSES", lkNone, CheckvarYes)
	pkg("USE_OLD_DES_API", lkNone, CheckvarYesNo)
	pkg("USE_PKGINSTALL", lkNone, CheckvarYes)
	pkg("USE_PKGLOCALEDIR", lkNone, CheckvarYesNo)
	usr("USE_PKGSRC_GCC", lkNone, CheckvarYes)
	acl("USE_TOOLS", lkShell, CheckvarTool, "*: append")
	pkg("USE_X11", lkNone, CheckvarYes)
	sys("WARNING_MSG", lkNone, CheckvarShellCommand)
	sys("WARNING_CAT", lkNone, CheckvarShellCommand)
	acl("WRAPPER_REORDER_CMDS", lkShell, CheckvarWrapperReorder, "Makefile, Makefile.common, buildlink3.mk: append")
	acl("WRAPPER_TRANSFORM_CMDS", lkShell, CheckvarWrapperTransform, "Makefile, Makefile.common, buildlink3.mk: append")
	sys("WRKDIR", lkNone, CheckvarPathname)
	pkg("WRKSRC", lkNone, CheckvarWrkdirSubdirectory)
	sys("X11_PKGSRCDIR.*", lkNone, CheckvarPathname)
	usr("XAW_TYPE", lkNone, enum("3d neXtaw standard xpm"))
	acl("XMKMF_FLAGS", lkShell, CheckvarShellWord, "")
}

func enum(values string) *VarChecker {
	vmap := make(map[string]bool)
	for _, value := range splitOnSpace(values) {
		vmap[value] = true
	}
	name := "enum: " + values + " " // See IsEnum
	return &VarChecker{name, func(cv *VartypeCheck) {
		if cv.op == opUseMatch {
			if !vmap[cv.value] && cv.value == cv.valueNovar {
				canMatch := false
				for value := range vmap {
					if ok, err := path.Match(cv.value, value); err != nil {
						cv.line.Warnf("Invalid match pattern %q.", cv.value)
					} else if ok {
						canMatch = true
					}
				}
				if !canMatch {
					cv.line.Warnf("The pattern %q cannot match any of { %s } for %s.", cv.value, values, cv.varname)
				}
			}
			return
		}

		if cv.value == cv.valueNovar && !vmap[cv.value] {
			cv.line.Warnf("%q is not valid for %s. Use one of { %s } instead.", cv.value, cv.varname, values)
		}
	}}
}

func acl(varname string, kindOfList KindOfList, checker *VarChecker, aclentries string) {
	m := mustMatch(varname, `^([A-Z_.][A-Z0-9_]*)(|\*|\.\*)$`)
	varbase, varparam := m[1], m[2]

	vtype := &Vartype{kindOfList, checker, parseAclEntries(varname, aclentries), false}

	if G.globalData.vartypes == nil {
		G.globalData.vartypes = make(map[string]*Vartype)
	}
	if varparam == "" || varparam == "*" {
		G.globalData.vartypes[varbase] = vtype
	}
	if varparam == "*" || varparam == ".*" {
		G.globalData.vartypes[varbase+".*"] = vtype
	}
}

func parseAclEntries(varname string, aclentries string) []AclEntry {
	if aclentries == "" {
		return nil
	}
	var result []AclEntry
	for _, arg := range strings.Split(aclentries, "; ") {
		var globs, perms string
		if fields := strings.SplitN(arg, ": ", 2); len(fields) == 2 {
			globs, perms = fields[0], fields[1]
		} else {
			globs = strings.TrimSuffix(arg, ":")
		}
		var permissions AclPermissions
		for _, perm := range strings.Split(perms, ", ") {
			switch perm {
			case "append":
				permissions |= aclpAppend
			case "default":
				permissions |= aclpSetDefault
			case "set":
				permissions |= aclpSet
			case "use":
				permissions |= aclpUse
			case "use-loadtime":
				permissions |= aclpUseLoadtime
			case "":
				break
			default:
				print(fmt.Sprintf("Invalid ACL permission %q for varname %q.\n", perm, varname))
			}
		}
		for _, glob := range strings.Split(globs, ", ") {
			switch glob {
			case "*",
				"Makefile", "Makefile.common", "Makefile.*",
				"buildlink3.mk", "builtin.mk", "options.mk", "hacks.mk", "*.mk",
				"bsd.options.mk", "pkgconfig-builtin.mk", "pyversion.mk":
				break
			default:
				print(fmt.Sprintf("Invalid ACL glob %q for varname %q.\n", glob, varname))
			}
			for _, prev := range result {
				if matched, err := path.Match(prev.glob, glob); err != nil || matched {
					print(fmt.Sprintf("Ineffective ACL glob %q for varname %q.\n", glob, varname))
				}
			}
			result = append(result, AclEntry{glob, permissions})
		}
	}
	return result
}

// A package-defined variable may be set in all Makefiles except buildlink3.mk and builtin.mk.
func pkg(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "Makefile: set, use; buildlink3.mk, builtin.mk:; Makefile.*, *.mk: default, set, use")
}

// A package-defined list may be appended to in all Makefiles except buildlink3.mk and builtin.mk.
// Simple assignment (instead of appending) is only allowed in Makefile and Makefile.common.
func pkglist(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "Makefile, Makefile.common, options.mk: append, default, set, use; buildlink3.mk, builtin.mk:; *.mk: append, default, use")
}

// A user-defined or system-defined variable must not be set by any
// package file. It also must not be used in buildlink3.mk and
// builtin.mk files or at load-time, since the system/user preferences
// may not have been loaded when these files are included.
func sys(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "buildlink3.mk:; *: use")
}
func usr(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "buildlink3.mk:; *: use-loadtime, use")
}
func bl3list(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "buildlink3.mk, builtin.mk: append")
}
func cmdline(varname string, kindOfList KindOfList, checker *VarChecker) {
	acl(varname, kindOfList, checker, "buildlink3.mk, builtin.mk:; *: use-loadtime, use")
}
